package com.cv.media.lib.push;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;

import com.cv.media.lib.common_utils.async.ThreadsBox;
import com.cv.media.lib.common_utils.provider.ContextProvider;
import com.cv.media.lib.common_utils.rxjava.StreamController;

import java.io.File;

import io.reactivex.Observable;
import io.reactivex.Scheduler;
import io.reactivex.android.schedulers.AndroidSchedulers;

/**
 * Client point
 * 职责:
 *
 * @author Damon
 */
class Pusher implements ServiceConnection {
    private final PushParam param;
    private final PushClient IClient = new PushClient();
    private IPushServer IServer;
    private final StreamController<PushMessage> scMsg = new StreamController<>();
    private final StreamController<ClientState> scState = new StreamController<>();
    private final Handler pushHandler;
    private final Scheduler pushScheduler;

    {
        Looper lp = ThreadsBox.getNewHandlerThread("PusherManager").getLooper();
        pushScheduler = AndroidSchedulers.from(lp);
        pushHandler = new Handler(lp);
    }

    private ClientState clientState = ClientState.UnReady;

    Pusher(PushParam param) {
        this.param = param;
    }

    public Observable<PushMessage> start() throws Exception {
        Context context = ContextProvider.getContext();
        if (param.getBrokers().isEmpty()) {
            throw new Exception("Brokers can't be empty");
        }
        File dir = new File(context.getCacheDir(), "push");
        if (!dir.exists() && !dir.mkdir()) {
            throw new Exception("Create Push Dir fail");
        }
        param.dir = dir.getAbsolutePath();
        context.startService(new Intent(context, PushService.class));
        if (!context.bindService(new Intent(context, PushService.class), this, Context.BIND_AUTO_CREATE)) {
            PusherManager.getInstance().log(Pusher.this + "Bind Pusher Service fail");
            throw new Exception("Push Service can't be bound");
        }
        PusherManager.getInstance().log(Pusher.this + "Bind Pusher Service success");
        return scMsg.stream().observeOn(pushScheduler);
    }


    public Observable<ClientState> watchState() {
        return scState.stream().observeOn(pushScheduler).startWith(clientState);
    }

    Runnable checkServiceAfterDeathRb = new Runnable() {
        @Override
        public void run() {
            try {
                if (IServer == null) {
                    PusherManager.getInstance().log(Pusher.this + "service didn't start, now start manually");
                    Pusher.this.start();
                } else {
                    PusherManager.getInstance().log(Pusher.this + "service has been started, no need to start manually");
                }
            } catch (Exception e) {
                PusherManager.getInstance().log(Pusher.this + "start manually fail" + e.getMessage());
                e.printStackTrace();
            }
        }
    };

    private void checkServiceAfterDeath() {
        ThreadsBox.getDefaultMainHandler().removeCallbacks(checkServiceAfterDeathRb);
        PusherManager.getInstance().log(Pusher.this + "checkServiceAfterDeath invoke");
        ThreadsBox.getDefaultMainHandler().postDelayed(checkServiceAfterDeathRb, 10_000);
    }

    @Override
    public void onServiceConnected(ComponentName name, IBinder service) {
        PusherManager.getInstance().log(Pusher.this + "onServiceConnected");
        pushHandler.post(() -> {
            IServer = IPushServer.Stub.asInterface(service);
            if (!checkRemoteAvailable()) return;

            try {
                IServer.asBinder().linkToDeath(new IBinder.DeathRecipient() {
                    @Override
                    public void binderDied() {
                        PusherManager.getInstance().log(Pusher.this + "Service die!!!!!!!");
                        IServer.asBinder().unlinkToDeath(this, 0);
                        IServer = null;
                        checkServiceAfterDeath();
                    }
                }, 0);
            } catch (Throwable e) {
                e.printStackTrace();
                if (!checkRemoteAvailable()) return;
            }

            try {
                IServer.setDebug(PusherManager.getInstance().isDebug);
                PusherManager.getInstance().log(Pusher.this + "Client request connection");
                IServer.connect(IClient, param);
                PusherManager.getInstance().log(Pusher.this + "Client request connection done");
            } catch (Throwable e) {
                PusherManager.getInstance().log(Pusher.this + "Client request connection fail: " + e.getMessage());
                e.printStackTrace();
            }
        });
    }

    private boolean checkRemoteAvailable() {
        if (!IServer.asBinder().isBinderAlive() || !IServer.asBinder().pingBinder()) {
            IServer = null;
            checkServiceAfterDeath();
            return false;
        }
        return true;
    }

    @Override
    public void onServiceDisconnected(ComponentName name) {
        PusherManager.getInstance().log(Pusher.this + "onServiceDisconnected");
    }

    public void setDebug(boolean ref) {
        if (IServer != null) {
            try {
                IServer.setDebug(ref);
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    class PushClient extends IPushClient.Stub {
        @Override
        public void recvMsg(long code, String msg) {
            scMsg.push(new PushMessage(code, msg));
        }

        @Override
        public void stateChange(ClientState state) {
            PusherManager.getInstance().log("Pusher receive state:" + state);
            state = state.equals(ClientState.Ready) ? ClientState.Ready : ClientState.UnReady;  //state可能是深拷贝的一个, 转成全局唯一的
            if (state != clientState) {
                scState.push(state);
            }
            clientState = state;
        }
    }


    @Override
    public String toString() {
        return "Pusher[" + Integer.toHexString(hashCode()) + "] ";
    }
}
