package com.cv.media.lib.storage.database;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.collection.LruCache;
import androidx.room.Room;
import androidx.room.RoomDatabase;
import androidx.room.migration.Migration;

import com.cv.media.lib.common_utils.utils.Singleton;

import java.util.ArrayList;
import java.util.List;

import static com.cv.media.lib.common_utils.provider.ContextProvider.getContext;

/**
 * Created by zhangyaoa on 2020/6/2.
 */
public class DatabaseService {
    public static DatabaseService getInstance() {
        return mSingleton.get();
    }

    private static final Singleton<DatabaseService> mSingleton = new Singleton<DatabaseService>() {
        @Override
        protected DatabaseService create() {
            return new DatabaseService();
        }
    };

    private DatabaseService() {
    }

    private LruCache<String, Object> mRoomDatabaseCache;

    public static final String DEFAULT_DATABASE_NAME = "cv_media_room.db";
    public static final int DEFAULT_DAO_SIZE = 20;

    private List<Migration> mMigrationList = new ArrayList<>();

    public <T extends RoomDatabase> T getRoomDatabase(@NonNull Class<T> database) {
        return getRoomDatabase(database, DEFAULT_DATABASE_NAME);
    }

    public <T extends RoomDatabase> T getRoomDatabase(@NonNull Class<T> database, @Nullable String dbName) {
        if (mRoomDatabaseCache == null) {
            mRoomDatabaseCache = new LruCache<>(DEFAULT_DAO_SIZE);
        }
        Preconditions.checkNotNull(mRoomDatabaseCache);

        T roomDatabase = (T) mRoomDatabaseCache.get(database.getCanonicalName());
        if (roomDatabase == null) {
            synchronized (mRoomDatabaseCache) {
                if (roomDatabase == null) {
                    RoomDatabase.Builder<T> builder = Room.databaseBuilder(getContext().getApplicationContext(), database, TextUtils.isEmpty(dbName) ? DEFAULT_DATABASE_NAME : dbName);

                    //todo 错错错错错  临时处理
                    builder.addMigrations(mMigrationList.get(0),mMigrationList.get(1));
                    builder.fallbackToDestructiveMigration();
                    roomDatabase = builder.build();
                    //缓存
                    mRoomDatabaseCache.put(database.getCanonicalName(), roomDatabase);
                }
            }
        }
        return roomDatabase;
    }

    public void addMigrations(Migration migration){
        mMigrationList.add(migration);
    }
}
