package com.cv.media.lib.tracker;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.SparseArray;


import com.cv.media.lib.common_utils.appstat.AppManager;
import com.cv.media.lib.common_utils.appstat.features.IAppFeatures;
import com.cv.media.lib.common_utils.async.ThreadUtils;
import com.cv.media.lib.common_utils.utils.io.IOUtils;
import com.cv.media.lib.metric.data.MetricData;
import com.cv.media.lib.metric.manager.MetricManager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.reactivex.functions.Consumer;

/**
 * 职责: 负责将 埋点需求 以 接口形式 展现
 * 不负责 埋点需求数据的来源
 *
 * @author Damon
 */
public class TrackingService {
    @SuppressLint("StaticFieldLeak")
    protected static Context context;
    private static Class<?> workingTrackingAPIClz;
    private static final SparseArray<Tracking> mTrackings = new SparseArray<>();
    static final Consumer<IAppFeatures.AppStat> appStatConsumer = appStat -> {
        if (appStat == IAppFeatures.AppStat.Foreground) {
            TrackingService.dispatch(PendingStrategy.WHEN_APP_START);
        }
    };

    public static void setWorkingTrackingAPI(Class<?> track) {
        workingTrackingAPIClz = track;
    }

    public static <T extends Tracking> T get(Class<T> tracking) {
        Tracking trackingInstance = mTrackings.get(tracking.hashCode());
        if (trackingInstance == null) {
            if (tracking != workingTrackingAPIClz) {
                trackingInstance = (Tracking) Proxy.newProxyInstance(tracking.getClassLoader(), new Class[]{tracking}, new TrackingInvocationHandlerStub());
            } else {
                trackingInstance = (Tracking) Proxy.newProxyInstance(tracking.getClassLoader(), new Class[]{tracking}, new TrackingInvocationHandler());
            }
            mTrackings.put(tracking.hashCode(), trackingInstance);
        }
        return (T) trackingInstance;
    }

    static class TrackingInvocationHandlerStub implements InvocationHandler {
        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            return null;
        }
    }

    static class TrackingInvocationHandler implements InvocationHandler {
        SparseArray<MethodInfoCache> methodCaches = new SparseArray<>();

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) {
            MethodInfoCache methodInfoCache = methodCaches.get(method.hashCode());
            if (methodInfoCache == null) {
                methodInfoCache = new MethodInfoCache();
                Annotation[] annotations = method.getDeclaredAnnotations();
                for (Annotation annotation : annotations) {
                    if (annotation instanceof TrackEvent) {
                        methodInfoCache.action = ((TrackEvent) annotation).action();
                        methodInfoCache.isPending = ((TrackEvent) annotation).isPending();
                        methodInfoCache.strategy = ((TrackEvent) annotation).strategy();
                    } else if (annotation instanceof TrackEventParam) {
                        methodInfoCache.defaultParams.put(((TrackEventParam) annotation).key(), ((TrackEventParam) annotation).value());
                    }
                }
                if (methodInfoCache.action == null) return null;
                Annotation[][] parametersAnnotations = method.getParameterAnnotations();
                for (int i = 0; i < parametersAnnotations.length; i++) {
                    assert (parametersAnnotations[i].length == 1);
                    if (parametersAnnotations[i][0] instanceof TrackEventParamKey)
                        methodInfoCache.argNames.add(((TrackEventParamKey) parametersAnnotations[i][0]).value());
                }
            }

            String action = methodInfoCache.action;
            assert (action != null);
            HashMap<String, Object> params = new HashMap<>();
            for (int i = 0; i < methodInfoCache.argNames.size(); i++) {
                Object v = args[i];
                if (v instanceof Tracking.TrackEnum) {
                    v = ((Tracking.TrackEnum) v).getValue();
                } else if (!(v instanceof Long || v instanceof Integer || v instanceof Double || v instanceof List || v == null)) {
                    v = String.valueOf(v);
                }
                params.put(methodInfoCache.argNames.get(i), v);
            }
            params.putAll(methodInfoCache.defaultParams);

            if (methodInfoCache.isPending) {
                Map<String, Object> commonParams = MetricManager.getInstance().getChainSnapshot(MetricManager.MetricKey.APP).value();
                for (Map.Entry<String, Object> entry : commonParams.entrySet()) {
                    Object v = entry.getValue();
                    if (v instanceof Tracking.TrackEnum) {
                        v = ((Tracking.TrackEnum) v).getValue();
                    } else if (!(v instanceof Long || v instanceof Integer || v instanceof Double || v instanceof List || v == null)) {
                        v = String.valueOf(v);
                    }
                    params.put(entry.getKey(), v);
                }

                MetricRequest request = new MetricRequest();
                request.action = action;
                request.params = params;
                request.strategy = methodInfoCache.strategy;
                saveMetricRequest(request);
            } else
                MetricManager.getInstance().sendAppMetric(action, params);
            return null;
        }
    }

    static File pendingTracksStoreFile() {
        return new File(context.getCacheDir(), "tracking" + File.separator + "pending_tracks");
    }

    static void saveMetricRequest(MetricRequest request) {
        new SaveMetricRequestRunnable(request, pendingTracksStoreFile()).run();
    }


    private static void dispatch(PendingStrategy strategy) {
        if (strategy == PendingStrategy.WHEN_APP_START) {
            ThreadUtils.execute(new ProcessMetricRequestRunnable(pendingTracksStoreFile(), strategy));
        }
    }


    static class MethodInfoCache {
        List<String> argNames = new ArrayList<>();
        Map<String, Object> defaultParams = new HashMap<>();
        String action;
        PendingStrategy strategy;
        boolean isPending;
    }

    static class MetricRequest implements Serializable {
        String action;
        Map<String, Object> params;
        PendingStrategy strategy;
    }

    static class ProcessMetricRequestRunnable implements Runnable {
        File source;
        PendingStrategy strategy;

        public ProcessMetricRequestRunnable(File source, PendingStrategy strategy) {
            this.source = source;
            this.strategy = strategy;
        }


        @Override
        public void run() {
            if (!source.exists()) return;
            synchronized (TrackingService.class) {
                ObjectInputStream inputStream = null;
                ObjectOutputStream outputStream = null;
                try {
                    inputStream = new ObjectInputStream(new FileInputStream(source));
                    ArrayList<MetricRequest> requests = (ArrayList<MetricRequest>) inputStream.readObject();
                    outputStream = new ObjectOutputStream(new FileOutputStream(source));
                    if (requests != null) {
                        for (int i = 0; i < requests.size(); i++) {
                            MetricRequest request = requests.get(i);
                            if (request.strategy == strategy) {
                                MetricData data = new MetricData.Builder().put(request.params).build();
                                MetricManager.getInstance().put(MetricManager.MetricKey.APP, request.action, data, false);
                                requests.remove(i);
                                --i;
                            }
                        }
                    }
                    outputStream.writeObject(requests);
                } catch (Exception e) {
                    e.printStackTrace();
                } finally {
                    IOUtils.closeQuietly(inputStream);
                    IOUtils.closeQuietly(outputStream);
                }
            }
        }
    }

    static class SaveMetricRequestRunnable implements Runnable {
        MetricRequest request;
        File dest;

        public SaveMetricRequestRunnable(MetricRequest request, File dest) {
            this.request = request;
            this.dest = dest;
        }

        @Override
        public void run() {
            synchronized (TrackingService.class) {
                ObjectInputStream inputStream = null;
                ObjectOutputStream outputStream = null;
                try {
                    dest.getParentFile().mkdirs();
                    ArrayList<MetricRequest> requests = null;
                    if (!dest.exists()) dest.createNewFile();
                    try {
                        inputStream = new ObjectInputStream(new FileInputStream(dest));
                        requests = (ArrayList<MetricRequest>) inputStream.readObject();
                    } catch (Exception e) {
                        requests = new ArrayList<>();
                    }
                    requests.add(request);
                    outputStream = new ObjectOutputStream(new FileOutputStream(dest));
                    outputStream.writeObject(requests);
                } catch (Exception e) {
                    e.printStackTrace();
                } finally {
                    IOUtils.closeQuietly(inputStream);
                    IOUtils.closeQuietly(outputStream);
                }
            }
        }
    }
}
