package com.cv.media.lib.ui.focus;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Rect;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import androidx.recyclerview.widget.OrientationHelper;


import com.cv.media.lib.ui.R;

import java.util.ArrayList;

/**
 * Saves the focused grandchild position.
 * Helps add persistent focus feature to various ViewGroups.
 *
 * @hide copy from leanback-v17
 */
public class PersistentFocusWrapper extends FrameLayout {
    private static final String TAG = "PersistentFocusWrapper";
    static final boolean DEBUG = false;

    protected int mSelectedPosition = -1;

    /**
     * By default, focus is persisted when searching vertically
     * but not horizontally.
     */
    protected boolean mPersistFocusVertical = true;

    public PersistentFocusWrapper(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }

    public PersistentFocusWrapper(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        TypedArray ta = context.obtainStyledAttributes(attrs, R.styleable.PersistentFocusWrapper);
        int orientation = ta.getInt(R.styleable.PersistentFocusWrapper_persistOrientation, 2);
        ta.recycle();
        mPersistFocusVertical = orientation == 2;
    }

    // 第一个子view的子view个数
    int getGrandChildCount() {
        ViewGroup wrapper = (ViewGroup) getChildAt(0); // 只负责维护第一个子view的焦点
        return wrapper == null ? 0 : wrapper.getChildCount();
    }

    /**
     * Clears the selected position and clears focus.
     */
    public void clearSelection() {
        mSelectedPosition = -1;
        if (hasFocus()) {
            clearFocus();
        }
    }

    public void resetSelection() {
        mSelectedPosition = -1;
    }

    /**
     * Persist focus when focus search direction is up or down.
     */
    public void persistFocusVertical() {
        mPersistFocusVertical = true;
    }

    /**
     * Persist focus when focus search direction is left or right.
     */
    public void persistFocusHorizontal() {
        mPersistFocusVertical = false;
    }

    private boolean shouldPersistFocusFromDirection(int direction) {
        return ((mPersistFocusVertical && (direction == FOCUS_UP || direction == FOCUS_DOWN)) ||
                (!mPersistFocusVertical && (direction == FOCUS_LEFT || direction == FOCUS_RIGHT)));
    }

    @Override
    public void addFocusables(ArrayList<View> views, int direction, int focusableMode) {
        if (DEBUG) Log.v(TAG, "addFocusables");
        if (hasFocus() || getGrandChildCount() == 0 ||
                !shouldPersistFocusFromDirection(direction)) {
            super.addFocusables(views, direction, focusableMode);
        } else {
            // Select a child in requestFocus
            views.add(this);
        }
    }

    // wrapper的子view获得焦点时，保存下来位置（如果获得焦点后再次滚动，mSelectedPosition可能与之前的不同）
    @Override
    public void requestChildFocus(View child, View focused) {
        super.requestChildFocus(child, focused);
        View view = focused;
        while (view != null && view.getParent() != child) {
            view = (View) view.getParent();
        }
        mSelectedPosition = view == null ? -1 : ((ViewGroup) child).indexOfChild(view);
        if (DEBUG)
            Log.v(TAG, "requestChildFocus focused " + focused + " mSelectedPosition " + mSelectedPosition);
    }

    @Override
    public boolean requestFocus(int direction, Rect previouslyFocusedRect) {
        if (DEBUG) Log.v(TAG, "requestFocus mSelectedPosition " + mSelectedPosition);
        ViewGroup wrapper = (ViewGroup) getChildAt(0);
        if (wrapper != null && mSelectedPosition >= 0 && mSelectedPosition < getGrandChildCount()) {
            View focusView = wrapper.getChildAt(mSelectedPosition);
            if (focusView != null && focusView.requestFocus(direction, previouslyFocusedRect)) {
                return true;
            }
        }
        return super.requestFocus(direction, previouslyFocusedRect);
    }

    static class SavedState extends BaseSavedState {

        int mSelectedPosition;

        SavedState(Parcel in) {
            super(in);
            mSelectedPosition = in.readInt();
        }

        SavedState(Parcelable superState) {
            super(superState);
        }

        @Override
        public void writeToParcel(Parcel dest, int flags) {
            super.writeToParcel(dest, flags);
            dest.writeInt(mSelectedPosition);
        }

        public static final Creator<SavedState> CREATOR
                = new Creator<SavedState>() {
            @Override
            public SavedState createFromParcel(Parcel in) {
                return new SavedState(in);
            }

            @Override
            public SavedState[] newArray(int size) {
                return new SavedState[size];
            }
        };
    }

    @Override
    protected Parcelable onSaveInstanceState() {
        if (DEBUG) Log.v(TAG, "onSaveInstanceState");
        SavedState savedState = new SavedState(super.onSaveInstanceState());
        savedState.mSelectedPosition = mSelectedPosition;
        return savedState;
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        if (!(state instanceof SavedState)) {
            super.onRestoreInstanceState(state);
            return;
        }
        SavedState savedState = (SavedState) state;
        mSelectedPosition = ((SavedState) state).mSelectedPosition;
        if (DEBUG) Log.v(TAG, "onRestoreInstanceState mSelectedPosition " + mSelectedPosition);
        super.onRestoreInstanceState(savedState.getSuperState());
    }

    public void setPersistFocusOrientation(int orientation) {
        if (OrientationHelper.HORIZONTAL == orientation) {
            mPersistFocusVertical = false;
        } else {
            mPersistFocusVertical = true;
        }
    }
}
