/*
 * Copyright 2014 Hieu Rocker
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cv.media.lib.ui.emoji;

import android.content.Context;
import android.text.Spannable;

/**
 * @author Hieu Rocker (rockerhieu@gmail.com)
 */
public final class EmojiconHandler {
    private EmojiconHandler() {
    }

    private static IEmojiProvider mEmojiProvider = new DefaultEmojiProvider();

    public static void initEmojiProvider(IEmojiProvider provider) {
        mEmojiProvider = provider;
    }

    public static void initEmojiData(String jsonPath) {
        mEmojiProvider.parsEmojiSheet(jsonPath);
    }

    /**
     * Convert emoji characters of the given Spannable to the according emojicon.
     *
     * @param context
     * @param text
     * @param emojiSize
     * @param emojiAlignment
     * @param textSize
     */
    public static void addEmojis(Context context, Spannable text, int emojiSize, int emojiAlignment, int textSize) {
        addEmojis(context, text, emojiSize, emojiAlignment, textSize, 0, -1, false);
    }

    /**
     * Convert emoji characters of the given Spannable to the according emojicon.
     *
     * @param context
     * @param text
     * @param emojiSize
     * @param emojiAlignment
     * @param textSize
     * @param index
     * @param length
     */
    public static void addEmojis(Context context, Spannable text, int emojiSize, int emojiAlignment, int textSize, int index, int length) {
        addEmojis(context, text, emojiSize, emojiAlignment, textSize, index, length, false);
    }

    /**
     * Convert emoji characters of the given Spannable to the according emojicon.
     *
     * @param context
     * @param text
     * @param emojiSize
     * @param emojiAlignment
     * @param textSize
     * @param useSystemDefault
     */
    public static void addEmojis(Context context, Spannable text, int emojiSize, int emojiAlignment, int textSize, boolean useSystemDefault) {
        addEmojis(context, text, emojiSize, emojiAlignment, textSize, 0, -1, useSystemDefault);
    }

    /**
     * Convert emoji characters of the given Spannable to the according emojicon.
     *
     * @param context
     * @param text
     * @param emojiSize
     * @param emojiAlignment
     * @param textSize
     * @param index
     * @param length
     * @param useSystemDefault
     */
    public static void addEmojis(Context context, Spannable text, int emojiSize, int emojiAlignment, int textSize, int index, int length, boolean useSystemDefault) {
        if (useSystemDefault || text == null) {
            return;
        }

        try {
            int textLength = text.length();
            int textLengthToProcessMax = textLength - index;
            int textLengthToProcess = length < 0 || length >= textLengthToProcessMax ? textLength : (length + index);

            for (int i = index; i < textLengthToProcess; i += 1) {
                int icon = 0;

                if (icon == 0) {
                    int unicode = Character.codePointAt(text, i);

                    if (unicode == 0x5b) {
                        //“[”是表情开始
                        //+5个字符如果是“]”说明是个表情
                        if (Character.codePointAt(text, i + 5) == 0x5d) {
                            icon = mEmojiProvider.getEmojiResource(text.subSequence(i, i + 6));
                        }
                    }
                }

                if (icon > 0) {
                    text.setSpan(new EmojiconSpan(context, icon, emojiSize, emojiAlignment, textSize), i, i + 6, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                    i += 5;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
