package com.cv.media.lib.ui.focus;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.AnimatorSet;
import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.Choreographer;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.animation.Animation;
import android.widget.FrameLayout;
import android.widget.ScrollView;

import androidx.annotation.NonNull;
import androidx.leanback.widget.BaseGridView;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.viewpager.widget.ViewPager;

import com.cv.media.lib.common_utils.callback.ICallBack;

import java.util.ArrayList;

/**
 * 跟随焦点的框
 * 职责: 外部使用只添加, <焦点跟随>内部自动去做
 *
 * @author Damon
 */
public abstract class FocusHighLightView extends View implements ViewTreeObserver.OnGlobalFocusChangeListener, Application.ActivityLifecycleCallbacks {
    View mTarget;

    View mPendingTarget;

    AnimatorSet mAnimSet;
    Rect mRect = new Rect();
    Activity mActivity;
    Choreographer mChoreographer = Choreographer.getInstance();
    TargetWatcherCallBack mTargetCb = new TargetWatcherCallBack();
    private Chain mFirstChain = new NormalWork(null);
    public static final int KEY_OFFSET_CALLBACK = 0xab << 24;

    private final int EVENT_FOCUS_CHANGE = 0x231;
    Handler mHandler = new Handler() {
        @Override
        public void handleMessage(Message msg) {
            try {
                switch (msg.what) {
                    case EVENT_FOCUS_CHANGE:
                        ((Runnable) msg.obj).run();
                        break;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };

    private ViewPager mViewPagerToBinded;
    private Pair<ViewPager, Integer> pair_vp_frame;
    private ObjectAnimator mVisibleAnimator;
    private ObjectAnimator mLastVisibleAnimator;
    private boolean isTargetInViewPager = false;
    private final Rect mInsets = new Rect();

    private OnLayoutChangeListener mOnLayoutChangeListener = new OnLayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
            if (v.getWidth() == 0 || v.getHeight() == 0) return;
            v.removeOnLayoutChangeListener(this);
            if (mTarget == v) {
                //target没变
                to(mTarget);
            }
        }
    };

    private boolean isFilterMatrix = true;

    private boolean isFocusEventHappening = false;

    public FocusHighLightView(Context context) {
        super(context);
    }

    /**
     * 焦点事件来 飞框响应之前
     *
     * @param old
     * @param target
     * @return 返回什么, 什么 就被飞框认作为目标, return null, 则飞框不作出响应
     */
    public abstract View beforeHighLight(View old, View target);

    /**
     * 焦点事件来 飞框响应之后
     *
     * @param target
     */
    public abstract void afterHighLight(View target);

    public void in(Activity activity) {
        mActivity = activity;
        final ViewGroup decor = (ViewGroup) activity.getWindow().getDecorView();
        FrameLayout.LayoutParams lp = new FrameLayout.LayoutParams(0, 0);
        decor.addView(this, lp);
        getViewTreeObserver().addOnGlobalFocusChangeListener(this);
        activity.getApplication().registerActivityLifecycleCallbacks(this);
        decor.post(new Runnable() {
            @Override
            public void run() {
                View v = decor.findFocus();
                if (v != null) {
                    onGlobalFocusChanged(null, v);
                }
            }
        });
    }

    public void out() {
        if (mActivity != null) {
            mActivity.getApplication().unregisterActivityLifecycleCallbacks(this);
            mActivity = null;
        }
        if (getParent() != null) {
            getViewTreeObserver().removeOnGlobalFocusChangeListener(this);
            mChoreographer.removeFrameCallback(mTargetCb);
            ((ViewGroup) getParent()).removeView(this);
        }
        if (mHandler != null) {
            mHandler.removeCallbacksAndMessages(null);
        }
    }


    @Override
    public void onGlobalFocusChanged(final View oldFocus, final View newFocus) {
        if (mHandler != null) {
            //过滤 一次主线程消息 的 栈中 的 焦点变换, 取最后一个
            mHandler.removeMessages(EVENT_FOCUS_CHANGE);
            mHandler.sendMessage(mHandler.obtainMessage(EVENT_FOCUS_CHANGE, new Runnable() {
                @Override
                public void run() {
                    onFocusChange(oldFocus, newFocus);
                }
            }));
            isFocusEventHappening = true;
            mFirstChain.processOnEventIn();
        }
    }

    public void flyTo(View newFocus) {
        onGlobalFocusChanged(null, newFocus);
    }

    private void onFocusChange(View oldFocus, View newFocus) {
        ViewGroup vp = (ViewGroup) getParent();
        for (int i = 0; i < vp.getChildCount(); i++) {
            if (vp.getChildAt(i) == this && i < vp.getChildCount() - 1) {
                bringToFront();
                break;
            }
        }
        final View view = beforeHighLight(oldFocus, newFocus);
        if (view != null) {
            mPendingTarget = view;
            //包容下一次Loop 要做Animation 动画的target可被感知
            postDelayed(new Runnable() {
                @Override
                public void run() {
                    to(view);
                }
            }, 17);
        }
    }

    public void bindViewPager(ViewPager vp) {
        if (vp != null) {
            this.mViewPagerToBinded = vp;
            pair_vp_frame = new Pair<>(vp, vp.getScrollX());
            mFirstChain = new ViewPagerWork(new NormalWork(null).setVaildWaitingTime(100));
        }
    }

    public void setFilterMatrix(boolean filterMatrix) {
        isFilterMatrix = filterMatrix;
    }

    private Animation mFocusedAnimation;

    public void setFocusedAnimation(Animation animation) {
        if (null != getAnimation() && getAnimation() == animation) {
            return;
        }
        mFocusedAnimation = animation;
        if (animation == null) clearAnimation();
    }

    public void setInsets(int l, int t, int r, int b) {
        this.mInsets.set(l, t, r, b);
    }

    public void setInsets(int inset) {
        this.mInsets.set(inset, inset, inset, inset);
    }

    private int[] targetLoc = new int[2];
    private int[] viewLoc = new int[2];
    private final ArrayList<Animator> tmpAnimators = new ArrayList<>();

    //matrix补偿, 因为有scale属性之类的变化导致影响
    RectF matrixFixRect = new RectF();
    Rect rectNew = new Rect();


    private void updateRect() {
        //当有focusEvent事件传入时, updateRect不做任何事
        if (isFocusEventHappening) return;

        if (mTarget != null) {
            //目标的父类进行动画或者可视性改变
            traversalParent(mTarget, new ParentTraversalVisitor() {
                @Override
                boolean onParent(View v) {
                    if (v.getAnimation() != null && v.getAnimation().hasStarted() && !v.getAnimation().hasEnded()) {
                        //有父类开始了动画
                        return true;
                    }

                    if (v.getVisibility() != VISIBLE) {
                        //有父类不可见
                        return true;
                    }
                    return false;
                }

                @Override
                void onEnd(boolean isTravelAll, View v) {
                    if (isTravelAll) {
                        //整个视图链没问题
                        if (getVisibility() != VISIBLE) {
                            setVisibility(VISIBLE);
                        }
                    } else {
                        //视图链有问题
                        if (getVisibility() == VISIBLE) {
                            setVisibility(INVISIBLE);
                        }
                    }
                }
            });

            if (mTarget.getVisibility() != VISIBLE && getVisibility() != INVISIBLE) {
                //目标不可见
                setVisibility(INVISIBLE);
            }

            //计算target目前实时位置
            layoutOn(mTarget);
        }
    }

    private void layoutOn(View view) {
        if (view != null) {
            view.getLocationOnScreen(targetLoc);
            matrixFixRect.setEmpty();
            view.getMatrix().mapRect(matrixFixRect);
            if (isFilterMatrix) {
                int x = (int) (targetLoc[0] - matrixFixRect.left);
                int y = (int) (targetLoc[1] - matrixFixRect.top);
                rectNew.set(x, y, x + view.getWidth(), y + view.getHeight());
            } else {
                rectNew.set(targetLoc[0], targetLoc[1], (int) (targetLoc[0] + view.getWidth() + -(matrixFixRect.left + matrixFixRect.right)), (int) (targetLoc[1] + view.getHeight() + -(matrixFixRect.top + matrixFixRect.bottom)));
            }
            adjustWithInsets(rectNew);
            if (!rectNew.equals(mRect)) {
                mRect.set(rectNew);
                layout();
            }
        }
    }

    private void traversalParent(@NonNull View v, @NonNull ParentTraversalVisitor visitor) {
        View parent = v;
        boolean tillTop;
        while (!(tillTop = isTopView(parent))) {
            boolean ref = visitor.onParent(parent);
            if (ref) break;
            parent = (View) parent.getParent();
        }
        visitor.onEnd(tillTop, parent);
    }

    abstract class ParentTraversalVisitor {
        abstract boolean onParent(View v);

        abstract void onEnd(boolean isTravelAll, View v);
    }


    private void to(View target) {
        isFocusEventHappening = false;
        if (mLastVisibleAnimator != null) {
            mLastVisibleAnimator.cancel();
        }
        mChoreographer.removeFrameCallback(mTargetCb);

        if (target.getAnimation() != null && target.getAnimation().hasStarted() && !target.getAnimation().hasEnded()) {
            setVisibility(INVISIBLE);
        } else {
            setVisibility(VISIBLE);
        }

        if (mAnimSet != null && mAnimSet.isRunning()) {
            if (getAnimation() != null) getAnimation().cancel();
            mAnimSet.removeAllListeners();
            mAnimSet.end();
            mAnimSet = null;
        }

        View oldTarget = mTarget;
        if (oldTarget != null) oldTarget.removeOnLayoutChangeListener(mOnLayoutChangeListener);

        mTarget = target;
        mPendingTarget = null;

        if (target.getWidth() == 0 || target.getHeight() == 0) {
            //还没布局过的或者尺寸异常的
            target.addOnLayoutChangeListener(mOnLayoutChangeListener);
            return;
        }

        if (!isFitToAnimateForTransformLocation(oldTarget)) {
            setAlpha(0);
            mVisibleAnimator = createVisibleAnimator();
            if (pair_vp_frame != null) pair_vp_frame.second = pair_vp_frame.first.getScrollX();
            mChoreographer.postFrameCallback(mTargetCb);
            return;
        }

        if (!isVisible()) {
            //当翻页后还没来得及Visible的时候就开始下一个焦点变换,此时位置信息还是之前的, 没更新
            //位置调整到上一个的target的位置
            layoutOn(oldTarget);
        }

        mAnimSet = new AnimatorSet();
        tmpAnimators.add(createTransformAnimator());
        mAnimSet.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationStart(Animator animation) {
                setAlpha(1);
            }

            @Override
            public void onAnimationEnd(Animator animation) {
                mAnimSet = null;
                if (mPendingTarget != null) return;
                afterHighLight(mTarget);
                if (mFocusedAnimation != null) {
                    startAnimation(mFocusedAnimation);
                }
                if (pair_vp_frame != null) pair_vp_frame.second = pair_vp_frame.first.getScrollX();
                mChoreographer.postFrameCallbackDelayed(mTargetCb, 17 * 6);   //wait for layout rendering done
            }
        });
        mAnimSet.playTogether(tmpAnimators);
        tmpAnimators.clear();
        mAnimSet.start();
    }

    public void adjustWithInsets(Rect rect) {
        rect.set(rect.left - mInsets.left, rect.top - mInsets.top, rect.right + mInsets.right, rect.bottom + mInsets.bottom);
    }

    //可能有种情况, ViewRoot底下那个不是R.id.content, 然后整个视图树中没有content
    //所以要优先判断父节点是否是View
    private boolean isTopView(View view) {
        if (view != null) {
            return !(view.getParent() instanceof View) || view.getId() == android.R.id.content;
        }
        return true;
    }

    //是否适合去做位移动画
    private boolean isFitToAnimateForTransformLocation(View oldTarget) {
        if (mViewPagerToBinded != null) {
            //有绑定ViewPager
            View parent = mTarget;
            View oldParent = oldTarget;
            boolean newIsDone = isTopView(parent);
            boolean oldIsDone = isTopView(oldParent);

            while (!oldIsDone || !newIsDone) {
                if (!newIsDone) {
                    parent = parent != null ? (View) parent.getParent() : null;
                    if (parent instanceof ViewPager) {
                        newIsDone = true;
                    } else if (isTopView(parent)) {
                        parent = null;
                        newIsDone = true;
                    } else if (parent == null) {
                        newIsDone = true;
                    }
                }

                if (!oldIsDone) {
                    oldParent = oldParent != null ? (View) oldParent.getParent() : null;
                    if (oldParent instanceof ViewPager) {
                        oldIsDone = true;
                    } else if (isTopView(oldParent)) {
                        oldIsDone = true;
                        oldParent = null;
                    } else if (oldParent == null) {
                        oldIsDone = true;
                    }
                }
            }

            if (parent != null) {
                //新 旧 都是在ViewPager中, 才考虑
                isTargetInViewPager = true;
                if (oldParent != null) {
                    if (getBelowViewPagerParent(oldTarget) != getBelowViewPagerParent(mTarget)) {
                        return false;
                    }
                }
            } else {
                isTargetInViewPager = false;
            }
        }
        return true;
    }

    private View getBelowViewPagerParent(View v) {
        View tmp = null;
        while (!isTopView(v) && !(v.getParent() instanceof ViewPager)) {
            v = (View) v.getParent();
            tmp = v;
        }
        return tmp;
    }

    private boolean isVisible() {
        return getAlpha() * 100 > 0;
    }

    private ObjectAnimator createVisibleAnimator() {
        ObjectAnimator animator = ObjectAnimator.ofFloat(this, "alpha", 0.5f, 1f).setDuration(100);
        animator.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationEnd(Animator animation) {
                postOnAnimation(() -> {
                    if (mFocusedAnimation != null) {
                        startAnimation(mFocusedAnimation);
                    }
                });
            }

            @Override
            public void onAnimationCancel(Animator animation) {
                super.onAnimationCancel(animation);
                setAlpha(0);
            }
        });
        return animator;
    }


    private Rect rectTarget = new Rect();

    /**
     * 根据不同的容器进行位置的调整
     */
    private void adjustTargetLoc(int[] loc) {
        View parent = mTarget;
        rectTarget.set(0, 0, mTarget.getWidth(), mTarget.getHeight());
        int dx = 0;
        int dy = 0;
        while (!isTopView(parent)) {
            if (parent.getParent() instanceof BaseGridView) {
                //target在BaseGridView中
                BaseGridView gv = (BaseGridView) parent.getParent();
                int[] tmp = new int[2];
                gv.getViewSelectedOffsets(parent, tmp);   //BaseGridView 是以自己的子View去做Align计算的, 不是FocusView
                dx += -tmp[0];
                dy += -tmp[1];
            } else if (parent.getParent() instanceof RecyclerView) {
                //target在RecyclerView中
                RecyclerView rv = (RecyclerView) parent.getParent();
                rv.offsetDescendantRectToMyCoords(mTarget, rectTarget);
                //dx = -Math.min(rectTarget.left - rv.getLayoutManager().getLeftDecorationWidth(parent), 0);
                int dx_left = Math.min((rectTarget.left - rv.getLayoutManager().getLeftDecorationWidth(parent)), 0);
                int dx_right = Math.min(rv.getWidth() - (rectTarget.right + rv.getLayoutManager().getRightDecorationWidth(parent)), 0);
                dx = Math.abs(dx_left) > Math.abs(dx_right) ? -dx_left : dx_right;
                int dy_top = Math.min((rectTarget.top - rv.getLayoutManager().getTopDecorationHeight(parent)), 0);
                int dy_bottom = Math.min(rv.getHeight() - (rectTarget.bottom + rv.getLayoutManager().getBottomDecorationHeight(parent)), 0);
                dy = Math.abs(dy_top) > Math.abs(dy_bottom) ? -dy_top : dy_bottom;
                break;
            } else if (parent.getParent() instanceof ScrollView) {
                break;
            }
            parent = (View) parent.getParent();
        }
        loc[0] = loc[0] + dx;
        loc[1] = loc[1] + dy;
    }

    private Rect rectView = new Rect();

    private void adjustViewLoc(final int[] loc) {
        rectView.set(0, 0, mTarget.getWidth(), mTarget.getHeight());
        traversalParent(mTarget, new ParentTraversalVisitor() {
            @Override
            boolean onParent(View v) {
                if (v instanceof RecyclerView) {
                    ((RecyclerView) v).offsetDescendantRectToMyCoords(mTarget, rectView);
                    loc[1] -= rectView.top;
                    return true;
                }
                return false;
            }

            @Override
            void onEnd(boolean isTravelAll, View v) {

            }
        });
        loc[0] = Math.max(0, loc[0]);
        loc[1] = Math.max(0, loc[1]);
    }

    private ValueAnimator createTransformAnimator() {
        ValueAnimator animator = ValueAnimator.ofFloat(0, 1f);
        animator.setDuration(150);
        animator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator animation) {
                if (!mTarget.isAttachedToWindow()) return;
                mTarget.getLocationOnScreen(targetLoc);
                adjustTargetLoc(targetLoc);
                getLocationOnScreen(viewLoc);
                matrixFixRect.setEmpty();
                mTarget.getMatrix().mapRect(matrixFixRect);
                float tmpX = targetLoc[0] - mInsets.left;
                tmpX -= isFilterMatrix ? matrixFixRect.left : 0;
                int targetX = (int) Math.ceil(tmpX);
                float tmpY = targetLoc[1] - mInsets.top;
                tmpY -= isFilterMatrix ? matrixFixRect.top : 0;
                int targetY = (int) tmpY;

                int targetWidth = mTarget.getWidth() + mInsets.left + mInsets.right;
                targetWidth += isFilterMatrix ? 0 : -(matrixFixRect.left + matrixFixRect.right);
                int targetHeight = mTarget.getHeight() + mInsets.top + mInsets.bottom;
                targetHeight += isFilterMatrix ? 0 : -(matrixFixRect.top + matrixFixRect.bottom);

                Object cbRef = mTarget.getTag(KEY_OFFSET_CALLBACK);
                if (cbRef != null) {
                    FocusHighLightClosure closure = (FocusHighLightClosure) cbRef;
                    Rect finalRect = new Rect(targetX, targetY, targetX + targetWidth, targetY + targetHeight);
                    closure.call(finalRect);
                    targetX = finalRect.left;
                    targetY = finalRect.top;
                    targetWidth = finalRect.width();
                    targetHeight = finalRect.height();
                    mTarget.setTag(KEY_OFFSET_CALLBACK, null);
                }

                final int viewX = viewLoc[0];
                final int viewY = !isVisible() && mViewPagerToBinded != null ? targetY : viewLoc[1];
                final int deltaX = targetX - viewX;
                final int deltaY = targetY - viewY;


                final int viewWidth = mRect.width();
                final int viewHeight = mRect.height();
                final int deltaWidth = targetWidth - viewWidth;
                final int deltaHeight = targetHeight - viewHeight;

                int curX = (int) (viewX + deltaX * animation.getAnimatedFraction());
                int curY = (int) (viewY + deltaY * animation.getAnimatedFraction());
                int curWidth = (int) (viewWidth + deltaWidth * animation.getAnimatedFraction());
                int curHeight = (int) (viewHeight + deltaHeight * animation.getAnimatedFraction());
                FrameLayout.LayoutParams lp = (FrameLayout.LayoutParams) getLayoutParams();
                lp.width = curWidth;
                lp.height = curHeight;
                mRect.set(curX, curY, curX + curWidth, curY + curHeight);
                layout();
            }
        });
        return animator;
    }


    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        setMeasuredDimension(mRect.width(), mRect.height());
    }

    @Override
    public void requestLayout() {
        super.requestLayout();
    }

    private void layout() {
        layout(0, 0, 0, 0);
    }

    @Override
    public void layout(int l, int t, int r, int b) {
        // Log.v("ttt", "left " + mRect.left);
        super.layout(mRect.left, mRect.top, mRect.right, mRect.bottom);
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (getVisibility() == INVISIBLE || (mRect.width() == 0 && mRect.height() == 0)) return;
        super.onDraw(canvas);
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
    }

    @Override
    public void onActivityStarted(Activity activity) {
        if (activity == mActivity && mTarget != null) {
            mChoreographer.postFrameCallback(mTargetCb);
        }
    }

    @Override
    public void onActivityResumed(Activity activity) {

    }

    @Override
    public void onActivityPaused(Activity activity) {

    }

    @Override
    public void onActivityStopped(Activity activity) {
        if (activity == mActivity) mChoreographer.removeFrameCallback(mTargetCb);
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        if (activity == mActivity) out();
    }


    /**
     * 接入Frame回调
     *
     * @author Damon
     */
    class TargetWatcherCallBack implements Choreographer.FrameCallback {

        @Override
        public void doFrame(long frameTimeNanos) {
            Log.i("liao",mActivity  + " ---doFrame ");
            if (mFirstChain != null) mFirstChain.processOnFrameIn();
            mChoreographer.postFrameCallbackDelayed(this, 32);
        }
    }

    class Pair<First, Second> {
        First first;
        Second second;

        public Pair(First first, Second second) {
            this.first = first;
            this.second = second;
        }
    }

    /**
     * 针对ViewPager使用场景下做的处理
     * 比如透明度操作
     */
    class ViewPagerWork extends Chain {
        final int MSG_ViewPageStatic = 0x86;
        final int Vaild_WaitTime = 100;
        Long staticTime = -1L;

        ViewPagerWork(Chain next) {
            super(next);
        }

        @Override
        protected void onEventIn() {
            mHandler.removeMessages(MSG_ViewPageStatic);
        }

        @Override
        protected boolean onProcess() {
            if (mViewPagerToBinded == null) return false;
            if (!isTargetInViewPager) {
                //停止处
                return false;
            }

            if (isStaticState()) {
                if (!mHandler.hasMessages(MSG_ViewPageStatic)) {
                    if (staticTime <= 0) {
                        staticTime = System.currentTimeMillis();
                        mHandler.sendEmptyMessageDelayed(MSG_ViewPageStatic, Vaild_WaitTime);
                    } else {
                        if (mVisibleAnimator != null) {
                            mVisibleAnimator.start();
                            mLastVisibleAnimator = mVisibleAnimator;
                            mVisibleAnimator = null;
                        }
                    }
                }
            } else {
                staticTime = -1L;
                setAlpha(0);
                if (getAnimation() != null) {
                    getAnimation().cancel();
                    setAnimation(null);
                }
                if (mLastVisibleAnimator != null) {
                    mLastVisibleAnimator.removeAllListeners();
                    mLastVisibleAnimator.cancel();
                }
                if (mVisibleAnimator == null) mVisibleAnimator = createVisibleAnimator();
                if (mHandler.hasMessages(MSG_ViewPageStatic))
                    mHandler.removeMessages(MSG_ViewPageStatic);
            }
            return false;
        }

        private boolean isStaticState() {
            if (pair_vp_frame != null && pair_vp_frame.first.getScrollX() != pair_vp_frame.second) {
                pair_vp_frame.second = pair_vp_frame.first.getScrollX();
                return false;
            }
            return true;
        }
    }

    /**
     * 一般处理 即所有场景下都需要做的处理
     * 比如 位置更新
     */
    class NormalWork extends Chain {
        final int MSG_TargetStatic = 0x286;
        int mStaticVaild_WaitingTime = 3000;
        Long staticTime = -1L;

        NormalWork(Chain next) {
            super(next);
        }

        @Override
        protected void onEventIn() {
            mHandler.removeMessages(MSG_TargetStatic);
            staticTime = -1L;
        }

        public NormalWork setVaildWaitingTime(int timeMs) {
            mStaticVaild_WaitingTime = timeMs;
            return this;
        }

        @Override
        protected boolean onProcess() {
            if (mAnimSet != null) {
                //停止处
                return true;
            }

            if (isStaticState()) {
                if (!mHandler.hasMessages(MSG_TargetStatic)) {
                    if (staticTime <= 0) {
                        staticTime = System.currentTimeMillis();
                        mHandler.sendEmptyMessageDelayed(MSG_TargetStatic, mStaticVaild_WaitingTime);
                    } else {
                        updateRect();
                    }
                }
            } else {
                staticTime = -1L;
            }
            return true;
        }

        private boolean isStaticState() {
            return !mHandler.hasMessages(EVENT_FOCUS_CHANGE);
        }

    }

    /**
     * Frame回调处理 链式
     * 为了分离 职责代码
     */
    abstract class Chain {
        private Chain next;

        Chain(Chain next) {
            this.next = next;
        }

        public final void processOnEventIn() {
            onEventIn();
            if (next != null) {
                next.processOnEventIn();
            }
        }

        public final void processOnFrameIn() {
            if (!onProcess() && next != null) {
                next.processOnFrameIn();
            }
        }

        abstract protected void onEventIn();

        abstract protected boolean onProcess();
    }


    public abstract static class FocusHighLightClosure extends ICallBack<Rect> {

    }
}