
/**
 * Copyright (c) 2015-present, Facebook, Inc. All rights reserved.
 *
 * <p>You are hereby granted a non-exclusive, worldwide, royalty-free license to use, copy, modify,
 * and distribute this software in source code or binary form for use in connection with the web
 * services and APIs provided by Facebook.
 *
 * <p>As with any software that integrates with the Facebook platform, your use of this software is
 * subject to the Facebook Developer Principles and Policies [http://developers.facebook.com/policy/].
 * This copyright notice shall be included in all copies or substantial portions of the software.
 *
 * <p>THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package com.facebook.ads.sdk.signal;

import com.facebook.ads.sdk.serverside.GenderEnum;

public class UserData {

    private com.facebook.ads.sdk.businessdataapi.UserData businessDataUserData = new com.facebook.ads.sdk.businessdataapi.UserData();

    private com.facebook.ads.sdk.serverside.UserData serverSideUserData = new com.facebook.ads.sdk.serverside.UserData();

    public UserData() {
    }

    /**
     * @param email           An email address, in lowercase.
     * @param phone           A phone number. Include only digits with country code,
     *                        area code, and number.
     * @param firstName       A first name in lowercase.
     * @param lastName        A last name in lowercase.
     * @param dateOfBirth     A date of birth given as year, month, and day in
     *                        YYYYMMDD format.
     * @param city            A city in lower-case without spaces or punctuation.
     * @param state           A two-letter state code in lowercase.
     * @param countryCode     A two-letter country code in lowercase.
     * @param zipcode         Postal code of the city in your country standard
     * @param externalId      Any unique ID from the advertiser,
     * @param gender          Gender, in lowercase. Either f or m.
     * @param clientIpAddress The IP address of the browser corresponding to the
     *                        event.
     * @param clientUserAgent The user agent for the browser corresponding to the
     *                        event.
     * @param fbp             The Facebook click ID value stored in the _fbc browser
     *                        cookie under your domain.
     * @param fbc             The Facebook browser ID value stored in the _fbp
     *                        browser cookie under your domain.
     * @param subscriptionId  The subscription ID for the user in this transaction.
     * @param fbLoginId       The FB login ID for the user.
     * @param leadId          The Id associated with a lead generated by Facebook's
     *                        Lead Ads.
     * @param dobd            The date of birth day in DD format.
     * @param dobm            The date of birth month in MM format.
     * @param doby            The date of birth year in YYYY format.
     * @param f5first         The first 5 characters of the first name.
     * @param f5last          The first 5 characters of the last name.
     * @param fi              The first Name Initial
     * @param address         An physical address.
     */
    public UserData(String email, String phone, GenderEnum gender, String dateOfBirth, String lastName,
            String firstName, String city, String state, String zipcode, String countryCode, String externalId,
            String clientIpAddress, String clientUserAgent, String fbc, String fbp, String subscriptionId,
            String fbLoginId, String leadId, String f5first, String f5last, String fi, String dobd, String dobm,
            String doby, String address) {

        this.businessDataUserData.setEmail(email);
        this.businessDataUserData.setPhone(phone);
        this.businessDataUserData.setDateOfBirth(dateOfBirth);
        this.businessDataUserData.setLastName(lastName);
        this.businessDataUserData.setFirstName(firstName);
        this.businessDataUserData.setCity(city);
        this.businessDataUserData.setState(state);
        this.businessDataUserData.setZipcode(zipcode);
        this.businessDataUserData.setCountryCode(countryCode);
        this.businessDataUserData.setExternalId(externalId);
        this.businessDataUserData.setAddress(address);

        this.serverSideUserData.setEmail(email);
        this.serverSideUserData.setPhone(phone);
        this.serverSideUserData.setGender(gender);
        this.serverSideUserData.setDateOfBirth(dateOfBirth);
        this.serverSideUserData.setLastName(lastName);
        this.serverSideUserData.setFirstName(firstName);
        this.serverSideUserData.setCity(city);
        this.serverSideUserData.setState(state);
        this.serverSideUserData.setZipcode(zipcode);
        this.serverSideUserData.setCountryCode(countryCode);
        this.serverSideUserData.setExternalId(externalId);
        this.serverSideUserData.setClientIpAddress(clientIpAddress);
        this.serverSideUserData.setClientUserAgent(clientUserAgent);
        this.serverSideUserData.setFbc(fbc);
        this.serverSideUserData.setFbp(fbp);
        this.serverSideUserData.setSubscriptionId(subscriptionId);
        this.serverSideUserData.setFbLoginId(fbLoginId);
        this.serverSideUserData.setLeadId(leadId);
        this.serverSideUserData.setF5first(f5first);
        this.serverSideUserData.setF5last(f5last);
        this.serverSideUserData.setFi(fi);
        this.serverSideUserData.setDobd(dobd);
        this.serverSideUserData.setDobm(dobm);
        this.serverSideUserData.setDoby(doby);

    }

    /**
     * Set an email address, in lowercase.
     *
     * Example: joe@eg.com
     *
     * @param email an email address
     * @return UserData
     */
    public UserData email(String email) {
        this.businessDataUserData.setEmail(email);
        this.serverSideUserData.setEmail(email);
        return this;
    }

    /**
     * An email address, in lowercase.
     *
     * @return email
     */
    public SignalResponse<String, String> getEmail() {
        return new SignalResponse<String, String>(this.serverSideUserData.getEmail(),
                this.businessDataUserData.getEmail());
    }

    /**
     * Set an email address, in lowercase.
     *
     * <p>
     * Example: joe@eg.com
     *
     * @param email an email address
     */
    public void setEmail(String email) {
        this.businessDataUserData.setEmail(email);
        this.serverSideUserData.setEmail(email);
    }

    /**
     * Set a phone number. Include only digits with countryCode code, area code, and
     * number.
     *
     * <p>
     * Example: 16505551212
     *
     * @param phone a phone number
     * @return UserData
     */
    public UserData phone(String phone) {
        this.businessDataUserData.setPhone(phone);
        this.serverSideUserData.setPhone(phone);
        return this;
    }

    /**
     * A phone number. Include only digits with countryCode code, area code, and
     * number.
     *
     * @return phone
     */
    public SignalResponse<String, String> getPhone() {
        return new SignalResponse<String, String>(this.serverSideUserData.getPhone(),
                this.businessDataUserData.getPhone());
    }

    /**
     * Set a phone number. Include only digits with countryCode code, area code, and
     * number.
     *
     * <p>
     * Example: 16505551212
     *
     * @param phone a phone number
     */
    public void setPhone(String phone) {
        this.businessDataUserData.setPhone(phone);
        this.serverSideUserData.setPhone(phone);
    }

    /**
     * Gender. Male or Female.
     *
     * @param gender Male or Female.
     * @return UserData
     */
    public UserData gender(GenderEnum gender) {
        this.serverSideUserData.setGender(gender);
        return this;
    }

    /**
     * Gender, Male or Female.
     *
     * @return gender
     */
    public SignalResponse<GenderEnum, GenderEnum> getGender() {
        return new SignalResponse<GenderEnum, GenderEnum>(this.serverSideUserData.getGender(), null);
    }

    /**
     * Gender, Male or Female.
     *
     * @param gender Male or Female.
     */
    public void setGender(GenderEnum gender) {
        this.serverSideUserData.setGender(gender);
    }

    /**
     * Set a date of birth given as year, month, and day.
     *
     * <p>
     * Example: 19971226 for December 26, 1997.
     *
     * @param dateOfBirth a date of birth given as year, month, and day.
     * @return UserData
     */
    public UserData dateOfBirth(String dateOfBirth) {
        this.businessDataUserData.setDateOfBirth(dateOfBirth);
        this.serverSideUserData.setDateOfBirth(dateOfBirth);
        return this;
    }

    /**
     * Return a date of birth given as year, month, and day.
     *
     * @return dateOfBirth
     */
    public SignalResponse<String, String> getDateOfBirth() {
        return new SignalResponse<String, String>(this.serverSideUserData.getDateOfBirth(),
                this.businessDataUserData.getDateOfBirth());
    }

    /**
     * A date of birth given as year, month, and day.
     *
     * <p>
     * Example: 19971226 for December 26, 1997.
     *
     * @param dateOfBirth a date of birth given as year, month, and day.
     */
    public void setDateOfBirth(String dateOfBirth) {
        this.businessDataUserData.setDateOfBirth(dateOfBirth);
        this.serverSideUserData.setDateOfBirth(dateOfBirth);
    }

    /**
     * Set a last name in lowercase.
     *
     * <p>
     * Example: smith
     *
     * @param lastName last name
     * @return UserData
     */
    public UserData lastName(String lastName) {
        this.businessDataUserData.setLastName(lastName);
        this.serverSideUserData.setLastName(lastName);
        return this;
    }

    /**
     * A last name in lowercase.
     *
     * @return lastName
     */
    public SignalResponse<String, String> getLastName() {
        return new SignalResponse<String, String>(this.serverSideUserData.getLastName(),
                this.businessDataUserData.getLastName());
    }

    /**
     * Set a last name in lowercase.
     *
     * <p>
     * Example: smith
     *
     * @param lastName last name
     */
    public void setLastName(String lastName) {
        this.businessDataUserData.setLastName(lastName);
        this.serverSideUserData.setLastName(lastName);
    }

    /**
     * Set a first name in lowercase.
     *
     * <p>
     * Example: joe
     *
     * @param firstName first name
     * @return firstName
     */
    public UserData firstName(String firstName) {
        this.businessDataUserData.setFirstName(firstName);
        this.serverSideUserData.setFirstName(firstName);
        return this;
    }

    /**
     * A first name in lowercase.
     *
     * @return firstName
     */
    public SignalResponse<String, String> getFirstName() {
        return new SignalResponse<String, String>(this.serverSideUserData.getFirstName(),
                this.businessDataUserData.getFirstName());
    }

    /**
     * Set a first name in lowercase.
     *
     * Example: joe
     *
     * @param firstName first name
     */
    public void setFirstName(String firstName) {
        this.businessDataUserData.setFirstName(firstName);
        this.serverSideUserData.setFirstName(firstName);
    }

    /**
     * Set a city in lower-case without spaces or punctuation.
     *
     * Example: menlopark
     *
     * @param city city
     * @return UserData
     */
    public UserData city(String city) {
        this.businessDataUserData.setCity(city);
        this.serverSideUserData.setCity(city);
        return this;
    }

    /**
     * A city in lower-case without spaces or punctuation.
     *
     * @return city
     */
    public SignalResponse<String, String> getCity() {
        return new SignalResponse<String, String>(this.serverSideUserData.getCity(),
                this.businessDataUserData.getCity());
    }

    /**
     * Set a city in lower-case without spaces or punctuation.
     *
     * <p>
     * Example: menlopark
     *
     * @param city city
     */
    public void setCity(String city) {
        this.businessDataUserData.setCity(city);
        this.serverSideUserData.setCity(city);
    }

    /**
     * Set a two-letter state code in lowercase.
     *
     * <p>
     * Example: ca
     *
     * @param state two-letter state code
     * @return UserData
     */
    public UserData state(String state) {
        this.businessDataUserData.setState(state);
        this.serverSideUserData.setState(state);
        return this;
    }

    /**
     * A two-letter state code in lowercase.
     *
     * @return state
     */
    public SignalResponse<String, String> getState() {
        return new SignalResponse<String, String>(this.serverSideUserData.getState(),
                this.businessDataUserData.getState());
    }

    /**
     * Set a two-letter state code in lowercase.
     *
     * <p>
     * Example: ca
     *
     * @param state two-letter state code
     */
    public void setState(String state) {
        this.businessDataUserData.setState(state);
        this.serverSideUserData.setState(state);
    }

    /**
     * Set a five-digit zip code.
     *
     * <p>
     * Example: 94035
     *
     * @param zipcode five-digit zip code
     * @return UserData
     */
    public UserData zipcode(String zipcode) {
        this.businessDataUserData.setZipcode(zipcode);
        this.serverSideUserData.setZipcode(zipcode);
        return this;
    }

    /**
     * A five-digit zip code.
     *
     * @return zipcode
     */
    public SignalResponse<String, String> getZipcode() {
        return new SignalResponse<String, String>(this.serverSideUserData.getZipcode(),
                this.businessDataUserData.getZipcode());
    }

    /**
     * Set a five-digit zip code.
     *
     * <p>
     * Example: 94035
     *
     * @param zipcode five-digit zip code
     */
    public void setZipcode(String zipcode) {
        this.businessDataUserData.setZipcode(zipcode);
        this.serverSideUserData.setZipcode(zipcode);
    }

    /**
     * Set a two-letter country code in lowercase.
     *
     * <p>
     * Example: us
     *
     * @param countryCode two-letter country code
     * @return UserData
     */
    public UserData countryCode(String countryCode) {
        this.businessDataUserData.setCountryCode(countryCode);
        this.serverSideUserData.setCountryCode(countryCode);
        return this;
    }

    /**
     * A two-letter country code in lowercase.
     *
     * <p>
     * Example: us
     *
     * @return countryCode
     */
    public SignalResponse<String, String> getCountryCode() {
        return new SignalResponse<String, String>(this.serverSideUserData.getCountryCode(),
                this.businessDataUserData.getCountryCode());
    }

    /**
     * Set a two-letter country code in lowercase.
     *
     * <p>
     * Example: us
     *
     * @param countryCode two-letter country code
     */
    public void setCountryCode(String countryCode) {
        this.businessDataUserData.setCountryCode(countryCode);
        this.serverSideUserData.setCountryCode(countryCode);
    }

    /**
     * Set any unique ID from the advertiser, such as loyalty membership IDs, user
     * IDs, and external cookie IDs. In the Offline Conversions API, this is known
     * as extern_id. For more information, see Offline Conversions, Providing
     * External IDs. If External ID is being sent via other channels, then it should
     * be sent in the same format via the server-side API.
     *
     * @param externalId unique ID from the advertiser
     * @return UserData
     */
    public UserData externalId(String externalId) {
        this.businessDataUserData.setExternalId(externalId);
        this.serverSideUserData.setExternalId(externalId);
        return this;
    }

    /**
     * Any unique ID from the advertiser, such as loyalty membership IDs, user IDs,
     * and external cookie IDs. In the Offline Conversions API, this is known as
     * extern_id. For more information, see Offline Conversions, Providing External
     * IDs. If External ID is being sent via other channels, then it should be sent
     * in the same format via the server-side API.
     *
     * @return externalId
     */
    public SignalResponse<String, String> getExternalId() {
        return new SignalResponse<String, String>(this.serverSideUserData.getExternalId(),
                this.businessDataUserData.getExternalId());
    }

    /**
     * Set any unique ID from the advertiser, such as loyalty membership IDs, user
     * IDs, and external cookie IDs. In the Offline Conversions API, this is known
     * as extern_id. For more information, see Offline Conversions, Providing
     * External IDs. If External ID is being sent via other channels, then it should
     * be sent in the same format via the server-side API.
     *
     * @param externalId unique ID from the advertiser
     */
    public void setExternalId(String externalId) {
        this.businessDataUserData.setExternalId(externalId);
        this.serverSideUserData.setExternalId(externalId);
    }

    /**
     * Set IP address of the browser corresponding to the event.
     *
     * @param clientIpAddress IP address of the browser corresponding to the event
     * @return UserData
     */
    public UserData clientIpAddress(String clientIpAddress) {
        this.serverSideUserData.setClientIpAddress(clientIpAddress);
        return this;
    }

    /**
     * The IP address of the browser corresponding to the event.
     *
     * @return clientIpAddress
     */
    public SignalResponse<String, String> getClientIpAddress() {
        return new SignalResponse<String, String>(this.serverSideUserData.getClientIpAddress(), null);
    }

    /**
     * Set IP address of the browser corresponding to the event.
     *
     * @param clientIpAddress IP address of the browser corresponding to the event
     */
    public void setClientIpAddress(String clientIpAddress) {
        this.serverSideUserData.setClientIpAddress(clientIpAddress);
    }

    /**
     * Set user agent for the browser corresponding to the event.
     *
     * @param clientUserAgent user agent for the browser corresponding to the event
     * @return UserData
     */
    public UserData clientUserAgent(String clientUserAgent) {
        this.serverSideUserData.setClientUserAgent(clientUserAgent);
        return this;
    }

    /**
     * The user agent for the browser corresponding to the event.
     *
     * @return clientUserAgent
     */
    public SignalResponse<String, String> getClientUserAgent() {
        return new SignalResponse<String, String>(this.serverSideUserData.getClientUserAgent(), null);
    }

    /**
     * Set user agent for the browser corresponding to the event.
     *
     * @param clientUserAgent user agent for the browser corresponding to the event
     */
    public void setClientUserAgent(String clientUserAgent) {
        this.serverSideUserData.setClientUserAgent(clientUserAgent);
    }

    /**
     * Set Facebook click ID value stored in the _fbc browser cookie under your
     * domain. See Managing fbc and fbp Parameters for how to get this value
     * (https://developers.facebook.com/docs/marketing-api/facebook-pixel/server-side-api/parameters#fbc),
     * or generate this value from a fbclid query parameter.
     *
     * @param fbc Facebook click ID
     * @return UserData
     */
    public UserData fbc(String fbc) {
        this.serverSideUserData.setFbc(fbc);
        return this;
    }

    /**
     * The Facebook click ID value stored in the _fbc browser cookie under your
     * domain. See Managing fbc and fbp Parameters for how to get this value
     * (https://developers.facebook.com/docs/marketing-api/facebook-pixel/server-side-api/parameters#fbc),
     * or generate this value from a fbclid query parameter.
     *
     * @return fbc
     */
    public SignalResponse<String, String> getFbc() {
        return new SignalResponse<String, String>(this.serverSideUserData.getFbc(), null);
    }

    /**
     * Set Facebook click ID value stored in the _fbc browser cookie under your
     * domain. See Managing fbc and fbp Parameters for how to get this value
     * (https://developers.facebook.com/docs/marketing-api/facebook-pixel/server-side-api/parameters#fbc),
     * or generate this value from a fbclid query parameter.
     *
     * @param fbc Facebook click ID
     */
    public void setFbc(String fbc) {
        this.serverSideUserData.setFbc(fbc);
    }

    /**
     * Set Facebook browser ID value stored in the _fbp browser cookie under your
     * domain. See Managing fbc and fbp Parameters for how to get this value
     * (https://developers.facebook.com/docs/marketing-api/facebook-pixel/server-side-api/parameters#fbc),
     * or generate this value from a fbclid query parameter.
     *
     * @param fbp Facebook browser ID
     * @return UserData
     */
    public UserData fbp(String fbp) {
        this.serverSideUserData.setFbp(fbp);
        return this;
    }

    /**
     * The Facebook browser ID value stored in the _fbp browser cookie under your
     * domain. See Managing fbc and fbp Parameters for how to get this value
     * (https://developers.facebook.com/docs/marketing-api/facebook-pixel/server-side-api/parameters#fbc),
     * or generate this value from a fbclid query parameter.
     *
     * @return fbp
     */
    public SignalResponse<String, String> getFbp() {
        return new SignalResponse<String, String>(this.serverSideUserData.getFbp(), null);
    }

    /**
     * Set Facebook browser ID value stored in the _fbp browser cookie under your
     * domain. See Managing fbc and fbp Parameters for how to get this value
     * (https://developers.facebook.com/docs/marketing-api/facebook-pixel/server-side-api/parameters#fbc)
     *
     * @param fbp Facebook browser ID
     */
    public void setFbp(String fbp) {
        this.serverSideUserData.setFbp(fbp);
    }

    /**
     * Set subscription ID for the user in this transaction. This is similar to the
     * order ID for an individual product.
     *
     * @param subscriptionId subscription ID for the user in this transaction
     * @return UserData
     */
    public UserData subscriptionId(String subscriptionId) {
        this.serverSideUserData.setSubscriptionId(subscriptionId);
        return this;
    }

    /**
     * The subscription ID for the user in this transaction. This is similar to the
     * order ID for an individual product.
     *
     * @return subscriptionId
     */
    public SignalResponse<String, String> getSubscriptionId() {
        return new SignalResponse<String, String>(this.serverSideUserData.getSubscriptionId(), null);
    }

    /**
     * Set subscription ID for the user in this transaction. This is similar to the
     * order ID for an individual product.
     *
     * @param subscriptionId subscription ID for the user in this transaction
     */
    public void setSubscriptionId(String subscriptionId) {
        this.serverSideUserData.setSubscriptionId(subscriptionId);
    }

    /**
     * Set Facebook Login ID for the user in this event.
     *
     * @param fbLoginId Facebook Login ID for the user in this event
     * @return UserData
     */
    public UserData fbLoginId(String fbLoginId) {
        this.serverSideUserData.setFbLoginId(fbLoginId);
        return this;
    }

    /**
     * The Facebook Login ID for the user in this event.
     *
     * @return fbLoginId
     */
    public SignalResponse<String, String> getFbLoginId() {
        return new SignalResponse<String, String>(this.serverSideUserData.getFbLoginId(), null);
    }

    /**
     * Set Facebook Login ID for the user in this event
     *
     * @param fbLoginId Facebook Login ID for the user in this event
     */
    public void setFbLoginId(String fbLoginId) {
        this.serverSideUserData.setFbLoginId(fbLoginId);
    }

    /**
     * Set LeadId for the event.
     *
     * @param leadId is the ID associated with a lead generated by Facebook's Lead
     *               Ads.
     * @return UserData
     */
    public UserData leadId(String leadId) {
        this.serverSideUserData.setLeadId(leadId);
        return this;
    }

    /**
     * The LeadId for the event.
     *
     * @return leadId
     */
    public SignalResponse<String, String> getLeadId() {
        return new SignalResponse<String, String>(this.serverSideUserData.getLeadId(), null);
    }

    /**
     * Set LeadId for the event.
     *
     * @param leadId is the ID associated with a lead generated by Facebook's Lead
     *               Ads.
     */
    public void setLeadId(String leadId) {
        this.serverSideUserData.setLeadId(leadId);
    }

    /**
     * Set f5first for the event.
     *
     * @param f5first is the first 5 letters of the first name.
     * @return UserData
     */
    public UserData f5first(String f5first) {
        this.serverSideUserData.setF5first(f5first);
        return this;
    }

    /**
     * The f5first for the event.
     *
     * @return f5first
     */
    public SignalResponse<String, String> getF5first() {
        return new SignalResponse<String, String>(this.serverSideUserData.getF5first(), null);
    }

    /**
     * Set f5first for the event.
     *
     * @param f5first is the first 5 letters of the first name.
     */
    public void setF5first(String f5first) {
        this.serverSideUserData.setF5first(f5first);
    }

    /**
     * Set f5last for the event.
     *
     * @param f5last is the first 5 letters of the last name.
     * @return UserData
     */
    public UserData f5last(String f5last) {
        this.serverSideUserData.setF5last(f5last);
        return this;
    }

    /**
     * The f5last for the event.
     *
     * @return f5last
     */
    public SignalResponse<String, String> getF5last() {
        return new SignalResponse<String, String>(this.serverSideUserData.getF5last(), null);
    }

    /**
     * Set f5last for the event.
     *
     * @param f5last is the first 5 letters of the last name.
     */
    public void setF5last(String f5last) {
        this.serverSideUserData.setF5last(f5last);
    }

    /**
     * Set fi for the event.
     *
     * @param fi is the first initial.
     * @return UserData
     */
    public UserData fi(String fi) {
        this.serverSideUserData.setFi(fi);
        return this;
    }

    /**
     * The fi for the event.
     *
     * @return fi
     */
    public SignalResponse<String, String> getFi() {
        return new SignalResponse<String, String>(this.serverSideUserData.getFi(), null);
    }

    /**
     * Set fi for the event.
     *
     * @param fi is the first initial.
     */
    public void setFi(String fi) {
        this.serverSideUserData.setFi(fi);
    }

    /**
     * Set dobd for the event.
     *
     * @param dobd is the date of birth day for the event.
     * @return UserData
     */
    public UserData dobd(String dobd) {
        this.serverSideUserData.setDobd(dobd);
        return this;
    }

    /**
     * The dobd for the event.
     *
     * @return dobd
     */
    public SignalResponse<String, String> getDobd() {
        return new SignalResponse<String, String>(this.serverSideUserData.getDobd(), null);
    }

    /**
     * Set dobd for the event.
     *
     * @param dobd is the date of birth day for the event.
     */
    public void setDobd(String dobd) {
        this.serverSideUserData.setDobd(dobd);
    }

    /**
     * Set dobm for the event.
     *
     * @param dobm is the date of birth month for the event.
     * @return UserData
     */
    public UserData dobm(String dobm) {
        this.serverSideUserData.setDobm(dobm);
        return this;
    }

    /**
     * The dobm for the event.
     *
     * @return dobm
     */
    public SignalResponse<String, String> getDobm() {
        return new SignalResponse<String, String>(this.serverSideUserData.getDobm(), null);
    }

    /**
     * Set dobm for the event.
     *
     * @param dobm is the date of birth month for the event.
     */
    public void setDobm(String dobm) {
        this.serverSideUserData.setDobm(dobm);
    }

    /**
     * Set doby for the event.
     *
     * @param doby is the date of birth year for the event.
     * @return UserData
     */
    public UserData doby(String doby) {
        this.serverSideUserData.setDoby(doby);
        return this;
    }

    /**
     * The doby for the event.
     *
     * @return doby
     */
    public SignalResponse<String, String> getDoby() {
        return new SignalResponse<String, String>(this.serverSideUserData.getDoby(), null);
    }

    /**
     * Set doby for the event.
     *
     * @param doby is the date of birth year for the event.
     */
    public void setDoby(String doby) {
        this.serverSideUserData.setDoby(doby);
    }

    /**
     * Set physical address for the event.
     *
     * @param address is the physical address for the event
     * @return UserData
     */
    public UserData address(String address) {
        this.businessDataUserData.setAddress(address);
        return this;
    }

    /**
     * The physical address for the event.
     *
     * @return doby
     */
    public SignalResponse<String, String> getAddress() {
        return new SignalResponse<String, String>(null, this.businessDataUserData.getAddress());
    }

    /**
     * * The physical address for the event.
     *
     * @param address is the physical address for the event
     */
    public void setAddress(String address) {
        this.businessDataUserData.setAddress(address);
    }

    /**
     * Gets user data for business data api
     *
     * @return com.facebook.ads.sdk.businessdataapi.UserData
     */
    public com.facebook.ads.sdk.businessdataapi.UserData getBusinessDataUserData() {
        return this.businessDataUserData;
    }

    /**
     * Gets user data for conversion api
     *
     * @return com.facebook.ads.sdk.businessdataapi.UserData
     */
    public com.facebook.ads.sdk.serverside.UserData getServerSideUserData() {
        return this.serverSideUserData;
    }

}
