/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.bag.ImmutableBag;
import com.gs.collections.api.bag.primitive.ImmutableFloatBag;
import com.gs.collections.api.bag.primitive.FloatBag;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatIntProcedure;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.bag.mutable.HashBag;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.factory.primitive.FloatBags;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import net.jcip.annotations.Immutable;

/**
 * ImmutableFloatSingletonBag is an optimization for {@link ImmutableFloatBag} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonBag.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableFloatSingletonBag implements ImmutableFloatBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final float element1;

    ImmutableFloatSingletonBag (float element)
    {
        this.element1 = element;
    }

    public ImmutableFloatBag newWith(float element)
    {
        return FloatBags.immutable.with(this.element1, element);
    }

    public ImmutableFloatBag newWithout(float element)
    {
        return Float.compare(this.element1, element) == 0 ? FloatBags.immutable.with() : this;
    }

    public ImmutableFloatBag newWithAll(FloatIterable elements)
    {
        return FloatHashBag.newBag(elements).with(this.element1).toImmutable();
    }

    public ImmutableFloatBag newWithoutAll(FloatIterable elements)
    {
        return elements.contains(this.element1) ? FloatBags.immutable.with() : this;
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public boolean contains(float value)
    {
        return Float.compare(this.element1, value) == 0;
    }

    public boolean containsAll(FloatIterable source)
    {
        for (FloatIterator iterator = source.floatIterator(); iterator.hasNext(); )
        {
            if (Float.compare(this.element1, iterator.next()) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(float... source)
    {
        for (float value : source)
        {
            if (Float.compare(this.element1, value) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(FloatProcedure procedure)
    {
        procedure.value(this.element1);
    }

    public ImmutableFloatBag select(FloatPredicate predicate)
    {
        return predicate.accept(this.element1) ? FloatHashBag.newBagWith(this.element1).toImmutable()
                : new FloatHashBag().toImmutable();
    }

    public ImmutableFloatBag reject(FloatPredicate predicate)
    {
        return predicate.accept(this.element1) ? new FloatHashBag().toImmutable()
                : FloatHashBag.newBagWith(this.element1).toImmutable();
    }

    public <V> ImmutableBag<V> collect(FloatToObjectFunction<? extends V> function)
    {
        return HashBag.newBagWith(function.valueOf(this.element1)).toImmutable();
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newListWith(this.element1);
    }

    public int sizeDistinct()
    {
        return 1;
    }

    public int occurrencesOf(float item)
    {
        return Float.compare(this.element1, item) == 0 ? 1 : 0;
    }

    public void forEachWithOccurrences(FloatIntProcedure floatIntProcedure)
    {
        floatIntProcedure.value(this.element1, 1);
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        return predicate.accept(this.element1) ? this.element1 : ifNone;
    }

    public int count(FloatPredicate predicate)
    {
        return predicate.accept(this.element1) ? 1 : 0;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public double sum()
    {
        return this.element1;
    }

    public float min()
    {
        return this.element1;
    }

    public float max()
    {
        return this.element1;
    }

    public float maxIfEmpty(float defaultValue)
    {
        return this.element1;
    }

    public float minIfEmpty(float defaultValue)
    {
        return this.element1;
    }

    public double average()
    {
        return this.element1;
    }

    public double median()
    {
        return this.element1;
    }

    public float[] toSortedArray()
    {
        return new float[]{this.element1};
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newListWith(this.element1);
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return !predicate.accept(this.element1);
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element1);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof FloatBag))
        {
            return false;
        }
        FloatBag bag = (FloatBag) obj;
        if (bag.size() != 1)
        {
            return false;
        }
        return this.occurrencesOf(this.element1) == bag.occurrencesOf(this.element1);
    }

    @Override
    public int hashCode()
    {
        return Float.floatToIntBits(this.element1) ^ 1;
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSetWith(this.element1);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBagWith(this.element1);
    }

    public ImmutableFloatBag toImmutable()
    {
        return this;
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public float[] toArray()
    {
        return new float[]{this.element1};
    }

    @Override
    public String toString()
    {
        return '[' + this.makeString() + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element1));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public FloatIterator floatIterator()
    {
        return FloatHashBag.newBagWith(this.element1).floatIterator();
    }
}

