/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import java.io.IOException;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import com.gs.collections.impl.block.factory.primitive.CharPredicates;

/**
 * This file was automatically generated from template file selectPrimitiveIterable.stg.
 */
public class SelectCharIterable
    extends AbstractLazyCharIterable
{
    private final CharIterable delegate;
    private final CharPredicate predicate;

    public SelectCharIterable(CharIterable delegate, CharPredicate predicate)
    {
        this.delegate = delegate;
        this.predicate = predicate;
    }

    public CharIterator charIterator()
    {
        return new SelectCharIterator(this.delegate, this.predicate);
    }

    public void forEach(CharProcedure procedure)
    {
        this.delegate.forEach(new IfCharProcedure(procedure));
    }

    @Override
    public int size()
    {
        return this.delegate.count(this.predicate);
    }

    @Override
    public boolean isEmpty()
    {
        return !this.charIterator().hasNext();
    }

    @Override
    public boolean notEmpty()
    {
        return this.charIterator().hasNext();
    }

    @Override
    public int count(CharPredicate predicate)
    {
        CountCharProcedure countCharProcedure = new CountCharProcedure(predicate);
        this.forEach(countCharProcedure);
        return countCharProcedure.getCount();
    }

    @Override
    public boolean anySatisfy(CharPredicate predicate)
    {
        return this.delegate.anySatisfy(CharPredicates.and(this.predicate, predicate));
    }

    @Override
    public boolean allSatisfy(CharPredicate predicate)
    {
        return this.noneSatisfy(CharPredicates.not(predicate));
    }

    @Override
    public boolean noneSatisfy(CharPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

        public long sum()
        {
            long sum = 0L;
            for (CharIterator charIterator = this.charIterator(); charIterator.hasNext() ;)
            {
                sum += charIterator.next();
            }
            return sum;
        }

        public char max()
        {
            CharIterator charIterator = this.charIterator();
            char max = charIterator.next();
            while (charIterator.hasNext())
            {
                max = (char) Math.max(max, charIterator.next());
            }
            return max;
        }

        public char min()
        {
            CharIterator charIterator = this.charIterator();
            char min = charIterator.next();
            while (charIterator.hasNext())
            {
                min = (char) Math.min(min, charIterator.next());
            }
            return min;
        }

        public char minIfEmpty(char defaultValue)
        {
            try
            {
                return this.min();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public char maxIfEmpty(char defaultValue)
        {
            try
            {
                return this.max();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double)this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            char[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                char first = sortedArray[middleIndex];
                char second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public char[] toSortedArray()
        {
            char[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableCharList toSortedList()
        {
            return CharArrayList.newList(this).sortThis();
        }

    @Override
    public char[] toArray()
    {
        final char[] array = new char[this.size()];
        this.forEach(new CharProcedure()
        {
            @SuppressWarnings("FieldMayBeFinal")
            private int index = 0;
            public void value(char each)
            {
                array[this.index++] = each;
            }
        });
        return array;
    }

    @Override
    public boolean containsAll(char... source)
    {
        for (char value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean containsAll(CharIterable source)
    {
        for (CharIterator iterator = source.charIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    @Override
    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    @Override
    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    private static final class CountCharProcedure implements CharProcedure
    {
        private static final long serialVersionUID = 1L;
        private final CharPredicate predicate;
        private int counter = 0;

        private CountCharProcedure(CharPredicate predicate)
        {
            this.predicate = predicate;
        }

        public void value(char each)
        {
            if (this.predicate.accept(each))
            {
                this.counter++;
            }
        }

        public int getCount()
        {
            return this.counter;
        }
    }

    private final class IfCharProcedure implements CharProcedure
    {
        private static final long serialVersionUID = 1L;
        private final CharProcedure procedure;

        private IfCharProcedure(CharProcedure procedure)
        {
            this.procedure = procedure;
        }

        public void value(char each)
        {
            if (SelectCharIterable.this.predicate.accept(each))
            {
                this.procedure.value(each);
            }
        }
    }

    private static final class SelectCharIterator
            implements CharIterator
    {
        private final CharIterator iterator;
        private final CharPredicate predicate;
        private char next;
        private boolean verifiedHasNext = false;

        private SelectCharIterator(CharIterable iterable, CharPredicate predicate)
        {
            this(iterable.charIterator(), predicate);
        }

        private SelectCharIterator(CharIterator iterator, CharPredicate predicate)
        {
            this.iterator = iterator;
            this.predicate = predicate;
        }

        public boolean hasNext()
        {
            if (this.verifiedHasNext)
            {
                return true;
            }
            while (this.iterator.hasNext())
            {
                char temp = this.iterator.next();
                if (this.predicate.accept(temp))
                {
                    this.next = temp;
                    this.verifiedHasNext = true;
                    return true;
                }
            }
            return false;
        }

        public char next()
        {
            if (this.verifiedHasNext || this.hasNext())
            {
                this.verifiedHasNext = false;
                return this.next;
            }
            throw new NoSuchElementException();
        }
    }
}
