/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.ByteIntProcedure;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.ByteList;
import com.gs.collections.api.list.primitive.ImmutableByteList;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedByteCollection;
import com.gs.collections.impl.factory.primitive.ByteLists;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.lazy.primitive.ReverseByteIterable;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableByteList}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link ByteIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveList.stg.
 * </p>
 *
 * @see MutableByteList#asSynchronized()
 * @see MutableList#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedByteList
        extends AbstractSynchronizedByteCollection
        implements MutableByteList
{
    private static final long serialVersionUID = 1L;

    SynchronizedByteList(MutableByteList list)
    {
        super(list);
    }

    SynchronizedByteList(MutableByteList list, Object newLock)
    {
        super(list, newLock);
    }

    @GuardedBy("getLock()")
    private MutableByteList getMutableByteList()
    {
        return (MutableByteList) this.getByteCollection();
    }

    public byte get(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().get(index);
        }
    }

    public byte getFirst()
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().getFirst();
        }
    }

    public byte getLast()
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().getLast();
        }
    }

    public int indexOf(byte value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().indexOf(value);
        }
    }

    public int lastIndexOf(byte value)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().lastIndexOf(value);
        }
    }

    public void addAtIndex(int index, byte element)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().addAtIndex(index, element);
        }
    }

    public boolean addAllAtIndex(int index, byte... source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().addAllAtIndex(index, source);
        }
    }

    public boolean addAllAtIndex(int index, ByteIterable source)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().addAllAtIndex(index, source);
        }
    }

    public byte removeAtIndex(int index)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().removeAtIndex(index);
        }
    }

    public byte set(int index, byte element)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().set(index, element);
        }
    }

    @Override
    public SynchronizedByteList with(byte element)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedByteList without(byte element)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedByteList withAll(ByteIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedByteList withoutAll(ByteIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableByteList select(BytePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().select(predicate);
        }
    }

    @Override
    public MutableByteList reject(BytePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().reject(predicate);
        }
    }

    @Override
    public <V> MutableList<V> collect(ByteToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().collect(function);
        }
    }

    public MutableByteList sortThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().sortThis();
        }
        return this;
    }

    public long dotProduct(ByteList list)
    {
        return this.getMutableByteList().dotProduct(list);
    }

    @Override
    public boolean equals(Object otherList)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().equals(otherList);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().hashCode();
        }
    }

    @Override
    public LazyByteIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyByteIterableAdapter(this);
        }
    }

    @Override
    public MutableByteList asUnmodifiable()
    {
        return new UnmodifiableByteList(this);
    }

    @Override
    public MutableByteList asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableByteList toImmutable()
    {
        int size = this.size();
        if (size == 0)
        {
            return ByteLists.immutable.with();
        }
        if (size == 1)
        {
            return ByteLists.immutable.with(this.getFirst());
        }
        return ByteLists.immutable.with(this.toArray());
    }

    public MutableByteList reverseThis()
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().reverseThis();
        }
        return this;
    }

    public MutableByteList toReversed()
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().toReversed();
        }
    }

    public LazyByteIterable asReversed()
    {
        return ReverseByteIterable.adapt(this);
    }

    public void forEachWithIndex(ByteIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableByteList().forEachWithIndex(procedure);
        }
    }

    public <T> T injectInto(T injectedValue, ObjectByteToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().injectInto(injectedValue, function);
        }
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectByteIntToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableByteList().injectIntoWithIndex(injectedValue, function);
        }
    }

    public MutableByteList subList(int fromIndex, int toIndex)
    {
        throw new UnsupportedOperationException("subList not yet implemented!");
    }
}
