/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.stack.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.list.primitive.FloatList;
import com.gs.collections.api.list.primitive.MutableFloatList;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.api.stack.ImmutableStack;
import com.gs.collections.api.stack.primitive.FloatStack;
import com.gs.collections.api.stack.primitive.ImmutableFloatStack;
import com.gs.collections.impl.bag.mutable.primitive.FloatHashBag;
import com.gs.collections.impl.factory.Stacks;
import com.gs.collections.impl.factory.primitive.FloatStacks;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.FloatArrayList;
import com.gs.collections.impl.set.mutable.primitive.FloatHashSet;
import com.gs.collections.impl.stack.mutable.primitive.FloatArrayStack;
import net.jcip.annotations.Immutable;

/**
 * ImmutableFloatSingletonStack is an optimization for {@link ImmutableFloatStack} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonStack.stg.
 */
@Immutable
final class ImmutableFloatSingletonStack implements ImmutableFloatStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final float element1;

    ImmutableFloatSingletonStack(float element)
    {
        this.element1 = element;
    }

    public FloatIterator floatIterator()
    {
        return FloatArrayStack.newStackWith(this.element1).floatIterator();
    }

    public void forEach(FloatProcedure procedure)
    {
        procedure.value(this.element1);
    }

    public int count(FloatPredicate predicate)
    {
        return predicate.accept(this.element1) ? 1 : 0;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return predicate.accept(this.element1);
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return !predicate.accept(this.element1);
    }

    public float peek()
    {
        return this.element1;
    }

    public FloatList peek(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return new FloatArrayList(0);
        }
        if (count == 1)
        {
            return FloatArrayList.newListWith(this.element1);
        }
        throw new IllegalArgumentException("Count must be less than or equal to size: Count = " + count + " Size = 1");
    }

    public float peekAt(int index)
    {
        this.checkNegativeCount(index);
        if (index == 0)
        {
            return this.element1;
        }
        throw new IllegalArgumentException("Index must be less than size: Index = " + index + " Size = 1");
    }

    public ImmutableFloatStack select(FloatPredicate predicate)
    {
        return predicate.accept(this.element1) ? FloatStacks.immutable.with(this.element1)
                : FloatStacks.immutable.with();
    }

    public ImmutableFloatStack reject(FloatPredicate predicate)
    {
        return predicate.accept(this.element1) ? FloatStacks.immutable.with()
                : FloatStacks.immutable.with(this.element1);
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        return predicate.accept(this.element1) ? this.element1 : ifNone;
    }

    public <V> ImmutableStack<V> collect(FloatToObjectFunction<? extends V> function)
    {
        return Stacks.immutable.with(function.valueOf(this.element1));
    }

    public double sum()
    {
        return this.element1;
    }

    public float max()
    {
        return this.element1;
    }

    public float maxIfEmpty(float defaultValue)
    {
        return this.element1;
    }

    public float min()
    {
        return this.element1;
    }

    public float minIfEmpty(float defaultValue)
    {
        return this.element1;
    }

    public double average()
    {
        return this.element1;
    }

    public double median()
    {
        return this.element1;
    }

    public float[] toSortedArray()
    {
        return new float[]{this.element1};
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newListWith(this.element1);
    }

    public float[] toArray()
    {
        return new float[]{this.element1};
    }

    public boolean contains(float value)
    {
        return Float.compare(this.element1, value) == 0;
    }

    public boolean containsAll(float... source)
    {
        for (float value : source)
        {
            if (Float.compare(this.element1, value) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        for (FloatIterator iterator = source.floatIterator(); iterator.hasNext(); )
        {
            if (Float.compare(this.element1, iterator.next()) != 0)
            {
                return false;
            }
        }
        return true;
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newListWith(this.element1);
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSetWith(this.element1);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBagWith(this.element1);
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public ImmutableFloatStack toImmutable()
    {
        return this;
    }

    public ImmutableFloatStack push(float element)
    {
        return FloatStacks.immutable.with(this.element1, element);
    }

    public ImmutableFloatStack pop()
    {
        return FloatStacks.immutable.with();
    }

    public ImmutableFloatStack pop(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return this;
        }
        if (count == 1)
        {
            return FloatStacks.immutable.with();
        }
        throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = 1");
    }

    private void checkNegativeCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element1);
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof FloatStack))
        {
            return false;
        }
        FloatStack stack = (FloatStack) otherStack;
        if (stack.size() != 1)
        {
            return false;
        }
        return Float.compare(this.element1, stack.peek()) == 0;
    }

    @Override
    public int hashCode()
    {
        return 31 + Float.floatToIntBits(this.element1);
    }

    @Override
    public String toString()
    {
        return "[" + this.element1 + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element1));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }
}
