package com.mm.live.player.catchup.proxy;

import com.stream.core.proxy.proxycommon.ForkedStream;
import com.stream.core.proxy.proxycommon.OnProxyInfoListener;
import com.stream.mrt.engine.*;
import com.stream.mrt.engine.callback.MrtDownloadCallback;
import com.stream.mrt.engine.model.MrtSlice;
import com.stream.mrt.engine.model.MrtSliceGroup;
import com.stream.mrt.engine.model.MrtUrlInfo;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;

import org.apache.commons.lang3.StringUtils;

import rx.Subscriber;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;

public class CpsMrtClient {
    private final int REQUEST_QUEUE_SIZE = 6;
    private Logger logger = LoggerFactory.getLogger(CpsMrtClient.class.getName());
    private MrtEngine mMrtEngine = new MrtEngine();

    private long mMrtSessionId = 0;
    private boolean mIsLoggedIn = false;
    private boolean mIsShutdown = false;
    private AtomicBoolean mIsInLogin = new AtomicBoolean(false);
    private MrtValueProvider mProvider;
    private CpsOnProxyInfoListener mOnProxyInfoListener;
    private String mDid;
    private String mAppVer;
    private String mPlatform;

    private final RequestWaitingQueue mWaitingQueue = new RequestWaitingQueue();

    public CpsMrtClient(String did, String appVer, String platform, MrtValueProvider provider) {
        mDid = StringUtils.trimToEmpty(did);
        mAppVer = appVer;
        mPlatform = platform;
        mProvider = provider;
    }

    public void setOnProxyInfoListener(CpsOnProxyInfoListener l) {
        mOnProxyInfoListener = l;
    }

    public boolean isShutdown() {
        return mIsShutdown;
    }

    public void shutdown() {
        mIsShutdown = true;
        // TODO: 2018/4/11 delete all downloading and waiting queue item
        // doCancel(currentDownloadSessionId);

        if (mMrtEngine != null) {
            logger.info("close mrt engine begin");
            mMrtEngine.shutdown();
            mMrtEngine = null;
            logger.info("close mrt engine end");
        }
        mIsLoggedIn = false;
        mOnProxyInfoListener = null;
    }

    public void download(long downloadSessionId, final long beginTime, final CpsBlock cpsBlock, final ForkedStream forkedStream, Subscriber<? super MrtSlice[]> subscriber) throws Exception {
        if (!mIsLoggedIn) {
            login(downloadSessionId, cpsBlock.getUrl()); // login in synchronous way
        }
        if (mIsShutdown) throw new IOException("mrt client is close");

        final MrtEngine engine = mMrtEngine;
        if (engine == null) throw new IOException("mrt engine is null");

        final long mrtSessionId = mMrtSessionId;

        MrtUrlInfo urlInfo = cpsBlock.getMrtUrlInfo();
        final QueueItem item = mWaitingQueue.getExists(urlInfo, beginTime);
        if (item != null) {
            logger.warn("MrtBlockSeq[%d] has been in the waiting queue, request session[%d], committed session[%d]", urlInfo.getBlockSeq(), downloadSessionId, item.downloadSessionId);
            throw new IOException("block has been is the queue");
        }
        // TODO: 2018/4/11 need control the request number?
        /*QueueItem overFlowItem = mWaitingQueue.cancelOverFlow();
        if (overFlowItem != null) {
            cancel(overFlowItem);
            logger.warn("MrtBlockReq[%d]@%d is out of queue capacity[%d], wait[%dms], removed.",
                    overFlowItem.urlInfo.getBlockSeq(),
                    overFlowItem.time,
                    REQUEST_QUEUE_SIZE,
                    System.currentTimeMillis() - overFlowItem.time);
        }*/
        mWaitingQueue.add(downloadSessionId, urlInfo, beginTime);
        logger.info("MrtSession[%d] waiting queue size[%d]", mrtSessionId, mWaitingQueue.size());
        MrtDownloadCallback callback = createCallback(mrtSessionId, beginTime, forkedStream, urlInfo, subscriber);
        sendRequest(downloadSessionId, urlInfo, beginTime, engine, callback);
    }

    public synchronized void cancelBeforeBlocks(CpsBlock cpsBlock) {
        LinkedBlockingQueue<QueueItem> queue = mWaitingQueue.queue;
        for (QueueItem item : queue) {
            if (item.urlInfo.getBlockSeq() < cpsBlock.getMrtUrlInfo().getBlockSeq()) {
                try {
                    cancel(item);
                } catch (IOException e) {
                    logger.error(e, "cancel error[%d]", item.urlInfo.getBlockSeq());
                }
            }
        }
    }

    public synchronized void cancelAfterBlocks(CpsBlock cpsBlock) {
        LinkedBlockingQueue<QueueItem> queue = mWaitingQueue.queue;
        for (QueueItem item : queue) {
            if (item.urlInfo.getBlockSeq() > cpsBlock.getMrtUrlInfo().getBlockSeq()) {
                try {
                    cancel(item);
                } catch (IOException e) {
                    logger.error(e, "cancel error[%d]", item.urlInfo.getBlockSeq());
                }
            }
        }
    }

    private synchronized void cancel(QueueItem item) throws IOException {
        MrtEngine engine = mMrtEngine;
        if (engine == null) {
            throw new IOException("cancel error, mrt engine is close");
        }

        if (item == null)
            return;

        item.isCancel = true;
        try {
            if (engine.cancelDownload(item.urlInfo) == null) {
                logger.debug("sessionId[%d] mrt queue cancel block[%d] in engine is failed", item.downloadSessionId, item.urlInfo.getBlockSeq());
            } else {
                logger.debug("sessionId[%d] mrt queue cancel block[%d] in engine is successful", item.downloadSessionId, item.urlInfo.getBlockSeq());
            }
            mWaitingQueue.queue.remove(item);
            addMrtInfo(item.downloadSessionId, "request_status", String.format(Locale.ENGLISH, "Cancel: %d %s", item.urlInfo.getBlockSeq(), mWaitingQueue));
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    private void login(final long downloadSessionId, final String url) throws Exception {
        while (!mIsShutdown) {
            if (!mIsInLogin.get()) {
                synchronized ("mrt_login") {
                    if (mIsLoggedIn) {
                        return;
                    }
                    if (mIsInLogin.compareAndSet(false, true)) {
                        // set in login is true success
                        break;
                    }
                }
            }
            logger.info("wait previous mrt login process begin");
            // wait for the prev login finished
            while (mIsInLogin.get() && !mIsShutdown) {
                try {
                    Thread.sleep(100);
                } catch (Exception ignored) {
                }
            }

            logger.info("wait previous mrt login process end. isShutDown [%s], isLoggedIn[%s] isInLogin[%s]", mIsShutdown, mIsLoggedIn, mIsInLogin);
            // if login has been done, just return, else to do login
            if (mIsLoggedIn) {
                return;
            }
        }
        // do actual login
        MrtEngine engine = mMrtEngine;
        if (engine == null) {
            throw new IOException("mrt engine is null");
        }
        final IOException[] loginError = new IOException[]{null};
        final OnProxyInfoListener listener = mOnProxyInfoListener;

        mMrtSessionId = engine.initModule(mDid, url, "", mAppVer, mPlatform, new MrtEngineListener() {
            private void fireClientInfo(int what, int extra1, int extra2, String info) {
                if (listener != null) {
                    listener.onProxyInfo(what, (int) downloadSessionId, extra1, extra2, info);
                }
            }

            @Override
            public void onStart(long sessionId) {
                addMrtInfo(sessionId, "login", "Login: OnStart[" + sessionId + "]");
                fireClientInfo(CpsOnProxyInfoListener.MRT_CLIENT_LOGIN_BEGIN, 0, 0, "");
            }

            @Override
            public void onAuthPassed(long sessionId, String msg) {
                try {
                    addMrtInfo(sessionId, "login", "Login: OnAuthPassed[" + msg + "]");
                    fireClientInfo(CpsOnProxyInfoListener.MRT_CLIENT_LOGIN_END, CpsOnProxyInfoListener.MRT_CLIENT_LOGIN_END_OK, 0, msg);
                    mIsLoggedIn = true;
                } finally {
                    mIsInLogin.set(false);
                }
            }

            @Override
            public void onAuthFailed(long sessionId, int errorCode, String errorMsg) {
                try {
                    addMrtInfo(sessionId, "login", "Login: OnAuthFailed[" + errorMsg + "]");
                    fireClientInfo(CpsOnProxyInfoListener.MRT_CLIENT_LOGIN_END, CpsOnProxyInfoListener.MRT_CLIENT_LOGIN_END_FAILED, errorCode, errorMsg);
                    loginError[0] = new IOException(String.format(Locale.ENGLISH, "errorCode: %d , errorMsg: %s", errorCode, errorMsg));
                } finally {
                    mIsInLogin.set(false);
                }
            }

            @Override
            public void onStop(long sessionId, boolean serverReason, int lastMsgCode, int innerErrCode) {
                String msg = String.format(Locale.ENGLISH, "OnStop[%d]:%s,%d,%d", sessionId, serverReason, lastMsgCode, innerErrCode);
                addMrtInfo(sessionId, "on_stop", msg);
                if (sessionId != mMrtSessionId) {
                    logger.warn("MrtSession[%d] onStop, but not current session[%d], ignored!", serverReason, mMrtSessionId);
                    return;
                }
                try {
                    mIsLoggedIn = false;
                    // if it is server reason, then throw the error
                    if (serverReason) {
                        fireClientInfo(CpsOnProxyInfoListener.MRT_CLIENT_STOP, lastMsgCode, innerErrCode, msg);
                        loginError[0] = new IOException(msg);
                    }
                } finally {
                    mIsInLogin.set(false);
                }
            }
        }, mProvider);
        // wait login process
        while (!mIsShutdown && mIsInLogin.get()) {
            sleep(5);
        }
        if (loginError[0] != null)
            throw loginError[0];
        if (!mIsLoggedIn)
            throw new IOException("login failed, unknown error");
    }

    private MrtDownloadCallback createCallback(final long mrtSessionId,
                                               final long beginTime,
                                               final ForkedStream forkedStream,
                                               final MrtUrlInfo urlInfo,
                                               final Subscriber<? super MrtSlice[]> subscriber) throws IOException {
        if (mIsShutdown) {
            throw new IOException("mrt client is close.");
        }
        return new MrtDownloadCallback() {

            private String printSpeed(double speedInKB) {
                if (speedInKB > 1024) {
                    return String.format(Locale.ENGLISH, "%.1fMB", speedInKB / 1024);
                } else {
                    return String.format(Locale.ENGLISH, "%.1fKB", speedInKB);
                }
            }

            @Override
            public void onComplete(MrtBlockTask mrtBlockTask) {
                try {
                    MrtTrace.t(mrtSessionId).addTrace(urlInfo.getBlockSeq(), "ME_COMPLETE");
                    String msg = String.format(Locale.ENGLISH, "OnComplete: block %d (%dms) [ %s | %s | %d parallel ]",
                            urlInfo.getBlockSeq(),
                            System.currentTimeMillis() - beginTime,
                            printSpeed(ResendStat.getLatestBlockSpeedKB()),
                            printSpeed(ResendStat.getAvgSpeedKB()),
                            ResendStat.getParallelRequestCnt());
                    addMrtInfo(mrtSessionId, "on_complete", msg);
                    OnProxyInfoListener listener = mOnProxyInfoListener;
                    if (listener != null)
                        listener.onProxyInfo(CpsOnProxyInfoListener.MRT_BLOCK_REQUEST_COMPLETED, (int) mrtSessionId, urlInfo.getBlockSeq(), mrtBlockTask.getReqCnt(), "");
                    MrtSliceGroup mrtSliceGroup = mrtBlockTask.getSliceGroup();
                    MrtSlice[] slices = mrtSliceGroup.getSliceTreeMap();
                    synchronized (CpsMrtClient.this) {
                        QueueItem item = mWaitingQueue.get(urlInfo, beginTime);
                        if (item == null) {
                            subscriber.onError(new Exception(String.format(Locale.ENGLISH,
                                    "MrtSession[%d] onComplete, block[%d] is not in the waiting queue, ignored",
                                    mrtSessionId,
                                    mrtBlockTask.getSeq())));
                            return;
                        }
                        subscriber.onNext(slices);
                        subscriber.onCompleted();
                    }
                } finally {
                    mWaitingQueue.remove(urlInfo, beginTime);
                }
            }

            @Override
            public void onRetry(MrtBlockTask mrtBlockTask) {
                addMrtInfo(mrtSessionId, "on_retry", String.format(Locale.ENGLISH,
                        "OnRetry: request block %d(%d ms, %d times)",
                        urlInfo.getBlockSeq(),
                        System.currentTimeMillis() - beginTime,
                        mrtBlockTask.getReqCnt()));
                OnProxyInfoListener listener = mOnProxyInfoListener;
                if (listener != null)
                    listener.onProxyInfo(CpsOnProxyInfoListener.MRT_BLOCK_REQUEST_RETRY,
                            (int) mrtSessionId,
                            urlInfo.getBlockSeq(),
                            mrtBlockTask.getReqCnt(),
                            "");
            }

            @Override
            public void onReceived(MrtSlice mrtSlice) {
                if (mrtSlice.seq == 0) {
                    MrtTrace.t(mrtSessionId).addTrace(urlInfo.getBlockSeq(), "ME_ON_RECEIVED");
                    addMrtInfo(mrtSessionId, "on_received", String.format(Locale.ENGLISH, "OnReceived: block %d (%dms)", urlInfo.getBlockSeq(), System.currentTimeMillis() - beginTime));
                }
                if (forkedStream != null) {
                    try {
                        byte[] data = mrtSlice.data;
                        forkedStream.write(data, 0, data.length);
                    } catch (IOException e) {
                    }
                }
            }

            @Override
            public void onCancel(MrtBlockTask mrtBlockTask) {
                subscriber.onError(new CancelException(String.format(Locale.ENGLISH, "cancel block [%d]", urlInfo.getBlockSeq())));
                mWaitingQueue.remove(urlInfo, beginTime);
            }

            @Override
            public void onError(MrtBlockTask mrtBlockTask, String s) {
                try {
                    MrtTrace.t(mrtSessionId).addTrace(urlInfo.getBlockSeq(), "ME_ERR1:" + s);
                    String msg = String.format(Locale.ENGLISH, "Error in download MrtBlock[%d]@%d:%s",
                            mrtBlockTask.getSeq(), beginTime, s);
                    addMrtInfo(mrtSessionId, "on_error", "OnError: block " + urlInfo.getBlockSeq() + ", " + s);
                    throw new IOException(msg);
                } catch (Throwable throwable) {
                    subscriber.onError(throwable);
                } finally {
                    mWaitingQueue.remove(urlInfo, beginTime);
                }
            }
        };
    }

    private void sendRequest(long mrtSessionId, MrtUrlInfo urlInfo, long beginTime, MrtEngine mrtEngine, MrtDownloadCallback callback) throws IOException {
        MrtTrace.t(mrtSessionId).addTrace(urlInfo.getBlockSeq(), "ME_BEGIN");
        addMrtInfo(mrtSessionId, "request_status", String.format(Locale.ENGLISH,
                "Enqueue: %d interval %d|%d %s",
                urlInfo.getBlockSeq(),
                ResendStat.getGroupFirstRequestInterval(),
                ResendStat.getGroupFinalRequestInterval(),
                mWaitingQueue));
        while (!mIsShutdown) {
            QueueItem item = mWaitingQueue.get(urlInfo, beginTime);
            if (item == null || item.isCancel) {
                throwCancelException(mrtSessionId, urlInfo, beginTime, item);
            }
            try {
                boolean isCommitSucceed = mrtEngine.requestDownload(urlInfo, callback);
                if (!isCommitSucceed) {
                    sleep(2);
                    continue;
                }

                addMrtInfo(mrtSessionId, "request", String.format(Locale.ENGLISH, "Request: block %d (%d ms)", urlInfo.getBlockSeq(), System.currentTimeMillis() - beginTime));
                MrtTrace.t(mrtSessionId).addTrace(urlInfo.getBlockSeq(), "ME_REQUEST");
                mWaitingQueue.commit(urlInfo, beginTime);
                break;
            } catch (Exception e) {
                logger.error(e, "commit block[%d] error", urlInfo.getBlockSeq());
                mWaitingQueue.remove(urlInfo, beginTime);
                throw new IOException(e);
            }
        }
    }

    private void throwCancelException(long mrtSessionId, MrtUrlInfo urlInfo, long beginTime, QueueItem item) throws CancelException {
        if (item != null && !item.isCancel) {
            try {
                cancel(item);
            } catch (Exception e) {
                logger.error(e, "cancel error");
            }
            logger.debug("session[%d] mrt queue remove 0 %d", mrtSessionId, urlInfo.getBlockSeq());
        }
        throw new CancelException(String.format(Locale.ENGLISH,
                "Cancel request block[%d]@%d(%dms), is in queue[%s], mrtSessionId[%d]",
                urlInfo.getBlockSeq(),
                beginTime,
                System.currentTimeMillis() - beginTime,
                item == null,
                mrtSessionId));
    }

    //++++++++++++++++++++++++++++++++++++++++++++++++++++
    // the request waiting queue
    private class QueueItem {
        MrtUrlInfo urlInfo;
        long downloadSessionId;
        long time;
        boolean isCommitted = false;
        boolean isCancel = false;

        public QueueItem(MrtUrlInfo urlInfo, long time, long downloadSessionId) {
            this.urlInfo = urlInfo;
            this.time = time;
            this.downloadSessionId = downloadSessionId;
        }

        @Override
        public String toString() {
            return Integer.toString(urlInfo.getBlockSeq());
        }
    }

    private class RequestWaitingQueue {
        private LinkedBlockingQueue<QueueItem> queue = new LinkedBlockingQueue<>();

        public int size() {
            return queue.size();
        }

        public synchronized void add(long downloadSessionId, MrtUrlInfo urlInfo, long time) {
            logger.debug("session[%d] mrt queue add %d", downloadSessionId, urlInfo.getBlockSeq());
            queue.offer(new QueueItem(urlInfo, time, downloadSessionId));
        }

        public synchronized QueueItem remove(MrtUrlInfo urlInfo, long time) {
            QueueItem removedItem = null;
            for (QueueItem item : queue) {
                // if (item.downloadSessionId <= downloadSessionId && item.time <= time && item.urlInfo.getBlockSeq() == urlInfo.getBlockSeq()) {
                if (item.time <= time && item.urlInfo.getBlockSeq() == urlInfo.getBlockSeq()) {
                    removedItem = item;
                    break;
                }
            }
            if (removedItem != null) {
                logger.debug("session[%d] mrt queue remove %d", mMrtSessionId, urlInfo.getBlockSeq());
                queue.remove(removedItem);
                if (removedItem.isCancel) {
                    addMrtInfo(mMrtSessionId, "request_status", String.format(Locale.ENGLISH, "Cancel: %d %s", urlInfo.getBlockSeq(), mWaitingQueue));
                } else {
                    addMrtInfo(mMrtSessionId, "request_status", String.format(Locale.ENGLISH, "Request: %d %s", urlInfo.getBlockSeq(), mWaitingQueue));
                }
            }

            return removedItem;
        }

        // TODO: 2018/4/11 logic right?
        public synchronized QueueItem getExists(MrtUrlInfo urlInfo, long time) {
            QueueItem oldItem = null;
            for (QueueItem item : queue) {
                // if (item.downloadSessionId <= downloadSessionId && item.time <= time && item.urlInfo.getBlockSeq() == urlInfo.getBlockSeq()) {
                if (item.time <= time && item.urlInfo.getBlockSeq() == urlInfo.getBlockSeq()) {
                    oldItem = item;
                    break;
                }
            }

            return oldItem;
        }

        public synchronized QueueItem cancelOverFlow() throws IOException {
            QueueItem item = null;
            if (queue.size() >= REQUEST_QUEUE_SIZE) {
                item = queue.peek();
                cancel(item);
            }

            return item;
        }

        public void commit(MrtUrlInfo urlInfo, long time) {
            for (QueueItem item : queue) {
                if (item.time == time && item.urlInfo.getBlockSeq() == urlInfo.getBlockSeq()) {
                    item.isCommitted = true;
                    return;
                }
            }
        }

        public QueueItem get(MrtUrlInfo urlInfo, long time) {
            for (QueueItem item : queue) {
                if (item.time == time && item.urlInfo.getBlockSeq() == urlInfo.getBlockSeq()) {
                    return item;
                }
            }
            return null;
        }

        @Override
        public String toString() {
            StringBuilder waiting = new StringBuilder();
            StringBuilder committed = new StringBuilder();
            StringBuilder canceling = new StringBuilder();
            List<QueueItem> items = new ArrayList<>(queue);
            for (int i = items.size() - 1; i >= 0; i--) {
                QueueItem item = items.get(i);
                if (item.isCancel)
                    canceling.append(item.urlInfo.getBlockSeq()).append(",");
                if (item.isCommitted)
                    committed.append(item.urlInfo.getBlockSeq()).append(",");

                if (!item.isCancel && !item.isCommitted)
                    waiting.append(item.urlInfo.getBlockSeq()).append(",");
            }

            StringBuilder builder = new StringBuilder();
            if (committed.length() > 0) {
                committed.setLength(committed.length() - 1);
                builder.append("q[").append(waiting.toString()).append("] ");
            }
            if (waiting.length() > 0) {
                waiting.setLength(waiting.length() - 1);
                builder.append("d[").append(committed.toString()).append("] ");
            }
            if (canceling.length() > 0) {
                canceling.setLength(canceling.length() - 1);
                builder.append("c[").append(canceling.toString()).append("] ");
            }

            if (builder.length() > 0) {
                builder.setLength(builder.length() - 1);
                return builder.toString();
            }

            return "empty";
        }
    }
    //--------------------------------------------------------------

    //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    // for mrt log
    private Map<String, String> mrtInfos = Collections.synchronizedMap(new LinkedHashMap<String, String>());

    private void addMrtInfo(long sessionId, String event, String msg) {
        if (mOnProxyInfoListener != null) {
            mrtInfos.put(event, String.format(Locale.ENGLISH, "session[%d] %s", sessionId, msg));
            fireProxyInfo();
        }
    }

    private void fireProxyInfo() {
        OnProxyInfoListener listener = mOnProxyInfoListener;
        if (listener == null)
            return;

        StringBuilder builder = new StringBuilder();
        for (String m : new ArrayList<>(mrtInfos.values())) {
            builder.append(m).append("\n");
        }
        if (builder.length() > 0)
            builder.setLength(builder.length() - 1);
        listener.onProxyInfo(CpsOnProxyInfoListener.MRT_CLIENT_INFO, 0, 0, 0, builder.toString());
    }
    //---------------------------------------------------------------

    //++++++++++++++++++++++++
    // common method
    private void sleep(long millis) {
        try {
            Thread.sleep(millis);
        } catch (Exception ignored) {
        }
    }
    //------------------------
}
