package com.mm.live.player.catchup;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.media.MediaPlayer;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import android.util.ArrayMap;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.MediaController;
import android.widget.TextView;
import android.widget.Toast;

import com.mm.live.player.R;
import com.mm.live.player.catchup.proxy.CatchUpProxyConfig;
import com.mm.live.player.catchup.proxy.CatchUpStreamProxy;
import com.mm.live.player.catchup.proxy.CpsHandler;
import com.mm.live.player.catchup.proxy.CpsOnProxyInfoListener;
import com.mm.live.player.catchup.proxy.MrtTrace;
import com.mm.live.player.catchup.proxy.PrpdStreamProxy;
import com.mm.live.player.catchup.util.PureStringUtils;
import com.mm.live.player.ijkmedia.DefSettingsProvider;
import com.mm.live.player.ijkmedia.IMediaController;
import com.mm.live.player.ijkmedia.ISettingsProvider;
import com.mm.live.player.ijkmedia.IjkVideoView;
import com.stream.core.proxy.proxycommon.StreamProxy;
import com.stream.mrt.engine.MrtStatusListener;
import com.stream.mrt.engine.ResendStat;
import com.stream.prt.utils.JniApiImpl;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import rx.Observable;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Action1;
import rx.functions.Func1;
import rx.schedulers.Schedulers;
import timber.log.Timber;
import tv.danmaku.ijk.media.player.IMediaPlayer;

public class CatchUpVideoView extends FrameLayout implements MediaController.MediaPlayerControl {
    public static final int MRT_ERROR_DISCONNECTED = 90001;
    protected Activity mContext;
    private IjkVideoView mVideoView;
    private View mLoadingIcon;
    private ViewGroup mLoadingHint;
    private TextView mLoadingMsg;
    private TextView mLoadingSpeed;
    private LinearLayout mDebugInfo;
    private TextView mPrpdDebugInfo;
    private TextView mStatisticsDebugInfo;
    private StreamProxy mStreamProxy;
    private IMediaController mMediaController;
    private OnPlayEventListener mPlayEventListener;
    protected int MRT_RETRY_LIMIT = 100;

    private boolean mIsDebugViewVisible;
    private boolean mIsLoadingViewVisible;
    private boolean mInLowLevelBuffering = false;
    private Runnable mOnLowLevelBufferingEndAction = null;

    private CatchUpPlayInfo mPlayInfo;
    private boolean showPlayerSwitchDialog = false;

    //for drm
    private int mDrmType = 0;
    private boolean mMultiSession = false;
    private String mLicenceServerUrl = "";
    private Map<String, String> mDrmReqHeader = new ArrayMap<>();
    private String mDrmReqMethod = IMediaPlayer.DRM_REQ_GET;

    public CatchUpVideoView(@NonNull Context context) {
        super(context);
        initView(context);
    }

    public CatchUpVideoView(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        initView(context);
    }

    private void initView(Context context) {
        mContext = (Activity) context;
        View.inflate(context, getLayoutId(), this);
        setFocusable(true);
        setFocusableInTouchMode(true);
        mVideoView = findViewById(R.id.video_view);
        mLoadingIcon = findViewById(R.id.loading_icon);
        mLoadingHint = findViewById(R.id.loading_hint);
        mLoadingMsg = findViewById(R.id.loading_msg);
        mLoadingSpeed = findViewById(R.id.loading_speed);
        mDebugInfo = findViewById(R.id.debug_info);
        mPrpdDebugInfo = (TextView) this.findViewById(R.id.debug_info_prpd_info);
        mStatisticsDebugInfo = (TextView) this.findViewById(R.id.debug_info_statistics_info);

        mVideoView.setOnPreparedListener(mOnPreparedListener);
        mVideoView.setOnInfoListener(mOnInfoListener);
        mVideoView.setOnCompletionListener(mOnCompletionListener);
        mVideoView.setOnErrorListener(mOnErrorListener);
        mVideoView.setOnSeekCompleteListener(mOnSeekCompleteListener);
        // Settings.setSettingsProvider(getPlayerSettings());
        mVideoView.setSettings(getPlayerSettings());

        ResendStat.setListener(new MrtStatusListener() {
            @Override
            public void onStatus(String s) {
                if (canShowDebugInfo()) {
                    Timber.i(s);
                }
            }
        });
        startPlayChecker();
    }

    public void getAndPlay() {
        if (getCatchUpPlayInfo() == null) {
            showToast(mContext, "no catchup play info provider!");
            if (mPlayEventListener != null) {
                mPlayEventListener.onPreaparError(new RuntimeException("No play info provider!"));
            } else {
                showToast(mContext, "Prepare error,no play info provider!");
            }
        } else {
            showLoadingView(getConnectingMsg());
            getCatchUpPlayInfo()
                    .subscribeOn(Schedulers.io())
                    .flatMap(new Func1<CatchUpPlayInfo, Observable<String>>() {
                        @Override
                        public Observable<String> call(CatchUpPlayInfo playInfo) {
                            return getProxiedUrl(playInfo);
                        }
                    })
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Action1<String>() {
                        @Override
                        public void call(String playUrl) {
                            showLoadingView(getLoadingMsg());
                            mVideoView.setDrmInfo(mDrmType, mMultiSession, mLicenceServerUrl, mDrmReqHeader, mDrmReqMethod);
                            mVideoView.setVideoPath(playUrl);
                            mVideoView.start();
                        }
                    }, new Action1<Throwable>() {
                        @Override
                        public void call(Throwable throwable) {
                            Timber.e(throwable, "prepare error");
                            if (mPlayEventListener != null) {
                                mPlayEventListener.onPreaparError(throwable);
                            } else {
                                showToast(mContext, "prepare error," + throwable.getMessage());
                            }
                        }
                    });
        }
    }

    public void setCatchUpPlayInfo(CatchUpPlayInfo catchUpPlayInfo) {
        getProxiedUrl(catchUpPlayInfo)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Action1<String>() {
                    @Override
                    public void call(String proxiedUrl) {
                        showLoadingView(getLoadingMsg());
                        mVideoView.setDrmInfo(mDrmType, mMultiSession, mLicenceServerUrl, mDrmReqHeader, mDrmReqMethod);
                        mVideoView.setVideoPath(proxiedUrl);
                        mVideoView.start();
                    }
                }, new Action1<Throwable>() {
                    @Override
                    public void call(Throwable throwable) {
                        Timber.e(throwable, "getProxiedUrl error");
                        showToast(mContext, "generate play url failed!");
                        // TODO: 2018/4/9 send error message to host activity
                    }
                });
    }

    private Observable<String> getProxiedUrl(CatchUpPlayInfo playInfo) {
        return Observable.just(playInfo)
                .map(new Func1<CatchUpPlayInfo, String>() {
                    @Override
                    public String call(CatchUpPlayInfo playInfo) {
                        StreamProxy streamProxy = mStreamProxy;
                        if (streamProxy != null) {
                            try {
                                Timber.w("current stream proxy is not null! shutdown it.");
                                streamProxy.shutdown();
                                mStreamProxy = null;
                            } catch (Throwable ignored) {
                            }
                        }

                        mPlayInfo = playInfo;
                        if ((mPlayInfo != null) && ("mrt".equals(mPlayInfo.getProtocol()))) {
                            CatchUpProxyConfig proxyConfig = getCatchUpProxyConfig();
                            mStreamProxy = new CatchUpStreamProxy(proxyConfig, proxyConfig.getSettings(), playInfo, getPlayPositionProvider());
                            mStreamProxy.setOnInfoListener(mOnProxyInfoListener);
                        } else {
                            mStreamProxy = PrpdStreamProxy.getInstance();
                            ((PrpdStreamProxy) mStreamProxy).setPlayInfo(playInfo);
                        }
                        mStreamProxy.start();
                        try {
                            return mStreamProxy.getProxiedLink("", "catchupproxy", "");
                        } catch (Exception e) {
                            Timber.e(e, "getProxiedUrl failed!");
                            throw new RuntimeException("can not get proxy link!");
                        }
                    }
                });
    }

    private IMediaPlayer.OnPreparedListener mOnPreparedListener = new IMediaPlayer.OnPreparedListener() {
        @Override
        public void onPrepared(IMediaPlayer mediaPlayer) {
            Timber.i("onPrepared");
            if (mPlayEventListener != null) {
                mPlayEventListener.onPrepared(mediaPlayer);
            }
        }
    };

    private IMediaPlayer.OnInfoListener mOnInfoListener = new IMediaPlayer.OnInfoListener() {
        @Override
        public boolean onInfo(IMediaPlayer mediaPlayer, int what, int extra, String info) {
            Timber.i("onInfo: what[%d] extra[%d]", what, extra);
            boolean result = true;
            switch (what) {
                case MediaPlayer.MEDIA_INFO_BUFFERING_START:
                    Timber.d("OnInfoListener MEDIA_INFO_BUFFERING_START");
                    mHandler.sendEmptyMessage(what);
                    break;
                case MediaPlayer.MEDIA_INFO_BUFFERING_END:
                    Timber.d("OnInfoListener MEDIA_INFO_BUFFERING_END");
                    mHandler.sendEmptyMessage(what);
                    break;
                case MediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START:
                    Timber.d("OnInfoListener MEDIA_INFO_VIDEO_RENDERING_START");
                    break;
                case MediaPlayer.MEDIA_INFO_VIDEO_TRACK_LAGGING:
                    Timber.d("OnInfoListener MEDIA_INFO_VIDEO_TRACK_LAGGING value[%d]", extra);
                    break;
                case 703: // MEDIA_INFO_NETWORK_BANDWIDTH
                    Timber.d("OnInfoListener MEDIA_INFO_NETWORK_BANDWIDTH value[%d]", extra);
                    break;
            }
            if (mPlayEventListener != null) {
                result = mPlayEventListener.onInfo(mediaPlayer, what, extra);
            }
            return result;
        }
    };

    private IMediaPlayer.OnCompletionListener mOnCompletionListener = new IMediaPlayer.OnCompletionListener() {
        @Override
        public void onCompletion(IMediaPlayer mediaPlayer) {
            Timber.i("onCompletion");
            if (mPlayEventListener != null) {
                mPlayEventListener.onCompletion(mediaPlayer);
            }
        }
    };

    private IMediaPlayer.OnErrorListener mOnErrorListener = new IMediaPlayer.OnErrorListener() {
        @Override
        public boolean onError(IMediaPlayer mediaPlayer, int what, int extra) {
            Timber.i("onError: what[%d] extra[%d]", what, extra);
            boolean result = true;
            if (mPlayEventListener != null) {
                result = mPlayEventListener.onError(mediaPlayer, what, extra, "");
            }
            return result;
        }
    };

    private IMediaPlayer.OnSeekCompleteListener mOnSeekCompleteListener = new IMediaPlayer.OnSeekCompleteListener() {
        @Override
        public void onSeekComplete(IMediaPlayer mediaPlayer) {
            Timber.i("onSeekComplete");
            if (mPlayEventListener != null) {
                mPlayEventListener.onSeekComplete(mediaPlayer);
            }
        }
    };

    //++++++++++++++++++++++++++++++++++++++++++++
    // MediaController.MediaPlayerControl begin
    //++++++++++++++++++++++++++++++++++++++++++++
    @Override
    public void start() {
        mVideoView.start();
    }

    @Override
    public void pause() {
        mVideoView.pause();
    }

    @Override
    public int getDuration() {
        return mVideoView.getDuration();
    }

    @Override
    public int getCurrentPosition() {
        return mVideoView.getCurrentPosition();
    }

    @Override
    public void seekTo(int pos) {
        mVideoView.seekTo(pos);
    }

    @Override
    public boolean isPlaying() {
        return mVideoView.isPlaying();
    }

    @Override
    public int getBufferPercentage() {
        return mVideoView.getBufferPercentage();
    }

    @Override
    public boolean canPause() {
        return mVideoView.canPause();
    }

    @Override
    public boolean canSeekBackward() {
        return mVideoView.canSeekBackward();
    }

    @Override
    public boolean canSeekForward() {
        return mVideoView.canSeekForward();
    }

    @Override
    public int getAudioSessionId() {
        return mVideoView.getAudioSessionId();
    }
    //------------------------------------------
    // MediaController.MediaPlayerControl end
    //------------------------------------------


    //+++++++++++++++++++++++++++++++++++
    // Delegate IjkVideoView begin
    //+++++++++++++++++++++++++++++++++++

    public int getPlayerType() {
        return this.mVideoView.getPlayerType();
    }

    public void setMediaController(IMediaController controller) {
        mMediaController = controller;
        mMediaController.setOnShownListener(new IMediaController.OnShownListener() {
            @Override
            public void onShown() {
                if (mVideoView.getPlayerState() == IjkVideoView.STATE_PAUSED) {
                    mPlayChecker.onPause();
                    hideLoadingView();
                }
            }
        });
        mMediaController.setOnSeekListener(new IMediaController.OnSeekListener() {
            @Override
            public void onSeek(long position) {
                if (mVideoView.getPlayerState() == IjkVideoView.STATE_PAUSED) {
                    mVideoView.start(); // start play when seek complete
                }

                if (mPlayEventListener != null) {
                    mPlayEventListener.onSeek(position);
                }
            }
        });
        mMediaController.setOnPauseListener(new IMediaController.OnPauseListener() {
            @Override
            public void onParse() {
                hideLoadingView();
            }
        });
        mVideoView.setMediaController(controller);
    }

    public void setSeekStyle(int seekStyle) {
        mVideoView.setSeekStyle(seekStyle);
    }

    //-----------------------------------
    // Delegate IjkVideoView end
    //-----------------------------------

    //debug info view

    protected boolean canShowDebugInfo() {
        return true;
    }

    private void toggleDebugInfo() {
        if (mDebugInfo.getVisibility() == VISIBLE) {
            mDebugInfo.setVisibility(GONE);
            mIsDebugViewVisible = false;
        } else {
            mDebugInfo.setVisibility(VISIBLE);
            mIsDebugViewVisible = true;
        }
    }

    public void toggleAspectRatio() {
        mVideoView.toggleAspectRatio();
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        boolean isKeyCodeSupported = keyCode != KeyEvent.KEYCODE_BACK &&
                keyCode != KeyEvent.KEYCODE_VOLUME_UP &&
                keyCode != KeyEvent.KEYCODE_VOLUME_DOWN &&
                keyCode != KeyEvent.KEYCODE_VOLUME_MUTE &&
                keyCode != KeyEvent.KEYCODE_MENU &&
                keyCode != KeyEvent.KEYCODE_CALL &&
                keyCode != KeyEvent.KEYCODE_ENDCALL;
        if (keyCode == KeyEvent.KEYCODE_9 && event.getRepeatCount() == 5) {
            if (canShowDebugInfo()) {
                toggleDebugInfo();
                return true;
            }
        } else if (keyCode == KeyEvent.KEYCODE_8 && event.getRepeatCount() == 5) {
            if (showPlayerSwitchDialog) {
                return true;
            }

            if (isPlaying()) {
                String[] keys = new String[]{"AndroidMediaPlayer", "IjkMediaPlayer", "IjkExoPlayer"};
                final List<Integer> values = Arrays.asList(IjkVideoView.PV_PLAYER__AndroidMediaPlayer, IjkVideoView.PV_PLAYER__IjkMediaPlayer, IjkVideoView.PV_PLAYER__IjkExoMediaPlayer);

                showPlayerSwitchDialog = true;
                new AlertDialog.Builder(getContext())
                        .setTitle("Change Player")
                        .setSingleChoiceItems(keys, values.indexOf(mVideoView.getPlayerType()), new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                                //mVideoView.setIjkVideoLayout(values.get(which));
                                dialog.dismiss();
                                int position = getCurrentPosition();
                                clean();
                                mVideoView.setPlayerType(values.get(which));
                                retryWhenError(position, true);
                                showPlayerSwitchDialog = false;
                            }
                        }).show();
            }
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_DOWN && event.getRepeatCount() == 0) {
            int playerState = mVideoView.getPlayerState();
            // playerState != IjkVideoView.STATE_PAUSED && playerState != IjkVideoView.STATE_PREPARING &&
            if (playerState == IjkVideoView.STATE_PLAYING && mMediaController != null) {
                mMediaController.show();
                return true;
            }
        } else if (keyCode == KeyEvent.KEYCODE_MEDIA_NEXT && event.getRepeatCount() == 0) {
            Toast.makeText(getContext(), "Coming soon...", Toast.LENGTH_SHORT).show();
        } else if (keyCode == KeyEvent.KEYCODE_MEDIA_PREVIOUS && event.getRepeatCount() == 0) {
            Toast.makeText(getContext(), "Coming soon...", Toast.LENGTH_SHORT).show();
        } else if (mVideoView.isInPlaybackState() && isKeyCodeSupported && mMediaController != null) {
            if (keyCode == KeyEvent.KEYCODE_HEADSETHOOK ||
                    keyCode == KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE ||
                    keyCode == KeyEvent.KEYCODE_DPAD_CENTER ||
                    keyCode == KeyEvent.KEYCODE_ENTER) {
                if (event.getRepeatCount() == 0) {
                    return handlePlayPauseInput();
                } else {
                    Timber.d("onKeyDown onKeyDown code[%d] action[%d] ignore repeat[%d] key", keyCode, event.getAction(), event.getRepeatCount());
                    return false;
                }
            } else if (keyCode == KeyEvent.KEYCODE_MEDIA_PLAY) {
                if (!isPlaying()) {
                    start();
                    mMediaController.hide();
                }
                return true;
            } else if (keyCode == KeyEvent.KEYCODE_MEDIA_STOP || keyCode == KeyEvent.KEYCODE_MEDIA_PAUSE) {
                if (isPlaying()) {
                    pause();
                    mMediaController.show();
                }
                return true;
            } else if (keyCode == KeyEvent.KEYCODE_DPAD_RIGHT || keyCode == KeyEvent.KEYCODE_MEDIA_FAST_FORWARD) {
                return handleKeyEventForwardOrRewindBegin(event, true);
            } else if (keyCode == KeyEvent.KEYCODE_DPAD_LEFT || keyCode == KeyEvent.KEYCODE_MEDIA_REWIND) {
                return handleKeyEventForwardOrRewindBegin(event, false);
            }
        } /*else if (keyCode == KeyEvent.KEYCODE_MENU) {
            String[] keys = new String[]{"Fit Parent-Auto", "Fill Parent", "Wrap Content", "Fit Parent-16:9", "Fit Parent-4:3"};
            final List<Integer> values = Arrays.asList(0, 1, 2, 4, 5);
            new AlertDialog.Builder(getContext())
                    .setTitle("Change Video Aspect Ratio")
                    .setSingleChoiceItems(keys, values.indexOf(mVideoView.getAspectRatio()), new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            mVideoView.setAspectRatio(values.get(which));
                            dialog.dismiss();
                        }
                    }).show();
        }*/
        return super.onKeyDown(keyCode, event);
    }

    @Override
    public boolean onKeyUp(int keyCode, KeyEvent event) {
        boolean isKeyCodeSupported = keyCode != KeyEvent.KEYCODE_VOLUME_UP && keyCode != KeyEvent.KEYCODE_VOLUME_DOWN && keyCode != KeyEvent.KEYCODE_MENU && keyCode != KeyEvent.KEYCODE_CALL && keyCode != KeyEvent.KEYCODE_ENDCALL;
        if (isKeyCodeSupported && mMediaController != null) {
            if (event.getAction() == KeyEvent.ACTION_UP) {
                if (mMediaController.isTracking()) {
                    return handleKeyEventForwardOrRewindEnd();
                }
            }
        }
        return false;
    }

    private long getBufferedLeading() {
        return mVideoView.getBufferPercentage() * mVideoView.getDuration() / 100 - mVideoView.getCurrentPosition();
    }

    private boolean handlePlayPauseInput() {
        if (mVideoView.isInPlaybackState()) {
            if (isPlaying()) {
                pause();
                //if (!mMediaController.isShowing()) {
                mMediaController.showForever();
                //}
                hideLoadingView();
            } else {
                start();
                if (mMediaController.isShowing()) {
                    mMediaController.show();
                }
                /**
                 * 问题：
                 * 回放界面缓冲过程中按暂停，再按开始，无loading提示，只有静止画面”这个问题没解决，网速比较快的情况下，也会出现一定时间画面静止
                 */
                if (getBufferedLeading() <= 0) {
                    // 需要缓冲
                    showLoadingView(getLoadingMsg());
                    //mPlayChecker.setAffectLoadingView(false);
                    /*mHandler.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            long leading = getBufferedLeading();
                            Timber.e("buffer leading: [%d]", leading);
                            if (leading > 500) {
                                hideLoadingView();
                                mPlayChecker.setAffectLoadingView(true);
                            } else {
                                mHandler.postDelayed(this, 500);
                            }
                        }
                    }, 500);*/
                }
            }
            return true;
        }
        return false;
    }

    private long mForwardRewindPosition;//fast forward/rewind position

    private boolean handleKeyEventForwardOrRewindBegin(KeyEvent event, boolean forward) {
        if (!mMediaController.isShowing()) mMediaController.show();
        if (!mVideoView.isInPlaybackState()) {
            return false;
        }
        if (!mMediaController.isTracking()) {
            mMediaController.startTracking();
            mForwardRewindPosition = getCurrentPosition();
        }

        long duration = getDuration();
        long step = duration / 240L;
        if (step < 1000L) step = 1000L;
        if (event.getRepeatCount() > 10) {
            step = step * 2;
        }
        if (duration > 0) {
            mForwardRewindPosition = forward ? Math.min(duration, mForwardRewindPosition + step) : Math.max(0, mForwardRewindPosition - step);
            long posPercent = 1000L * mForwardRewindPosition / duration;
            mMediaController.setTrackingProgress((int) posPercent);
            Timber.i("handleKeyEventFastRewindBegin at [%d]", posPercent);
            // onLowLevelPlayerBufferingStart(); should not here because not pause when begin track
        } else {
            Timber.w("Invalid duration [%d]", duration);
        }

        return true; // ??
    }

    private boolean handleKeyEventForwardOrRewindEnd() {
        mHandler.postDelayed(new Runnable() {
            @Override
            public void run() {
                if (mMediaController.isTracking()) {
                    mMediaController.stopTracking();
                    mHandler.sendEmptyMessage(MediaPlayer.MEDIA_INFO_BUFFERING_START);
                }
                Timber.i("handleKeyEventForwardOrRewindEnd");
            }
        }, 500);
        return true;
    }


    //++++++++++++++++++++++++++++++++++++++++
    // clean
    public void clean() {
        Timber.i("clean catch up video view");
        IjkVideoView videoView = mVideoView;
        if (videoView != null) {
            try {
                Timber.i("clean: stop playback");
                videoView.stopPlayback();
                videoView.cleanUri();
            } catch (Exception e) {
                Timber.e(e, "Error in close video view at clean step");
            }
        }
        final StreamProxy streamProxy = mStreamProxy;
        if (streamProxy != null) {
            Observable.just(1).observeOn(Schedulers.io()).subscribe(new Action1<Integer>() {
                @Override
                public void call(Integer integer) {
                    try {
                        Timber.i("clean: close proxy");
                        streamProxy.shutdown();
                        mStreamProxy = null;
                    } catch (Exception e) {
                        Timber.w(e, "Error in close mega streamProxy. Ignore it");
                        mStreamProxy = null;
                    }
                }
            });
        }
        mPlayChecker.running = false;
        stopPlayChecker();
        ResendStat.setListener(null);
        MrtTrace.shutdown();
    }

    public void stopProxy() {
        Timber.i("stopProxy");
        IjkVideoView videoView = mVideoView;
        if (videoView != null) {
            try {
                Timber.i("clean: stop playback");
                videoView.stopPlayback();
                videoView.cleanUri();
            } catch (Exception e) {
                Timber.e(e, "Error in close video view at clean step");
            }
        }
        final StreamProxy streamProxy = mStreamProxy;
        if (streamProxy != null) {
            Observable.just(1).observeOn(Schedulers.io()).subscribe(new Action1<Integer>() {
                @Override
                public void call(Integer integer) {
                    try {
                        Timber.i("clean: close proxy");
                        streamProxy.shutdown();
                        mStreamProxy = null;
                    } catch (Exception e) {
                        Timber.w(e, "Error in close mega streamProxy. Ignore it");
                        mStreamProxy = null;
                    }
                }
            });
        }
    }

    //----------------------------------------

    //++++++++++++++++++++++++++++++++++++++++
    @SuppressLint("CheckResult")
    public void retryWhenError(final long position, boolean restart) {
        if (restart) {
            mPlayChecker.cleanErrorState();
            if (!mPlayChecker.running) {
                mPlayChecker.running = true;
                startPlayChecker();
            }
            showLoadingView(getConnectingMsg());
            getCatchUpPlayInfo()
                    .subscribeOn(Schedulers.io())
                    .flatMap(new Func1<CatchUpPlayInfo, Observable<String>>() {
                        @Override
                        public Observable<String> call(CatchUpPlayInfo playInfo) {
                            return getProxiedUrl(playInfo);
                        }
                    })
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Action1<String>() {
                        @Override
                        public void call(String playUrl) {
                            showLoadingView(getLoadingMsg());
                            mVideoView.setDrmInfo(mDrmType, mMultiSession, mLicenceServerUrl, mDrmReqHeader, mDrmReqMethod);
                            mVideoView.setVideoPath(playUrl);
                            mVideoView.seekTo((int) position);
                            mVideoView.start();
                        }
                    }, new Action1<Throwable>() {
                        @Override
                        public void call(Throwable throwable) {
                            Timber.e(throwable, "retry error");
                            if (mPlayEventListener != null) {
                                mPlayEventListener.onPreaparError(throwable);
                            } else {
                                showToast(mContext, "retry error," + throwable.getMessage());
                            }
                        }
                    });

        } else {
            mPlayChecker.cleanErrorState();
            mVideoView.seekTo((int) position);
            mVideoView.start();
        }
    }

    //----------------------------------------

    //++++++++++++++++++++++++++++++++++++++++
    // state info

    private PlayChecker mPlayChecker = new PlayChecker();
    private CpsOnProxyInfoListener mOnProxyInfoListener = new CpsOnProxyInfoListener(new CpsOnProxyInfoListener.MrtClientStatusHolder() {
        @Override
        public void setStatus(int status) {
            mPlayChecker.mrtClientStatus = status;
        }

        @Override
        public void setBlockRetryInfo(int blockSeq, int retryCnt) {
            mPlayChecker.mrtRequestBlock = blockSeq;
            mPlayChecker.mrtRequestRetryCnt = retryCnt;
        }
    });

    private HandlerThread mCheckerThread;
    private Handler mCheckerHandler;

    private void startPlayChecker() {
        stopPlayChecker(); // stop firstly
        mCheckerThread = new HandlerThread("catchup-play-checker");
        mCheckerThread.start();
        mCheckerHandler = new Handler(mCheckerThread.getLooper());
        mCheckerHandler.post(mPlayChecker);
    }

    private void stopPlayChecker() {
        if (mCheckerHandler != null) {
            Timber.w("check handler is not null remove all message first!");
            mCheckerHandler.removeCallbacksAndMessages(null);
        }
        if (mCheckerThread != null) {
            Timber.w("check thread is not null quit it!");
            try {
                mCheckerThread.quit();
            } catch (Exception e) {
            }
        }
    }

    private void updateStatView() {
        if (!mIsDebugViewVisible) return;
        String playerState = "Unknown";
        switch (mVideoView.getPlayerState()) {
            case IjkVideoView.STATE_ERROR:
                playerState = "ERROR";
                break;
            case IjkVideoView.STATE_IDLE:
                playerState = "IDLE";
                break;
            case IjkVideoView.STATE_PREPARING:
                playerState = "PREPARING";
                break;
            case IjkVideoView.STATE_PREPARED:
                playerState = "PREPARED";
                break;
            case IjkVideoView.STATE_PLAYING:
                playerState = "PLAYING";
                break;
            case IjkVideoView.STATE_PAUSED:
                playerState = "PAUSED";
                break;
            case IjkVideoView.STATE_PLAYBACK_COMPLETED:
                playerState = "PLAYBACK_COMPLETED";
                break;
        }
        String playerType = "Unknown";
        switch (mVideoView.getPlayerType()) {
            case IjkVideoView.PV_PLAYER__AndroidMediaPlayer:
                playerType = "AndroidMediaPlayer";
                break;
            case IjkVideoView.PV_PLAYER__IjkMediaPlayer:
                playerType = "IjkMediaPlayer";
                break;
            case IjkVideoView.PV_PLAYER__IjkExoMediaPlayer:
                playerType = "IjkExoPlayer";
                break;
        }
        String statFormat = "========== Statistics Info ==========\n" +
                "Protocol: %s\n" +
                "Tracker: %s\n" +
                "DefaultPrt: %s\n" +
                "Player: %s\n" +
                "Player State : %s\n" +
                "Speed : %s  \n" +
                "Duration Time : %s\n" +
                "Play Time : %s in(%d.ts)\n" +
                // "Buffer State : %s\n" +
                "Buffer Total Progress : %s\n" +
                "Buffer Total Time : %s\n" +
                "========== Cps Handler Info ==========\n%s\n" +
                "========== Mrt Client Info ==========\n%s";
        long position = mVideoView.getCurrentPosition();
        int block = 0;
        String formattedText = String.format(Locale.ENGLISH, statFormat,
                (mPlayInfo == null) ? "" : mPlayInfo.getProtocol(),
                (mPlayInfo == null) ? "" : mPlayInfo.getTracker(),
                (mPlayInfo == null) ? "" : mPlayInfo.getDefaultPrt(),
                playerType,
                playerState,
                PureStringUtils.getDisplaySpeed(mPlayChecker.speed),
                PureStringUtils.generateTime(mVideoView.getDuration()),
                PureStringUtils.generateTime(position), ((mStreamProxy != null) && (mStreamProxy instanceof CatchUpStreamProxy)) ? ((CatchUpStreamProxy) mStreamProxy).getBlockContainsPosition(position) : 0,
                mVideoView.getBufferPercentage() + "%",
                PureStringUtils.generateTime(mVideoView.getBufferPercentage() * mVideoView.getDuration() / 100),
                mOnProxyInfoListener.getStatInfo("CpsHandlerInfo"),
                mOnProxyInfoListener.getStatInfo("MrtClientInfo"));
        mStatisticsDebugInfo.setText(formattedText);

        if ((mPlayInfo != null) && ("mrt".equals(mPlayInfo.getProtocol()))) {
            this.mPrpdDebugInfo.setVisibility(GONE);
        } else {
            this.mPrpdDebugInfo.setVisibility(VISIBLE);
            String prpdStatFormat = "========== Prpd Info ==========\n %s";
            String prpdFormattedText = String.format(Locale.ENGLISH, prpdStatFormat, ((mStreamProxy == null) ? "" : ((PrpdStreamProxy) mStreamProxy).getDebugInfo()));
            this.mPrpdDebugInfo.setText(prpdFormattedText);
        }
    }

    private void runOnUiThread(Runnable action) {
        mContext.runOnUiThread(action);
    }

    //play checker
    public class PlayChecker implements Runnable {
        public static final int PLAY_CHECK_STATE = 1000;
        private long lastPosition = 0;
        private boolean isPositionMoving = false;
        private long pausePosition = -1;
        private boolean running = true;
        private boolean affectLoadingView = true;
        private boolean isTimeout = false;
        private String timeoutInfo = "";
        private long timeoutReason = 0;
        private boolean isNetworkConnected = true;

        private long lastRxTime = 0;
        private long lastRxBytes = 0;
        private long speed = 0;

        private long lastPlayPositionModifyTime = 0;
        private long lastSeekTime = 0;
        private long lastSpeedTime = System.currentTimeMillis();
        private long lastFastSpeedTime = 0;

        private int mrtClientStatus;
        private int mrtRequestBlock;
        private int mrtRequestRetryCnt;

        boolean isPlaying() {
            return isPositionMoving;
        }

        synchronized void setPosition() {
            int playerState = mVideoView.getPlayerState();
            long current = playerState == IjkVideoView.STATE_PLAYBACK_COMPLETED ? 0 : mVideoView.getCurrentPosition();
            isPositionMoving = lastPosition == 0 && current > 0 || (current - lastPosition) / 300 != 0;
            //isPositionMoving = lastPosition == 0 && current > 0 || current != lastPosition;
            if (isPositionMoving) {
                lastPlayPositionModifyTime = System.currentTimeMillis();
                isTimeout = false;
            }
            // Timber.e("isPositionMoving:[%s] last[%d] currnet:[%d] move[%d]", isPositionMoving, lastPosition, current, current - lastPosition);
            lastPosition = current;
        }

        public void setTimeout(long reason, String info) {
            // only mrt timeout will be show
            if (reason != CpsOnProxyInfoListener.SOURCE_TIMEOUT_MRT_LOGIN && reason != CpsOnProxyInfoListener.SOURCE_TIMEOUT_MRT_DATA) {
                return;
            }
            timeoutReason = reason;
            timeoutInfo = info;
            isTimeout = true;
        }

        public void onPause() {
            Timber.i("PlayChecker onPause");
            pausePosition = mVideoView.getCurrentPosition();
        }

        private boolean isPause() {
            return pausePosition != -1;
        }

        @Override
        public void run() {
            try {
                exec();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        private void exec() {
            if (!running) return;
            setPosition();
            final boolean isPlaying = isPlaying();

            // 网速相关处理
            long rxBytes = getUidRxBytes();
            long current = System.currentTimeMillis();
            if (lastRxTime > 0) {
                long bytes = rxBytes - lastRxBytes;
                speed = (long) (bytes / ((current - lastRxTime) / 1000.0));
            }
            lastRxBytes = rxBytes;
            lastRxTime = current;
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (mPlayEventListener != null)
                        mPlayEventListener.onPlayCheck(mVideoView.getMediaPlayer(), PLAY_CHECK_STATE, isPlaying ? 1 : 0);
                    if (isPlaying) {
                        if (affectLoadingView)
                            hideLoadingView();
                        // 播放中处理
                        if (isPause() && Math.abs(pausePosition - lastPosition) >= 500) { // 如果之前设置了暂停，则恢复为非暂停
                            if (mMediaController != null && mMediaController.isShowing()) {
                                mMediaController.hide();
                            }
                            pausePosition = -1;
                        }
                    } else if (isPause() && Math.abs(pausePosition - lastPosition) < 500) {
                        // 暂停的处理
                        if (mMediaController != null && !mMediaController.isShowing()) {
                            mMediaController.showForever();
                        }
                    } else {
                        if (mMediaController != null && !mMediaController.isShowing() && affectLoadingView)
                            showLoadingView(getLoadingMsg());
                    }
                    updateSpeedInfo(speed);
                    updateStatView();
                    if (!isPlaying && pausePosition == -1) {
                        // 非暂停原因的播放停止了
                        if (mrtClientStatus == CpsOnProxyInfoListener.MRT_BLOCK_REQUEST_RETRY && mrtRequestRetryCnt > 6) {
                            if (mPlayEventListener != null)
                                mPlayEventListener.onInfo(mVideoView.getMediaPlayer(), MRT_ERROR_DISCONNECTED, mrtClientStatus);
                        }
                        if (mrtClientStatus == CpsOnProxyInfoListener.MRT_CLIENT_STOP
                                || mrtClientStatus == CpsOnProxyInfoListener.MRT_BLOCK_REQUEST_RETRY && mrtRequestRetryCnt > MRT_RETRY_LIMIT
                                || mrtClientStatus == CpsOnProxyInfoListener.MRT_CLIENT_LOGIN_END_FAILED) {
                            Timber.w("mrt client status error: %d", mrtClientStatus);
                            if (mPlayEventListener != null)
                                mPlayEventListener.onError(mVideoView.getMediaPlayer(), MRT_ERROR_DISCONNECTED, mrtClientStatus, mrtRequestBlock + "");
                        }
                    }
                }
            });
            mCheckerHandler.postDelayed(this, 1000);
        }

        void cleanErrorState() {
            mrtClientStatus = 0;
            mrtRequestRetryCnt = 0;
        }

        void setAffectLoadingView(boolean affectLoadingView) {
            this.affectLoadingView = affectLoadingView;
        }
    }

    //----------------------------------------

    //++++++++++++++++++++++++++++++++++++++++
    // play event
    public void setPlayEventListener(OnPlayEventListener playEventListener) {
        mPlayEventListener = playEventListener;
    }

    public interface OnPlayEventListener {
        void onPrepared(IMediaPlayer mediaPlayer);

        boolean onInfo(IMediaPlayer mediaPlayer, int what, int extra);

        void onCompletion(IMediaPlayer mediaPlayer);

        boolean onError(IMediaPlayer mediaPlayer, int what, int extra, String info);

        void onPreaparError(Throwable throwable);

        void onPlayCheck(IMediaPlayer mediaPlayer, int what, int extra);

        void onSeekComplete(IMediaPlayer mediaPlayer);

        void onSeek(long position);
    }

    private void onLowLevelPlayerBufferingStart() {
        // Timber.e("onLowLevelPlayerBufferingStart");
        showLoadingView("");
        mInLowLevelBuffering = true;
    }

    private void onLowLevelPlayerBufferingEnd() {
        // Timber.e("onLowLevelPlayerBufferingEnd");
        mInLowLevelBuffering = false;
        if (mOnLowLevelBufferingEndAction != null) {
            mOnLowLevelBufferingEndAction.run();
            mOnLowLevelBufferingEndAction = null;
        }
    }

    private Handler mHandler = new Handler(Looper.getMainLooper(), new Handler.Callback() {
        @Override
        public boolean handleMessage(Message msg) {
            switch (msg.what) {
                case MediaPlayer.MEDIA_INFO_BUFFERING_START:
                    Timber.i("MEDIA_INFO_BUFFERING_START");
                    onLowLevelPlayerBufferingStart();
                    break;

                case MediaPlayer.MEDIA_INFO_BUFFERING_END:
                    Timber.i("MEDIA_INFO_BUFFERING_END");
                    onLowLevelPlayerBufferingEnd();
                    break;
            }
            return false;
        }
    });
    //----------------------------------------

    //++++++++++++++++++++++++++++++++++++++++
    // Overridable method

    protected int getLayoutId() {
        return R.layout.catch_up_video_view;
    }

    protected void showLoadingView(String message) {
        mLoadingIcon.setVisibility(VISIBLE);
        mLoadingHint.setVisibility(VISIBLE);
        mIsLoadingViewVisible = true;
        mLoadingMsg.setText(message);
    }

    protected void updateSpeedInfo(long speed) {
        if (!mIsLoadingViewVisible) return;
        mLoadingSpeed.setText(PureStringUtils.getDisplaySpeed(speed));
    }

    protected void hideLoadingView() {
        mLoadingMsg.setText("");
        mLoadingIcon.setVisibility(GONE);
        mLoadingHint.setVisibility(GONE);
        mIsLoadingViewVisible = false;
    }

    protected long getUidRxBytes() {
        return 0;
    }

    protected Observable<CatchUpPlayInfo> getCatchUpPlayInfo() {
        return null;
    }

    protected String getConnectingMsg() {
        return "Connecting...";
    }

    protected String getLoadingMsg() {
        return "Loading...";
    }

    protected CatchUpProxyConfig getCatchUpProxyConfig() {
        Map<String, String> setting = new HashMap<>();
        setting.put(CpsHandler.KEY_DID, "123");
        setting.put(CpsHandler.KEY_APP_VER, "123");
        setting.put(CpsHandler.KEY_PLATFORM, "123");
        setting.put(CpsHandler.KEY_CACHE_SIZE, "16");
        CatchUpProxyConfig catchUpProxyConfig = new CatchUpProxyConfig();
        catchUpProxyConfig.setAllowOnlyOneConnection(false);
        catchUpProxyConfig.setSettings(setting);
        return catchUpProxyConfig;
    }

    protected ISettingsProvider getPlayerSettings() {
        return new DefSettingsProvider();
    }

    protected CatchUpStreamProxy.PlayPositionProvider getPlayPositionProvider() {
        return new CatchUpStreamProxy.PlayPositionProvider() {
            @Override
            public long getPlayPosition() {
                if (mVideoView.isInPlaybackState())
                    return getCurrentPosition();
                else return -1;
            }

            @Override
            public boolean isTrusted() {
                return true;
            }
        };
    }

    protected void stopPlayback() {
        if (mVideoView != null)
            mVideoView.stopPlayback();
    }

    protected void showToast(Context context, String message) {
        // Toast.makeText(mContext, message, Toast.LENGTH_SHORT).show();
    }
    //----------------------------------------

    public void notifyBuffering(int bufferingCount, long bufferingTime) {
        if (!("mrt".equals(mPlayInfo.getProtocol())) && (PrpdStreamProxy.getInstance().getChannelId() >= 0)) {
            JniApiImpl.getInstance().notifyBuffering(PrpdStreamProxy.getInstance().getChannelId(), bufferingCount, (int) bufferingTime);
        }
    }

    public void setDrmInfo(int drmType, boolean multiSession, String licenceServerUrl, Map<String, String> headers, String reqMethod) {
        mDrmType = drmType;
        mMultiSession = multiSession;
        mLicenceServerUrl = licenceServerUrl;
        mDrmReqHeader = headers;
        mDrmReqMethod = reqMethod;
    }

    public int getDrmType() {
        return mDrmType;
    }

    protected void showDebugInfo(boolean show) {
        if (canShowDebugInfo()) {
            if (show) {
                mDebugInfo.setVisibility(VISIBLE);
                mIsDebugViewVisible = true;
            } else {
                mDebugInfo.setVisibility(GONE);
                mIsDebugViewVisible = false;
            }
        }
    }
}
