package com.stream.brt.client;

import com.ott.trans.NetworkState;
import com.stream.brt.client.brtconn.BrtRuntime;
import com.stream.brt.client.brtconn.BrtStreamConnectionFactory;
import com.stream.brt.client.brtconn.BrtStreamCreateCallback;
import com.stream.brt.engine.BrtEngine;
import com.stream.brt.engine.PrtEngine;
import com.stream.brt.engine.ValueProvider;
import com.stream.brt.engine.base.BaseEngine;
import com.stream.brt.engine.base.BaseEngineListener;
import com.stream.brt.engine.metric.BrtMetricStorage;
import com.stream.core.proxy.proxycommon.ForkedStream;
import com.stream.core.proxy.proxycommon.ForkedStreamFactory;
import com.stream.core.proxy.proxycommon.StreamConnectionFactory;
import com.stream.core.proxy.proxycommon.StreamProxyConfig;
import com.stream.core.proxy.proxycommon.StreamProxyUtils;
import com.stream.core.proxy.simpleproxy.SimpleStreamProxy;
import com.stream.prt.PrtEngineCallback;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.tuple.ImmutableTriple;
import org.apache.commons.lang3.tuple.Triple;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 *
 */
public class BrtClient {
    private static Logger logger = LoggerFactory.getLogger(BrtClient.class.getSimpleName());

    //DRM类型
    final public static int DRMTYPE_NODRM = 0;
    final public static int DRMTYPE_GOOSE = 1;
    final public static int DRMTYPE_IRDETO = 2;

    final public static int SOLUTION_TYPE_NONE = 0;
    final public static int SOLUTION_TYPE_MPT = 1;
    final public static int SOLUTION_TYPE_MPQ = 2;
    final public static int SOLUTION_TYPE_VALO_HLS = 3;
    final public static int SOLUTION_TYPE_DRM_HLS = 4;
    final public static int SOLUTION_TYPE_DRM_DASH = 5;
    final public static int SOLUTION_TYPE_VOD_PLAIN_RPD = 6;
    final public static int SOLUTION_TYPE_VOD_DRM_HLS = 7;
    final public static int SOLUTION_TYPE_VOD_DRM_DASH = 8;
    final public static int SOLUTION_TYPE_LIVE_PLAIN_DASH = 9;

    final public static String USE_APP_PROXY = "0";
    final public static String USE_ENGINE_PROXY = "1";

    private String did;
    private String brtUrl;
    private String channelId;
    private String platformInfo;
    private String appVer;
    private ValueProvider valueProvider;
    private BrtClientListener clientListener;
    private BrtMetricStorage brtMetricStorage;

    private BaseEngine baseEngine;
    private SimpleStreamProxy streamProxy;
    private InputStream playerInputStream;
    private PrtEngineCallback prtEngineCallback;
    private boolean isShutdown = false;
    private int mSolutionType = SOLUTION_TYPE_LIVE_PLAIN_DASH;
    private int mDrmType = DRMTYPE_NODRM;

    private String mProxyType = USE_ENGINE_PROXY;

    public BrtClient(String did,
                     String channelId,
                     String brtUrl,
                     String platformInfo,
                     String appVer,
                     final ValueProvider valueProvider,
                     final BrtClientListener clientListener,
                     final BrtMetricStorage brtMetricStorage,
                     final PrtEngineCallback prtEngineCallback) {
        logger.info("BrtClient construct.");
        this.did = did;
        this.channelId = channelId;
        this.brtUrl = brtUrl.replaceFirst("prt", "brt");
        this.platformInfo = platformInfo;
        this.valueProvider = valueProvider;
        this.clientListener = clientListener;
        this.appVer = appVer;
        this.brtMetricStorage = brtMetricStorage;
        if (prtEngineCallback != null) {
            this.mSolutionType = prtEngineCallback.getSolutionType();
            this.mDrmType = prtEngineCallback.getDrmType();
            this.mProxyType = prtEngineCallback.getEproxy();
        }
        if (brtUrl.startsWith("prt")) {
            baseEngine = new PrtEngine(prtEngineCallback);
        } else {
            baseEngine = new BrtEngine();//new FileBrtEngine();//
        }

        if (mProxyType.equals(USE_APP_PROXY)) {
            initStreamProxy();
        }
    }

    public BrtClient(String did,
                     String channelId,
                     String brtUrl,
                     String platformInfo,
                     String appVer,
                     final ValueProvider valueProvider,
                     final BrtClientListener clientListener,
                     final BrtMetricStorage brtMetricStorage) {
        this(did, channelId, brtUrl, platformInfo, appVer, valueProvider, clientListener, brtMetricStorage, null);
    }

    private void initStreamProxy() {
        final StreamConnectionFactory factory = new BrtStreamConnectionFactory(valueProvider, new BrtStreamCreateCallback() {
            @Override
            public BrtRuntime getNextAvailableAccount(String mediaCode) {
                BrtRuntime accountRuntime = new BrtRuntime();
                accountRuntime.setId("xxxyyy");
                accountRuntime.setProviderId("test1");
                accountRuntime.setBrtUrl(brtUrl);
                accountRuntime.setSn(did);
                return accountRuntime;
            }

            @Override
            public InputStream getDataInputStream() {
                return playerInputStream;
            }
        });

        StreamProxyConfig proxyConfig = new StreamProxyConfig();
        //inputstream现在不支持多个连接同时读取数据
        proxyConfig.setAllowOnlyOneConnection(true);
        proxyConfig.setStreamConnectionFactory(factory);
        //客户端缓存block的数量,可以装下一个slice group的brt数据最好
        //proxyConfig.setCacheClientQueueSize(2000);
        proxyConfig.setCacheClientQueueSize(1024);
        //块大小正好是一个slice的长度
        //proxyConfig.setCacheOverflowBlockSize(1312);
        proxyConfig.setCacheOverflowBlockSize(8192);
        setupForkedStreamForProxy(proxyConfig);
        streamProxy = new SimpleStreamProxy(proxyConfig);
    }

    public void start() throws Exception {
        //0动态随机端口
        start(0);
    }

    public void start(int port) throws Exception {
        baseEngine.initModule(did, brtUrl, platformInfo, appVer, new HUdpPlayerReader(), valueProvider, brtMetricStorage);

        //baseEngine.startReader();
        if (mProxyType.equals(USE_APP_PROXY)) {
            streamProxy.start(port);
        }
    }

    public void shutdown() {
        //todo
        try {
            if (streamProxy != null && streamProxy.isRunning()) {
                streamProxy.shutdown();
            }
        } catch (Exception e) {
            logger.error(e, "shutdown stream proxy error");
        }
        try {
            if (baseEngine != null) {
                baseEngine.shutdown();
            }
        } catch (Exception e) {
            logger.error(e, "shutdown baseEngine error");
        }
    }

    public boolean isRunning() {
        if (streamProxy != null && streamProxy.isRunning() && baseEngine != null && baseEngine.isRunning()) {
            return true;
        }
        return false;
    }

    public void closeCurrentConnection() {
        if (streamProxy != null) {
            streamProxy.closeClientSockets(false);
        }
    }

    public int setNetworkState(NetworkState state) {
        return baseEngine.setNetworkState(state);
    }

    public int setTransMask(int mask) {
        return baseEngine.setTransMask(mask);
    }

    public void setLog(boolean enable, int period) {
        baseEngine.setLog(enable, period);
    }

    public int flushLog() {
        return baseEngine.flushLog();
    }

    private String getLocalPlayUrl() throws Exception {
        if (mProxyType.equals(USE_APP_PROXY)) {
            if (streamProxy == null) {
                throw new IllegalStateException("StreamProxy is not init");
            }
            if (!streamProxy.isRunning()) {
                throw new IllegalStateException("StreamProxy is not started yet");
            }
        }

        // String encodedBrtUrl = DESCipherUtils.encrypt(brtUrl, HConstants.proxy_url_des_key);
        String encodedBrtUrl = (RandomStringUtils.randomAlphanumeric(20) + Math.abs(brtUrl.hashCode())).replace("mpd", "aaa").replace("dash", "d");
        int port = streamProxy.getPort();
        final String playUrl = String.format("http://127.0.0.1:%s/%s.ts", port, encodedBrtUrl);
        return playUrl;
    }

    private String getLocalPlayUrl(int chanid, int port) throws Exception {
        String playUrl = "";
        if ((this.mSolutionType == SOLUTION_TYPE_DRM_HLS) || (this.mSolutionType == SOLUTION_TYPE_VALO_HLS)) {
            playUrl = String.format("http://127.0.0.1:%s/%s/master.m3u8", port, chanid);
        } else {
            playUrl = String.format("http://127.0.0.1:%s/%s/master.mpd", port, chanid);
        }
        return playUrl;
    }

    private class HUdpPlayerReader implements BaseEngineListener {

        @Override
        public void onStart() {
            try {
                //callback.onStart(accountRuntime.getSn(), mediaCode);
                logger.info("onStart");
                if (clientListener != null) {
                    clientListener.onStart();
                }
            } catch (Exception e) {
                //logger.error(e, "Error in onStart");
                logger.info(e, "Error in onStart");
            }
        }

        @Override
        public void onAuthPassed() {
            try {
                //callback.onAuthPassed(accountRuntime.getSn(), mediaCode);
                logger.info("onAuthPassed");
                if (clientListener != null) {
                    clientListener.onAuthPassed();
                }
            } catch (Exception e) {
                //logger.error(e, "Error in onAuthPassed");
                logger.info(e, "Error in onAuthPassed");
            }
        }

        @Override
        public void onAuthFailed(int errorCode, String errorMsg) {
            try {
                //createCallback.onAuthFailed(accountRuntime.getSn(), mediaCode, errorCode, errorMsg);
                //isAlive = false;
                logger.info("onAuthFailed:%s, %s", errorCode, errorMsg);
                if (clientListener != null) {
                    clientListener.onAuthFailed(errorCode, errorMsg);
                }
            } catch (Exception e) {
                //logger.error(e, "Error in onAuthFailed");
                logger.info(e, "Error in onAuthFailed");
            }
        }

        @Override
        public void onDataReceiveStart(InputStream inputStream) {
            if (!mProxyType.equals(USE_APP_PROXY)) {
                return;
            }

            playerInputStream = inputStream;
            logger.info("onDataReceiveStart");
            try {
                String playUrl = getLocalPlayUrl();
                if (clientListener != null) {
                    clientListener.onReady(playUrl, new HashMap(0));
                }
            } catch (Exception e) {
                logger.info(e, "Error in onDataReceiveStart");
            }
        }

        @Override
        public void onDataOverflow() {

        }

        @Override
        public void onDataReceiveStop(int errorCode, String errorMsg) {
            try {
                //createCallback.onDataFailed(accountRuntime.getSn(), mediaCode, errorCode, errorMsg);
                //isAlive = false;
                logger.info("onDataReceiveStop:%s, %s", errorCode, errorMsg);
                if (clientListener != null) {
                    clientListener.onError(errorCode, errorMsg);
                }
            } catch (Exception e) {
                //logger.error(e, "Error in onAuthFailed");
                logger.info(e, "Error in onAuthFailed");
            }
        }

        @Override
        public void onStop(boolean serverReason, int lastMsgCode, int innerErrCode) {
            try {
                //createCallback.onStop(accountRuntime.getSn(), mediaCode, serverReason, lastMsgCode, innerErrCode);
                //isAlive = false;
                String msg = String.format("onStop:%s, %s", serverReason, lastMsgCode);
                logger.info(msg);
                if (clientListener != null) {
                    clientListener.onStop(innerErrCode, msg);
                }
            } catch (Exception e) {
                //logger.error(e, "Error in onStop");
                logger.info(e, "Error in onStop");
            }
        }

        @Override
        public void onInfo(boolean serverReason, int lastMsgCode, int innerErrCode) {
            try {
                //clear read buffer if you add other case ,please Call carefully streamProxy.clearDataBuffer() whether or not Need to call
                streamProxy.clearDataBuffer();
                String msg = String.format("onInfor:%s, %s", serverReason, lastMsgCode);
                logger.info(msg);
                if (clientListener != null) {
                    clientListener.onInfo(innerErrCode, msg);
                }
            } catch (Exception e) {
                logger.info(e, "Error in onInfo");
            }
        }

        @Override
        public void onInfo(int what, Map<String, Object> extra) {
            try {
                if (clientListener != null) {
                    clientListener.onInfo(what, extra);
                }
            } catch (Exception e) {
                logger.info(e, "Error in onInfo");
            }
        }

        @Override
        public void onDataReady(int chanid, int port) {
            if (!mProxyType.equals(USE_ENGINE_PROXY)) {
                return;
            }

            try {
                String playUrl = getLocalPlayUrl(chanid, port);
                if (clientListener != null) {
                    clientListener.onReady(playUrl, new HashMap(0));
                }
            } catch (Exception e) {
                logger.info(e, "Error in onReady");
            }
        }
    }

    public void onAppEvent(int event, Map<String, Object> info) {
        BaseEngine engine = baseEngine;
        if (engine != null) {
            engine.onAppEvent(event, info);
        }
    }

    private void setupForkedStreamForProxy(StreamProxyConfig proxyConfig) {
        if (valueProvider != null && valueProvider.val("fork_proxy", false)) {
            final String savePath = valueProvider.val("fork_dir", "");
            proxyConfig.setStreamFactory(new ForkedStreamFactory() {
                @Override
                public ForkedStream createForkedStream(Properties props) {
                    final boolean[] isForkedStreamClosed = {false};
                    final ArrayBlockingQueue<Triple<byte[], Integer, Integer>> dataBufferQ = new ArrayBlockingQueue<>(1000, true);
                    new Thread(new Runnable() {
                        @Override
                        public void run() {
                            FileOutputStream fos = null;
                            try {
                                logger.info("start forked stream thread");
                                File forkedDir = new File(savePath);
                                if (!forkedDir.exists() || !forkedDir.isDirectory()) {
                                    logger.error("forked dir invalid");
                                    return;
                                }
                                fos = new FileOutputStream(new File(forkedDir, channelId + System.currentTimeMillis() + ".ts"));
                                while (!isForkedStreamClosed[0]) {
                                    Triple<byte[], Integer, Integer> data = dataBufferQ.poll(5, TimeUnit.SECONDS);
                                    if (data != null)
                                        fos.write(data.getLeft(), data.getMiddle(), data.getRight());
                                }
                                while (dataBufferQ.size() > 0) {
                                    Triple<byte[], Integer, Integer> data = dataBufferQ.poll(5, TimeUnit.SECONDS);
                                    if (data != null)
                                        fos.write(data.getLeft(), data.getMiddle(), data.getRight());
                                }
                                logger.warn("forked stream exit");
                            } catch (Exception e) {
                                logger.error(e, "forked stream thread error");
                            } finally {
                                StreamProxyUtils.closeQuietly(fos);
                            }
                        }
                    }).start();
                    return new ForkedStream() {
                        @Override
                        public void refresh() {

                        }

                        @Override
                        public void abort() {
                            isForkedStreamClosed[0] = true;
                        }

                        @Override
                        public void write(byte[] data, int offset, int len) throws IOException {
                            if (!dataBufferQ.offer(new ImmutableTriple<>(data, offset, len))) {
                                logger.error("forked stream data buffer Q lose data");
                            }
                        }

                        @Override
                        public void close() throws IOException {
                            isForkedStreamClosed[0] = true;
                        }

                        @Override
                        public void flush() throws IOException {

                        }
                    };
                }
            });
        }
    }
}
