package com.stream.brt.engine;

import com.stream.brt.engine.model.BrtUrlInfo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.net.util.SubnetUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.*;

/**
 */
public class BrtUrlParser {
    /**
     * String singleBrt = "brt://1.2.3.4:99/a/ch1?token=dssdfds&time=150000&valid=120&px=99";
     * String multipleBrt = "brt://1.2.3.4:99,8.8.8.8:21000,7.7.7.7:11000/a/ch1?token=dssdfds&time=150000&valid=120&px=99";
     * String group&maskBrt = "brt://111.222.1.0*24:8080-112.123.1.24:5050,100.23.12.8:10101-110.222.1.0*24:20302-100.23.12.9:9088/a/ch1?token=dssdfds&time=150000&valid=120&px=99";
     */
    public static BrtUrlInfo parse(String brtUrlStr) throws Exception {
        brtUrlStr = StringUtils.trimToNull(brtUrlStr);
        if (brtUrlStr == null) {
            throw new Exception("Brt url is null");
        }
        if (!brtUrlStr.startsWith("brt://")) {
            throw new Exception("Invalid brt url format");
        }
        String schema = "brt";
        //删除 brt:// 头
        String leftUrlPart = brtUrlStr.substring(6);
        int indexFirstSlash = leftUrlPart.indexOf('/');
        if (indexFirstSlash < 0) {
            throw new Exception("Invalid brt url format");
        }

        String hostPart = leftUrlPart.substring(0, indexFirstSlash);
        leftUrlPart = leftUrlPart.substring(indexFirstSlash);

        List<List<BrtUrlInfo.ServerInfo>> serverInfos = parseServerInfos(parseStreamServerGroups(hostPart));

        int indexParamSplitter = leftUrlPart.indexOf('?');
        int indexEndOfPath = indexParamSplitter;
        if (indexParamSplitter < 0) {
            indexEndOfPath = leftUrlPart.length();
        }
        String path = leftUrlPart.substring(0, indexEndOfPath);

        Map<String, String> paramMap = new HashMap<>();
        if (indexParamSplitter > 0) {
            String queryPart = leftUrlPart.substring(indexEndOfPath + 1);
            paramMap = splitQuery(queryPart);
        }
        BrtUrlInfo brtUrl = new BrtUrlInfo(schema, serverInfos, path, paramMap);
        return brtUrl;
    }

    private static List<List<BrtUrlInfo.ServerInfo>> parseServerInfos(List<List<String>> hostServerGroups) throws Exception{
        List<List<BrtUrlInfo.ServerInfo>> groups = new ArrayList<>();
        for (List<String> hostServers:hostServerGroups) {
            List<BrtUrlInfo.ServerInfo> serverInfos = new ArrayList<>();
            for (String hostServer : hostServers) {
                String[] hostParts = hostServer.split(":");
                if (hostParts.length != 2) {
                    throw new Exception("Invalid host format:" + hostServer);
                }
                String host = hostParts[0];
                int port = Integer.parseInt(hostParts[1]);
                BrtUrlInfo.ServerInfo serverInfo = new BrtUrlInfo.ServerInfo(host, port);
                serverInfos.add(serverInfo);
            }
            groups.add(serverInfos);
        }
        return groups;
    }

    private static List<List<String>> parseStreamServerGroups(String string) throws Exception{
        List<List<String>> serverGroups = new ArrayList<>();
        if (StringUtils.isEmpty(string))
            return serverGroups;

        String groupsArr[] = string.split("-");
        for (String group:groupsArr) {
            serverGroups.add(parseServers(group));
        }
        return serverGroups;
    }

    private static List<String> parseServers(String string) throws Exception{
        List<String> servers = new ArrayList<>();
        if (StringUtils.isEmpty(string))
            return servers;

        String serversArr[] = string.split(",");
        for (String server:serversArr) {
            if (server.contains("*")) {
                int portPos = server.indexOf(":");
                if (portPos < 0) {
                    throw new Exception("Invalid host format:" + server);
                }
                String port = server.substring(portPos);
                String real = server.replace("*", "/").substring(0,portPos);
                SubnetUtils subnetUtils = new SubnetUtils(real);
                for (String host:subnetUtils.getInfo().getAllAddresses()) {
                    servers.add(host+port);
                }
            } else {
                servers.add(server);
            }
        }
        return servers;
    }

    private static Map<String, String> splitQuery(String query) throws UnsupportedEncodingException {
        Map<String, String> query_pairs = new LinkedHashMap<String, String>();
        if (StringUtils.trimToNull(query) == null) {
            return query_pairs;
        }
        String[] pairs = query.split("&");
        for (String pair : pairs) {
            int idx = pair.indexOf("=");
            query_pairs.put(URLDecoder.decode(pair.substring(0, idx), "UTF-8"), URLDecoder.decode(pair.substring(idx + 1), "UTF-8"));
        }
        return query_pairs;
    }
}
