package com.stream.brt.engine;

import com.stream.brt.engine.model.BrtUrlInfo;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;

public class DefaultUdpServerProvider implements UdpServerProvider {
    private static Logger logger = LoggerFactory.getLogger(DefaultUdpServerProvider.class.getSimpleName());

    static class Score implements Comparable<Score> {
        List<BrtUrlInfo.ServerInfo> servers;
        String pingTask;
        long score = Long.MAX_VALUE - 1;
        boolean isAbnormal = false;

        public Score(List<BrtUrlInfo.ServerInfo> servers) {
            this.servers = servers;
            this.pingTask = String.format("Ping.%s:%d", servers.get(0).host, servers.get(0).port);
        }

        @Override
        public int compareTo(Score o) {
            return Long.compare(score, o.score);
        }
    }

    private final static Map<List<BrtUrlInfo.ServerInfo>, Score> HIST_SCORES = Collections.synchronizedMap(new HashMap<List<BrtUrlInfo.ServerInfo>, Score>());
    private static List<BrtUrlInfo.ServerInfo> LAST_BEST = null;

    private List<Score> scores = new ArrayList<>();
    private BrtUrlInfo brtUrlInfo = null;
    private Timer pingTimer = new Timer();

    public DefaultUdpServerProvider(String brtUrl, final Ping ping, final Performance performance) throws Exception {
        this.brtUrlInfo = BrtUrlParser.parse(StringUtils.trim(brtUrl));

        for (List<BrtUrlInfo.ServerInfo> serverInfos : brtUrlInfo.getServerInfos()) {
            if (serverInfos.isEmpty())
                continue;

            Score score = HIST_SCORES.get(serverInfos);
            if (score == null) {
                score = new Score(serverInfos);
                HIST_SCORES.put(serverInfos, score);
            }
            scores.add(score);
        }

        Collections.sort(scores);

        pingTimer.schedule(new TimerTask() {
            boolean running = false;

            @Override
            public void run() {
                if (running)
                    return;
                running = true;

                for (int i = scores.size() - 1; i >= 0; i--) {
                    Score score = scores.get(i);
                    long begin = System.currentTimeMillis();
                    performance.begin("Ping");
                    long bytes = 0;
                    try {
                        bytes = ping.ping(score.servers);
                        score.score = System.currentTimeMillis() - begin;
                        score.isAbnormal = score.score > BrtEngineConfig.pingTimeout;
                    } catch (Exception e) {
                        score.isAbnormal = true;
                        logger.error(e, score.pingTask);
                    }
                    performance.end("Ping", score.pingTask, bytes);
                }

                Collections.sort(scores);
//                if (logger.isInfo()) {
//                    for (Score score:scores) {
//                        logger.info("Ping %s:%d = %dms", score.server.host, score.server.port, score.score);
//                    }
//                }
                running = false;
            }
        }, 10, BrtEngineConfig.pingInterval);
    }

    @Override
    public BrtUrlInfo getBrtUrlInfo() throws Exception {
        return brtUrlInfo;
    }

    @Override
    public List<BrtUrlInfo.ServerInfo> best(List<List<BrtUrlInfo.ServerInfo>> excludes) throws Exception {
        List<List<BrtUrlInfo.ServerInfo>> serverInfos = new ArrayList<>(brtUrlInfo.getServerInfos());
        // none server or all servers are failed
        if (isEmpty(serverInfos) || excludes.size() == serverInfos.size()) {
            return new ArrayList<>();
        }

        // the excludes is the servers that has been failed in caller, mark it as abnormal
        for (List<BrtUrlInfo.ServerInfo> list : excludes) {
            Score score = HIST_SCORES.get(list);
            if (!score.isAbnormal)
                score.isAbnormal = true;

            if (list.equals(LAST_BEST)) {
                LAST_BEST = null;
            }
        }

        // if the last best server is in the server list, return it.
        if (LAST_BEST != null && serverInfos.contains(LAST_BEST))
            return LAST_BEST;

        // collect normal server and abnormal servers
        List<List<BrtUrlInfo.ServerInfo>> normalServerGroups = new ArrayList<>();
        List<List<BrtUrlInfo.ServerInfo>> abnormalServerGroups = new ArrayList<>();
        for (List<BrtUrlInfo.ServerInfo> list : serverInfos) {
            if (excludes.contains(list)) {
                abnormalServerGroups.add(list);
                continue;
            }

            Score score = HIST_SCORES.get(list);
            if (score.isAbnormal) {
                abnormalServerGroups.add(list);
            } else {
                normalServerGroups.add(list);
            }
        }

        // select normal server group firstly
        List<BrtUrlInfo.ServerInfo> selected = select(normalServerGroups);

        // if not any normal server group can be found, then try abnormal server group
        if (selected.isEmpty()) {
            selected = select(abnormalServerGroups);
        }

        // not any server group can be found
        if (selected.isEmpty())
            return selected;

        // if found one, then mark the servers as normal and best
        Score score = HIST_SCORES.get(selected);
        if (score != null) {
            score.isAbnormal = false;
        }

        LAST_BEST = selected;

        return LAST_BEST;
    }

    private boolean isEmpty(List<List<BrtUrlInfo.ServerInfo>> serverInfos) {
        if (serverInfos == null || serverInfos.isEmpty())
            return true;

        for (List<BrtUrlInfo.ServerInfo> list : serverInfos) {
            if (list.isEmpty())
                continue;

            return false;
        }

        return true;
    }

    private List<BrtUrlInfo.ServerInfo> select(List<List<BrtUrlInfo.ServerInfo>> serverInfos) {
        if (BrtEngineConfig.BRT_SELECT_SERVER_STRATEGY_SEQUENCE.equals(BrtEngineConfig.brtSelServerStrategy)) {
            return sequenceSelect(serverInfos);
        } else {
            return randomSelect(serverInfos);
        }
    }

    private List<BrtUrlInfo.ServerInfo> sequenceSelect(List<List<BrtUrlInfo.ServerInfo>> serverInfos) {
        if (serverInfos == null || serverInfos.isEmpty())
            return new ArrayList<>();

        return serverInfos.get(0);
    }

    private List<BrtUrlInfo.ServerInfo> randomSelect(List<List<BrtUrlInfo.ServerInfo>> serverInfos) {
        final int size = serverInfos == null ? 0 : serverInfos.size();

        switch (size) {
            case 0:
                return new ArrayList<>();

            case 1:
                return serverInfos.get(0);

            default:
                return serverInfos.get(RandomUtils.nextInt(0, size));
        }
    }

    @Override
    public void shutdown() {
        if (pingTimer != null) {
            pingTimer.cancel();
            pingTimer.purge();
            pingTimer = null;
        }
    }
}
