package com.stream.brt.engine;

import android.util.Log;

import com.stream.brt.engine.base.BaseEngine;
import com.stream.brt.engine.base.BaseEngineListener;
import com.stream.brt.engine.metric.BrtEngineMetric;
import com.stream.brt.engine.metric.BrtMetricStorage;
import com.stream.brt.engine.metric.BrtMetricStorageDefault;
import com.stream.brt.engine.model.LiveSliceGroup;
import com.stream.brt.tool.io.PipelineOutputStream;
import com.stream.prt.JniPrtListener;
import com.stream.prt.NetworkState;
import com.stream.prt.PrtConnect;
import com.stream.prt.PrtEngineCallback;
import com.stream.prt.PrtEvent;
import com.stream.prt.PrtMetric;
import com.stream.prt.utils.JniApiImpl;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

// import com.stream.prt.nat.Upnp;

public class PrtEngine extends BaseEngine implements ValueProvider.RefreshListener {
    private final static Logger logger = LoggerFactory.getLogger(PrtEngine.class.getSimpleName());
    private BaseEngineListener engineListener;
    private BrtEngineChannelData channelData;

    //    private BrtConnect connect;
    private String mediaCode;
    private Performance performance;
    private BrtEngineMetric engineMetric;
    private boolean isFirstData = true;
    private long lastWriteClientTime = 0;

    private PrtEngineCallback prtEngineCallback;
    private PrtConnect connect;
    private PipelineOutputStream outputStream;
    private ConcurrentLinkedQueue<PrtEvent> prtEventQueue = new ConcurrentLinkedQueue<>();
    private ArrayBlockingQueue<Map<String, Object>> prtDataQueue = new ArrayBlockingQueue<Map<String, Object>>(100);
    private Thread workThread;
    private Thread dumpThread;
    private boolean isWorking;
    private String channelID;
    private String pridID;
    private String releaseID;
    private String jniFullName;
    private String bikCoreSoFullName;
    private int nativeChannelId;
    private boolean notifyPushFailedEvent = false;
    private boolean pendingBufferClearFlag = false;
    private boolean waitPendingBufferClearFlag = false;
    private long waitPendingBufferClearFlagStartTime = 0L;

    private boolean pendingTestRestartPlayer = false;

    public PrtEngine(PrtEngineCallback callback) {
        super();
        if (null != callback) {
            prtEngineCallback = callback;
        } else {
            logger.error("The PrtEnvBuild is null.");
        }

        outputStream = new PipelineOutputStream(1024 * 1024);
    }

    public void initModule(String did, final String brtUrl, String platform, String appVer, BaseEngineListener playerReader, ValueProvider provider) throws Exception {
        initModule(did, brtUrl, platform, appVer, playerReader, provider, new BrtMetricStorageDefault());
    }

    //brt://10.9.8.7:21000,10.9.8.88:21000/live1/foxtv1?token=xxyyzz&time=1500000&valid=60
    @Override
    public void initModule(String did, final String url, String platform, String appVer, BaseEngineListener playerReader, ValueProvider provider, BrtMetricStorage brtMetricStorage) throws Exception {
        //todo test
        //private int completedGroupCount = 0;
        int statSeq = BrtStat.getSeq();
        BrtStat.put(statSeq, "initBrt");

        provider.addRefreshListener(this);
        initCfg(provider);
        // performance stat init
        performance = new Performance(BrtEngineConfig.isPerformanceStat);
        engineMetric = new BrtEngineMetric(BrtEngineConfig.isMetricOn, brtMetricStorage);
        this.engineListener = playerReader;
        this.channelData = new BrtEngineChannelData();
//        this.connect = new BrtConnect("1", provider, statSeq);
//        this.connect.initModule(did, url, platform, appVer, new ClientConnectListener(), performance, engineMetric);
//        this.mediaCode = this.connect.getMediaCode();
        channelID = prtEngineCallback.getChannelId();
        releaseID = provider.val(BrtEngineConfig.BRT_KEY_CHN, "");
        pridID = provider.val(BrtEngineConfig.BRT_KEY_PRID, "");
        notifyPushFailedEvent = provider.val(BrtEngineConfig.PRT_KEY_NOTIFY_PUSH_FAIL_EVENT, false);
        logger.warn("channel_id is " + channelID);
        jniFullName = provider.val(BrtEngineConfig.BRT_KEY_PRT_JNI, BrtEngineConfig.DFT_BRT_PRT_JNI_NAME);
        if (jniFullName.length() > 0) {
            File file = new File(jniFullName);
            if (!file.exists() || !file.isFile()) {
                jniFullName = "";
            }
        } else {
            jniFullName = "";
        }
        logger.warn("jni full name is " + jniFullName);

        bikCoreSoFullName = provider.val(BrtEngineConfig.BRT_KEY_BIK_CORD_SO_PATH, BrtEngineConfig.DFT_BRT_BIK_CORE_SO_PATH);
        if (bikCoreSoFullName.length() > 0) {
            File file = new File(bikCoreSoFullName);
            if (!file.exists() || !file.isFile()) {
                bikCoreSoFullName = "";
            }
        } else {
            bikCoreSoFullName = "";
        }
        logger.warn("bik core so full name is " + bikCoreSoFullName);

        PrtEngineConfig.prtParams = provider.val(PrtEngineConfig.KEY_PRT_PARAMS, "");
        connect = new PrtConnect(jniFullName, bikCoreSoFullName, prtEngineCallback, new JniPrtListener() {
            @Override
            public int onDataAvail(int channId, int blockId, byte[] bytes) {
//                logger.debug("tm:%d bytes:%d", System.currentTimeMillis(), bytes.length);
                if (channId != nativeChannelId) {
                    logger.error("onDataAvail invalid channel id: [%d] vs [%d]", nativeChannelId, channId);
                    return -1;
                }

                if (isFirstData) {
                    isFirstData = false;
                    engineListener.onDataReceiveStart(outputStream.getInputStream());
                }
                try {
                    if (pendingBufferClearFlag) {
                        pendingBufferClearFlag = false;
                        outputStream.clearBuffer();
                        return -2;
                    }
                    outputStream.write(bytes);
                    return 0;
                } catch (IOException e) {
                    e.printStackTrace();
                    logger.error(e.getMessage());
                    notifyWriteFailed(BaseEngineListener.WRITE_DATA_FAIL_EXCEPTION, e.getMessage());
                    return -1;
                }
            }

            @Override
            public int onDataAvail(int channId, int blockId, byte[] bytes, int offset, int len) {
//                logger.debug("tm:%d bytes:%d noblock", System.currentTimeMillis(), bytes.length);
                if (channId != nativeChannelId) {
                    logger.error("onDataAvail invalid channel id: [%d] vs [%d]", nativeChannelId, channId);
                    return -1;
                }
                if (isFirstData) {
                    isFirstData = false;
                    engineListener.onDataReceiveStart(outputStream.getInputStream());
                }
                int ret;
                try {
                    if (pendingBufferClearFlag) {
                        pendingBufferClearFlag = false;
                        outputStream.clearBuffer();
                        return -2;
                    }
                    ret = outputStream.writeNonBlock(bytes, offset, len);
                    if (ret < 0) {
                        notifyWriteFailed(BaseEngineListener.WRITE_DATA_FAIL_PARTIAL, "");
                    }
                    return ret;
                } catch (IOException e) {
                    e.printStackTrace();
                    logger.error(e.getMessage());
                    notifyWriteFailed(BaseEngineListener.WRITE_DATA_FAIL_EXCEPTION, e.getMessage());
                    return -1;
                }
            }

            @Override
            public int onDataAvail(int channId, int blockId, byte[] bytes, int offset, int len, long timeout) {
                if (channId != nativeChannelId) {
                    logger.error("onDataAvail invalid channel id: [%d] vs [%d]", nativeChannelId, channId);
                    return -1;
                }
//                logger.debug("tm:%d bytes:%d timeout:%d", System.currentTimeMillis(), bytes.length, timeout);
                if (isFirstData) {
                    isFirstData = false;
                    engineListener.onDataReceiveStart(outputStream.getInputStream());
                }
                try {
                    if (pendingTestRestartPlayer) {
                        onEvent(channId, PrtEvent.event_change_stream_restart_player, "", "");
                        pendingTestRestartPlayer = false;
                    }

                    if (waitPendingBufferClearFlag && !pendingBufferClearFlag) {
                        if (System.currentTimeMillis() - waitPendingBufferClearFlagStartTime >= 7000) {
                            logger.info("waitPendingBufferClearFlag timeout");
                            waitPendingBufferClearFlag = false;
                        }
                        return -2;
                    }
                    if (pendingBufferClearFlag) { // 播放器重启后希望清除buffer
                        logger.info("pendingBufferClearFlag is true");
                        waitPendingBufferClearFlag = false;
                        pendingBufferClearFlag = false;
                        outputStream.skipNextRead();
                        outputStream.clearBuffer();
                        return -2;
                    }
                    int writtenBytes = outputStream.write(bytes, offset, len, (int) timeout);
                    if (writtenBytes <= 0) {
                        logger.error("prt engine push data fail");
                        notifyWriteFailed(BaseEngineListener.WRITE_DATA_FAIL_PARTIAL, "");
                    }
                    if (BrtEngineConfig.dumpToFile) {
                        Map<String, Object> prtDataMap = new HashMap<>();
                        prtDataMap.put("data", bytes);
                        prtDataMap.put("channelId", channelID);
                        prtDataMap.put("blockId", blockId);
                        prtDataMap.put("offset", offset);
                        prtDataMap.put("len", writtenBytes);
                        try {
                            if (prtDataQueue.offer(prtDataMap, 1, TimeUnit.SECONDS)) {
                                logger.info("push dump prt data ok: %s - %s - %s - %s [%s]", channelID, blockId, offset, writtenBytes, len);
                            } else {
                                logger.error("push dump prt data failed: %s - %s - %s - %s [%s]", channelID, blockId, offset, writtenBytes, len);
                            }
                        } catch (Throwable t) {
                            logger.error(t, "push dump prt data error: %s - %s - %s - %s [%s]", channelID, blockId, offset, writtenBytes, len);
                        }

                    }
                    return writtenBytes;
                } catch (IOException e) {
                    e.printStackTrace();
                    logger.error(e.getMessage());
                    notifyWriteFailed(BaseEngineListener.WRITE_DATA_FAIL_EXCEPTION, e.getMessage());
                    return -1;
                }
            }

            @Override
            public int onDataAvail(int i, int i1, int i2, byte[] bytes, long l, int i3) {
                return 0;
            }

            @Override
            public int onCheckRecvDataBuffer(int i, int i1, int i2, long l) {
                return 0;
            }

            @Override
            public int onMetric(int channId, PrtMetric metric) {
                if (null != metric) {
                    logger.debug("id:%d metric:%s", channId, metric.toString());
                } else {
                    logger.debug("id:%d metric is null", channId);
                }
                return 0;
            }

            @Override
            public int onMetric(int channId, Map<String, String> map) {
                //logger.debug("id:%d metric size:%d", channId, map.size());
                map.put("channel_id", channelID);
                map.put("prid", pridID);
                map.put("release_id", releaseID);
                engineMetric.saveBrtMetric(map);
                return 0;
            }

            @Override
            public int onState(int i, Map<String, String> map) {
                if (null != prtEngineCallback) {
                    prtEngineCallback.onStateInfo(map);
                }
                return 0;
            }

            @Override
            public void onVersion(String version) {
                logger.debug("version: %s", version);
            }

            @Override
            public void onNatReq(int channId, String externHost, int port) {
                try {
                    logger.debug("channId: %d externHost:%s port:%d", channId, externHost, port);
                    Map<String, Object> ipInfo = new HashMap<>();
                    ipInfo.put(BaseEngineListener.KEY_IP_INFO, externHost);
                    engineListener.onInfo(BaseEngineListener.IP_INFO, ipInfo);
                    // Upnp.getInstance().start(channId, externHost, null, port);
                } catch (Exception e) {
                    logger.error(e, "error onNatReq");
                }
                /*new Thread(new Runnable() {
                    @Override
                    public void run() {
                        long begin = System.currentTimeMillis();
                        List<InetSocketAddress> natList = NatUtils.checkTraceRoute(externHost);
                        InetSocketAddress uPnpIp = NatUtils.checkUPNP(externHost, "Valoroso UPNP2 ", null, port, port, 10, "UDP");
                        natList.add(uPnpIp);
                        InetSocketAddress localIp = NatUtils.checkLocalIp();
                        natList.add(localIp);
                        long end = System.currentTimeMillis();
                        String ipList = null;
                        for (InetSocketAddress net : natList) {
                            if (!StringUtils.isEmpty(ipList)) {
                                ipList = ipList + "," + net.getHostString() + ":" + net.getPort();
                            } else {
                                ipList = net.getHostString() + ":" + net.getPort();
                            }
                        }
                        ipList = "upnp=" + ipList;
                        JniApiImpl.getInstance().setUpnpResult(channId, ipList);
                    }
                }).start();*/
            }

            @Override
            public void onEvent(int channId, int event, String param, String desc) {
                logger.debug("onEvent event:%d params:%s desc:%s", event, param, desc);
                if (event == PrtEvent.event_change_stream_restart_player) {
                    if (PrtEngineConfig.sscType == 0) {
                        waitPendingBufferClearFlag = true;
                        waitPendingBufferClearFlagStartTime = System.currentTimeMillis();
                    }
                }
                prtEventQueue.add(new PrtEvent(channId, event, param, desc));
            }

            @Override
            public int getPlayerCacheTime(int channId) {
                if (channId != nativeChannelId) {
                    logger.error("getPlayerCacheTime invalid channel id: [%d] vs [%d]", nativeChannelId, channId);
                    return -1;
                }
                return prtEngineCallback.getPlayerCacheTime();
            }

            private void notifyWriteFailed(int what, String reason) {
                try {
                    if (notifyPushFailedEvent && engineListener != null) {
                        Map<String, Object> extra = new HashMap<>();
                        extra.put(BaseEngineListener.KEY_EVENT_REASON, reason);
                        extra.put(BaseEngineListener.APP_CHANNEL_ID, channelID);
                        extra.put(BaseEngineListener.KEY_CHANNEL_ID_ENGINE, nativeChannelId);
                        engineListener.onInfo(what, extra);
                    }
                } catch (Throwable t) {
                    // ignore
                }
            }
        });
        connect.initModule(url, channelID, new ClientConnectListener(), provider);
        nativeChannelId = connect.getEngineChannelId();

        isWorking = true;
        workThread = new Thread(new Runnable() {
            @Override
            public void run() {
                while (isWorking) {
                    try {
                        PrtEvent prtEvent = prtEventQueue.poll();
                        if (null != prtEvent) {
                            int event_ID = prtEvent.getEvent();
                            Log.i("[prt]", " rec event_ID = " + event_ID);
                            switch (event_ID) {
                                case PrtEvent.event_proxy_no_signa:
                                case PrtEvent.event_change_stream_no_signal:
                                    connect.shutdown();
                                    engineListener.onStop(true, -1, event_ID);
                                    break;
                                case PrtEvent.event_proxy_verify:
                                    engineListener.onAuthFailed(BrtConstant.INFO_CODE_INVALID_TOKEN2, "invalid_token2");
                                    break;
                                case PrtEvent.event_change_stream_restart_player:
                                    if (PrtEngineConfig.sscType == 0) {
                                        engineListener.onInfo(true, -1, event_ID);
                                    } else {
                                        // do nothing
                                    }
                                    break;
                                case PrtEvent.event_front_start_player:
                                    int port = JniApiImpl.getInstance().setAndGetPort(34567);
                                    Log.i("[prt]", " start player, chid = " + prtEvent.getChannelId() + "; port = " + port);
                                    engineListener.onDataReady(prtEvent.getChannelId(), port);
                                    break;
                                default:
                                    Map<String, Object> extra = new HashMap<>();
                                    extra.put(BaseEngineListener.KEY_EVENT_INDEX, prtEvent.getChannelId());
                                    extra.put(BaseEngineListener.APP_CHANNEL_ID, channelID);
                                    extra.put(BaseEngineListener.KEY_EVENT_PARAM, prtEvent.getParam());
                                    extra.put(BaseEngineListener.KEY_EVENT_DESC, prtEvent.getDesc());
                                    engineListener.onInfo(event_ID, extra);
                            }
                        } else {
                            Thread.sleep(500);
                        }
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
            }
        });
        workThread.start();
        if (BrtEngineConfig.dumpToFile) {
            dumpThread = new Thread(new Runnable() {
                private String currFileName;
                private OutputStream currOut;

                @Override
                public void run() {
                    while (isWorking) {
                        try {
                            Map<String, Object> prtData = prtDataQueue.poll(5, TimeUnit.SECONDS);
                            if (prtData == null) continue;
                            byte[] data = (byte[]) prtData.get("data");
                            String channelId = (String) prtData.get("channelId");
                            int blockId = (Integer) prtData.get("blockId");
                            int offset = (Integer) prtData.get("offset");
                            int len = (Integer) prtData.get("len");
                            String fileName = String.format("%s-%s.ts", channelId, blockId);
                            if (!StringUtils.equals(currFileName, fileName)) {
                                IOUtils.closeQuietly(currOut);
                                currFileName = fileName;
                                currOut = new FileOutputStream(new File(BrtEngineConfig.dumpPath, fileName));
                                currOut.write(data, offset, len);
                                currOut.flush();
                            } else {
                                currOut.write(data, offset, len);
                                currOut.flush();
                            }
                            logger.info("dump to file ok: %s - %s - %s - %s", channelId, blockId, offset, len);
                        } catch (Exception e) {
                            logger.error(e, "dump to file error!");
                        }
                    }
                }
            }, "prt-dump-thread-" + channelID);
            dumpThread.start();
        }
        /*new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    runPlayerTask();
                } catch (Exception e) {
                    logger.error(e, "[%s] Error in player run()", mediaCode);
                }
            }
        }).start();*/
    }

    public void initCfg(ValueProvider provider) {
        //Engine config begin
        BrtEngineConfig.serverTimeout =
                provider.val(BrtEngineConfig.SERVER_TIMEOUT, BrtEngineConfig.DFT_SERVER_TIMEOUT);

        BrtEngineConfig.receiveGroupTimeout =
                provider.val(BrtEngineConfig.RECEIVE_GROUP_TIMEOUT, BrtEngineConfig.DFT_RECEIVE_GROUP_TIMEOUT);

        BrtEngineConfig.loginTimeout =
                provider.val(BrtEngineConfig.LOGIN_TIMEOUT, BrtEngineConfig.DFT_LOGIN_TIMEOUT);

        if (BrtEngineConfig.loginTimeout <= BrtEngineConfig.DFT_AUTH_MAX_RTT)
            BrtEngineConfig.loginTimeout = BrtEngineConfig.DFT_LOGIN_TIMEOUT;

        BrtEngineConfig.pingTimeout =
                provider.val(BrtEngineConfig.PING_TIMEOUT, BrtEngineConfig.DFT_PING_TIMEOUT);

        BrtEngineConfig.pingInterval =
                provider.val(BrtEngineConfig.PING_INTERVAL, BrtEngineConfig.DFT_PING_INTERVAL);

        BrtEngineConfig.groupResendDuration =
                provider.val(BrtEngineConfig.GROUP_RESEND_DURATION, BrtEngineConfig.DFT_GROUP_RESEND_DURATION);
        if (BrtEngineConfig.groupResendDuration <= 100)
            BrtEngineConfig.groupResendDuration = BrtEngineConfig.DFT_GROUP_RESEND_DURATION;

        BrtEngineConfig.isPerformanceStat = provider.val(BrtEngineConfig.IS_PERFORMANCE_STAT, false);

        BrtEngineConfig.performanceStatDur = provider.val(BrtEngineConfig.PERFORMANCE_STAT_DURATION, 600_000);
        if (BrtEngineConfig.performanceStatDur < 1000) {
            BrtEngineConfig.performanceStatDur = 0;
            BrtEngineConfig.isPerformanceStat = false;
        }

        BrtEngineConfig.receiveUdpTimeout =
                provider.val(BrtEngineConfig.RECEIVE_UDP_TIMEOUT, BrtEngineConfig.DFT_RECEIVE_GROUP_TIMEOUT);
        if (BrtEngineConfig.receiveUdpTimeout < 5000)
            BrtEngineConfig.receiveUdpTimeout = 19_000;

        BrtEngineConfig.groupCacheCount =
                provider.val(BrtEngineConfig.GROUP_CACHE_COUNT, BrtEngineConfig.DFT_GROUP_CACHE_CNT);

        BrtEngineConfig.serverPort =
                provider.val(BrtEngineConfig.SERVER_PORT, BrtEngineConfig.DFT_SERVER_PORT);

        BrtEngineConfig.resendOneByOneRatio =
                provider.val(BrtEngineConfig.RESEND_ONE_BY_ONE_RATIO, BrtEngineConfig.DFT_RESEND_ONE_BY_ONE_RATIO);

        BrtEngineConfig.groupChkResendDuration =
                provider.val(BrtEngineConfig.GROUP_CHK_RESEND_DURATION, BrtEngineConfig.DFT_GROUP_CHK_RESEND_DURATION);

        BrtEngineConfig.maxRTO =
                provider.val(BrtEngineConfig.MAX_RTO, BrtEngineConfig.DFT_MAX_RTO);

        BrtEngineConfig.minRTO =
                provider.val(BrtEngineConfig.MIN_RTO, BrtEngineConfig.DFT_MIN_RTO);

        BrtEngineConfig.rtoMultiFactor =
                provider.val(BrtEngineConfig.RTO_MULTI_FACTOR, BrtEngineConfig.DFT_RTO_MULTI_FACTOR);

        BrtEngineConfig.authMaxRTT =
                provider.val(BrtEngineConfig.AUTH_MAX_RTT, BrtEngineConfig.DFT_AUTH_MAX_RTT);

        if (BrtEngineConfig.authMaxRTT < 500 || BrtEngineConfig.authMaxRTT >= BrtEngineConfig.DFT_LOGIN_TIMEOUT)
            BrtEngineConfig.authMaxRTT = BrtEngineConfig.DFT_AUTH_MAX_RTT;

        BrtEngineConfig.maxThrottleSendBwLevel =
                provider.val(BrtEngineConfig.MAX_THROTTLE_SEND_BW_LEVEL, BrtEngineConfig.DFT_MAX_THROTTLE_SEND_BW_LEVEL);

        BrtEngineConfig.minThrottleSendBwLevel =
                provider.val(BrtEngineConfig.MIN_THROTTLE_SEND_BW_LEVEL, BrtEngineConfig.DFT_MIN_THROTTLE_SEND_BW_LEVEL);

        BrtEngineConfig.initRtoMultiFactor =
                provider.val(BrtEngineConfig.INIT_RTO_MULTI_FACTOR, BrtEngineConfig.DFT_INIT_RTO_MULTI_FACTOR);

        BrtEngineConfig.brtCheckResendStrategy =
                provider.val(BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY, BrtEngineConfig.DFT_BRT_CHECK_RESEND_STRATEGY);

        BrtEngineConfig.brtResendStrategy =
                provider.val(BrtEngineConfig.BRT_RESEND_STRATEGY, BrtEngineConfig.DFT_BRT_RESEND_STRATEGY);

        BrtEngineConfig.brtThrottleDuration =
                provider.val(BrtEngineConfig.THROTTLE_DURATION, BrtEngineConfig.DFT_THROTTLE_DURATION);
        if (BrtEngineConfig.brtThrottleDuration <= 0)
            BrtEngineConfig.brtThrottleDuration = BrtEngineConfig.DFT_THROTTLE_DURATION;

        BrtEngineConfig.brtDropTest =
                provider.val(BrtEngineConfig.BRT_DROP_TEST, BrtEngineConfig.DFT_BRT_DROP_TEST);

        BrtEngineConfig.brtDropRatio =
                provider.val(BrtEngineConfig.BRT_DROP_RATIO, BrtEngineConfig.DFT_BRT_DROP_RATIO);
        if (BrtEngineConfig.brtDropRatio < 0) {
            BrtEngineConfig.brtDropRatio = 0;
        } else if (BrtEngineConfig.brtDropRatio > 100) {
            BrtEngineConfig.brtDropRatio = 100;
        }

        if (BrtEngineConfig.brtDropRatio == 0) {
            BrtEngineConfig.brtDropTest = false;
        }

        BrtEngineConfig.brtResendStrategyGroupCnt =
                provider.val(BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY_GROUP_CNT, BrtEngineConfig.DFT_BRT_CHECK_RESEND_STRATEGY_GROUP_CNT);

        BrtEngineConfig.brtSelServerStrategy =
                provider.val(BrtEngineConfig.BRT_SELECT_SERVER_STRATEGY, BrtEngineConfig.DFT_BRT_SELECT_SERVER_STRATEGY);

        // 如果配置了短key，则用短key的配置，通常在brt url中配置
        BrtEngineConfig.brtSelServerStrategy =
                provider.val(BrtEngineConfig.BRT_SELECT_SERVER_STRATEGY_SHORT, BrtEngineConfig.brtSelServerStrategy);

        BrtEngineConfig.brtSoTimeout =
                provider.val(BrtEngineConfig.BRT_SO_TIMEOUT, BrtEngineConfig.DFT_BRT_SO_TIMEOUT);

        BrtEngineConfig.isMetricOn = provider.val(BrtEngineConfig.BRT_METRIC_STAT, true);

        BrtEngineConfig.brtKeepOverdueSize = provider.val(BrtEngineConfig.BRT_KEEP_OVERDUE_SIZE, BrtEngineConfig.DFT_KEEP_OVER_DUE_SIZE);

        BrtEngineConfig.brtHBInterval = provider.val(BrtEngineConfig.BRT_HB_INTERVAL, BrtEngineConfig.DFT_BRT_HB_INTERVAL);

        BrtEngineConfig.brtVer = provider.val(BrtEngineConfig.BRT_VER, BrtEngineConfig.DFT_BRT_VER);

        BrtEngineConfig.dumpToFile = provider.val(BrtEngineConfig.PRT_DUMP_TO_FILE, false);
        BrtEngineConfig.dumpPath = provider.val(BrtEngineConfig.PRT_DUMP_PATH, "");
        if (StringUtils.isEmpty(BrtEngineConfig.dumpPath)) {
            BrtEngineConfig.dumpToFile = false;
        }
        //Engine config end
    }

    @Override
    public void shutdown() {
        if (this.connect != null) {
            this.connect.shutdown();
        }

        try {
            outputStream.close();
        } catch (Exception e) {
            logger.error(e, "Error in close playerOutputStream, ignore it");
        }

        performance.stop();

        isWorking = false;
        workThread.interrupt();
        if (dumpThread != null) dumpThread.interrupt();
        try {
            workThread.join();
            if (dumpThread != null) dumpThread.join();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    @Override
    public boolean isRunning() {
        PrtConnect cnn = this.connect;
        if (cnn != null) {
            return this.connect.isConnected();
        }
        return false;
    }

    @Override
    public int setNetworkState(NetworkState state) {
        PrtConnect cnn = this.connect;
        if (cnn != null) {
            return cnn.setNetworkState(state);
        } else {
            logger.warn("connect is null");
            return -1;
        }
    }

    @Override
    public int setTransMask(int mask) {
        PrtConnect cnn = this.connect;
        if (cnn != null) {
            return cnn.setTransMask(mask);
        } else {
            logger.warn("connect is null");
            return -1;
        }
    }

    @Override
    public void setLog(boolean enable, int period) {
        PrtConnect cnn = this.connect;
        if (cnn != null) {
            cnn.setLog(enable, period);
        } else {
            logger.warn("connect is null");
        }
    }

    @Override
    public int flushLog() {
        PrtConnect cnn = this.connect;
        if (cnn != null) {
            return cnn.flushLog();
        } else {
            logger.warn("connect is null");
            return -1;
        }
    }

    @Override
    public void onAppEvent(int event, Map<String, Object> info) {
        PrtConnect cnn = this.connect;
        if (info != null && info.containsKey("channel_id")) {
            if (!StringUtils.equals(String.valueOf(info.get("channel_id")), channelID)) {
                // channel 改变后不进行处理
                return;
            }
        }
        switch (event) {
            case 101:
                logger.info("set pendingBufferClearFlag true");
                pendingBufferClearFlag = true;
                outputStream.clearBuffer();
                break;
            case 102: // token changed
                String token = (String) info.get("token");
                String eventChannelId = (String) info.get("channel_id");
                if (StringUtils.equals(eventChannelId, channelID)) {
                    if (StringUtils.isNotEmpty(token)) {
                        cnn.updateChannel("tkn", token);
                    }
                }
                break;
            case 701:
            case 702:
                if (cnn != null) {
                    int times = (int) info.get("times");
                    long duration = (long) info.get("duration");
                    int durationInt = (int) duration;
                    if (durationInt < 0) {
                        durationInt = Integer.MAX_VALUE;
                    }
                    cnn.notifyBuffering(times, durationInt);
                }
                break;
            case 999:
                pendingTestRestartPlayer = true;
                break;
        }
    }

    // 从LiveModule的本地缓冲读取数据，通过TcpClientSocket将数据发送给Video Player
    private void runPlayerTask() throws IOException {
        //logger.info("[%s] Started runPlayerTask()", this.mediaCode, playerReceiveMode);
        //1024k 临时缓冲
        PipelineOutputStream outputStream = new PipelineOutputStream(1024 * 1024);
        final InputStream playerInputStream = outputStream.getInputStream();

        Runnable onDataReceivedAction = new Runnable() {
            @Override
            public void run() {
                engineListener.onDataReceiveStart(playerInputStream);
            }
        };

        while (true) {
            //判断系统状态，决定是否退出
            if (!connect.isConnected()) {
                break;
            }
            try {
                while (connect.isConnected()) {
                    boolean gotData = processAvailableSliceData(outputStream, onDataReceivedAction);
                    //如果上一次读取到了数据，则再次尝试读取下一个，否则等一会再尝试
                    if (!gotData) {
                        break;
                    }
                }
                //TODO 测试代码
                // 默认设置是200ms，这里为了方便调试，设置为500
                Thread.sleep(5);
            } catch (Exception e) {
                if (e instanceof IOException && e.getMessage().equalsIgnoreCase("Pipe closed")) {
                    logger.info("[%s] Stopped runPlayerTask() by engine closed", this.mediaCode);
                } else {
                    logger.error(e, "[%s] Error in runPlayerTask for channel", this.mediaCode);
                }
            }
        }
        try {
            outputStream.close();
        } catch (Exception e) {
            logger.error(e, "Error in close playerOutputStream, ignore it");
        }

        logger.info("[%s] Ended runPlayerTask()", this.mediaCode);
    }

    private boolean processAvailableSliceData(OutputStream outputStream, Runnable dataReceivedAction) throws IOException, InterruptedException {
        LiveSliceGroup firstGroup = channelData.getFirstGroup();
        //todo c++代码没有找到超时逻辑。如果一个group长时间没有数据，则可以跳过
        if (firstGroup == null) {
            //第一个数据group是null
//            if (logger.isDebug())
//                logger.debug("[%s] Player: check top group is null. Ignore it", this.mediaCode);
            return false;
        }
        if (logger.isDebug())
            logger.debug("[%s] Player: check top group. group[%s], count %s, completed[%s]",
                    this.mediaCode, firstGroup.getExSeq(), firstGroup.getCount(), firstGroup.isCompleted());

        //删除已经写完的group
        if (!firstGroup.isCompleted()) {
            return false;
        }

        // only for local test
        //lost(firstGroup);

        firstGroup.setRtt(ResendStat.getRtt());
        // 数据已经到达，回调调用者，做好数据接收准备
        if (isFirstData && dataReceivedAction != null) {
            isFirstData = false;
            dataReceivedAction.run();
        }

        // todo:往output stream写数据, 线程可能会被挂住, output stream的write方法一直不返回，导致播放器卡住不播放，数据一直从缓冲区溢出丢弃
        try {
            firstGroup.readSliceTreeMap(mediaCode, outputStream);
            //completedGroupCount++;
        } catch (Exception e) {
            logger.error("processAvailableSliceData [%s] Error in flushing data group", this.mediaCode);
        }
//        BrtStat.put(firstGroup, connect.getResendStat(), BrtConnect.RemoveGrpEvent.COMPLETE);//todo

        channelData.removeGroup(firstGroup);
        lastWriteClientTime = System.currentTimeMillis();
        return true;
    }

// only for local test
//    private int lostCnt = -1;
//    private void lost(LiveSliceGroup group) {
//        for (Map.Entry<Integer, LiveSliceSubGroup> entry : group.getSubGroupTreeMap().entrySet()) {
//            LiveSliceSubGroup subGroup = entry.getValue();
//            Set<Integer> lost = new HashSet<>();
//            LiveSlice slices[] = subGroup.getSlices();
//            if (lostCnt < 0) {
//                lostCnt = group.getSubGroupParityShards();
//            }
//
//            while (lost.size() < lostCnt) {
//                int i = RandomUtils.nextInt(0, slices.length);
//                if (lost.contains(i))
//                    continue;
//
//                slices[i] = null;
//                lost.add(i);
//            }
//        }
//        lostCnt--;
//    }

    private void handleGroupCacheOverflow() {
        //保存数据成功
        //如果缓存超限，则删除旧数据。因为这里是BrtConnect同步逐个packet处理，所以最多只会多出一个数据
        int curGroupCount = channelData.getGroupCount();
        performance.begin("GroupCache", Performance.Level.IMPORTANT);
        if (curGroupCount > BrtEngineConfig.groupCacheCount) {
            performance.end("GroupCache", Performance.Level.IMPORTANT, "GroupCache.Overflow", curGroupCount);
            engineListener.onDataOverflow();

            do {
                LiveSliceGroup sliceGroup = channelData.getFirstGroup();
                if (sliceGroup == null)
                    break;

                channelData.removeGroup(sliceGroup);
//                BrtStat.put(sliceGroup, connect.getResendStat(), BrtConnect.RemoveGrpEvent.DROP_OVERCACHE);//todo
                curGroupCount = channelData.getGroupCount();
                if (logger.isInfo())
                    logger.info("[%s] Current group count[%s] is morn than limit %s, drop older group seq[%s], lastWriteClient:%dms ago",
                            mediaCode, curGroupCount, BrtEngineConfig.groupCacheCount, sliceGroup.getExSeq(), System.currentTimeMillis() - lastWriteClientTime);
            } while (curGroupCount > BrtEngineConfig.groupCacheCount);
        } else {
            performance.end("GroupCache", "GroupCache.normal", curGroupCount);
        }
    }

    @Override
    public void onValueRefresh(ValueProvider provider) {
        initCfg(provider);
    }

    private class ClientConnectListener implements BrtConnectListener {

        @Override
        public void onStart() {
            engineListener.onStart();
        }

        @Override
        public void onAuthPassed(Map info) {
            isFirstData = true;
            engineListener.onAuthPassed();
        }

        @Override
        public void onAuthFailed(int code, String msg) {
            engineListener.onAuthFailed(code, msg);
        }

        @Override
        public void onDataBegin(LiveSliceGroup sliceGroup) {
            boolean added = channelData.addGroup(sliceGroup);
            if (!added) {
                if (logger.isDebug())
                    logger.debug("[%s] duplicated slice group %s, ignore it", mediaCode, sliceGroup.getExSeq());
            }
        }

        @Override
        public void onDataCompleted(LiveSliceGroup sliceGroup) {
            handleGroupCacheOverflow();
        }

        @Override
        public void onDataDropped(LiveSliceGroup sliceGroup, String msg) {
            channelData.removeGroup(sliceGroup);
        }

        @Override
        public void onInfo(int code, String msg, Map info) {

        }

        @Override
        public void onDisconnected(boolean serverReason, int lastMsgCode, int innerErrCode) {
            engineListener.onStop(serverReason, lastMsgCode, innerErrCode);
        }

        @Override
        public void onDataReady(int chanid, int port) {
            engineListener.onDataReady(chanid, port);
        }
    }
}
