package com.stream.brt.prot;

import com.stream.brt.engine.Performance;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 */
public class ProtocolBuilder {
	public static LivePacket parseLivePacket(byte[] bytes) {
		return parseLivePacket(bytes, null);
	}

	public static LivePacket parseLivePacket(byte[] bytes, ByteBuffer byteBuffer, Performance performance) {
		if (bytes.length < LivePacketHeader.HEADER_LENGTH_VER1) {
			if (performance != null)
				performance.end("parseLivePacket", "parseLivePacket_lt" + LivePacketHeader.HEADER_LENGTH_VER1,
						bytes.length);
			throw new IllegalArgumentException(String.format("Live packet len should >= %d, but get %d",
					LivePacketHeader.HEADER_LENGTH_VER1, bytes.length));
		}
		byteBuffer.position(0);
		byteBuffer.put(bytes);
		
		byteBuffer.position(0);
		
		LivePacketHeader header = new LivePacketHeader();
		int protocolVer = byteBuffer.getInt();

		if (protocolVer == RelayLivePacketHeader.PROTOCOL_RELAY) {
			return parseRelayLivePacket(bytes,byteBuffer);
		}
		
		header.setProtocol(protocolVer);
		
		short version = byteBuffer.getShort();
		header.setVersion(version);
		
		short encryptType = byteBuffer.getShort();
		header.setEncryptType(encryptType);
		
		int nMsgCode = byteBuffer.getInt();
		header.setMsgCode(nMsgCode);
		
		int channelId = byteBuffer.getInt();
		header.setChannelId(channelId);
		
		long gap1 = byteBuffer.getLong();
		header.setSessionKey1(gap1);
		
		long gap2 = byteBuffer.getLong();
		header.setSessionKey2(gap2);
		
		int m_dataSize = byteBuffer.getInt();
		header.setGroupBaseSeq(m_dataSize);
		
		short field_36 = byteBuffer.getShort();
		header.setGroupBufferCount(field_36);
		
		byteBuffer.getShort();
		
		int field_40 = byteBuffer.getInt();
		// System.out.println("f40:" + f40);
		header.setGroupSeq(field_40);

		int field_44 =  byteBuffer.getInt();
		// System.out.println("f44:" + f44);
		header.setBackup1(field_44);

		int m_PacketTotalCount = byteBuffer.getInt();
		// System.out.println("nCount:" + nCount);
		header.setCount(m_PacketTotalCount);

		int m_nPacketSeq = byteBuffer.getInt();
		// System.out.println("nSeq:" + nSeq);
		header.setSeq(m_nPacketSeq);
		
		byte[] payload = null;
		if (header.getVersion() == LivePacketHeader.VERSION_1) {
			int payloadLen = bytes.length - LivePacketHeader.HEADER_LENGTH_VER1;

			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, LivePacketHeader.HEADER_LENGTH_VER1, payload, 0, payloadLen);
			}
		} else if (header.getVersion() == LivePacketHeader.VERSION_2 ||
				header.getVersion() == LivePacketHeader.VERSION_3) {
			int requestSeq = byteBuffer.getInt();
			header.setReqSeq(requestSeq);
			
			
			int payloadLen = bytes.length - LivePacketHeader.HEADER_LENGTH_VER2;

			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, LivePacketHeader.HEADER_LENGTH_VER2, payload, 0, payloadLen);
			}
		} else
			throw new IllegalArgumentException(
					String.format("Live packet version incorrect , get %d", header.getVersion()));
		LivePacket packet = new LivePacket();
		packet.setHeader(header);
		packet.setData(payload);
		packet.setRelayMode(false);
		if (payload != null) {
			packet.setDataSize(payload.length);
		}
		
		return packet;
	}

	private static LivePacket parseRelayLivePacket(byte[] bytes ,ByteBuffer byteBuffer) {
		RelayLivePacketHeader header = new RelayLivePacketHeader();

		header.setRelayProtocol(RelayLivePacketHeader.PROTOCOL_RELAY);

		int sourceIP =byteBuffer.getInt();
		header.setSourceIP(sourceIP);

		int sourcePort = byteBuffer.getShort()&0xFFFF;
		header.setSourcePort(sourcePort);

		int relayPort = byteBuffer.getShort()&0xFFFF;
		header.setRelayPort(relayPort);

		int checksum = byteBuffer.getShort()&0xFFFF;
		// System.out.println("field_36:" + field_36);
		header.setChecksum(checksum);

		int protocolVer = byteBuffer.getInt();
		header.setProtocol(protocolVer);

		short version = byteBuffer.getShort();
		header.setVersion(version);

		short encryptType = byteBuffer.getShort();
		// System.out.println("field_36:" + field_36);
		header.setEncryptType(encryptType);

		int nMsgCode = byteBuffer.getInt();
		// System.out.println("nMsgCode:" + nMsgCode);
		header.setMsgCode(nMsgCode);

		int m_field_8 = byteBuffer.getInt();
		// System.out.println("nChannelId:" + nChannelId);
		header.setChannelId(m_field_8);

		long gap1 = byteBuffer.getLong();
		// System.out.println("gap1:" + gap1);
		header.setSessionKey1(gap1);

		long gap2 = byteBuffer.getLong();
		// System.out.println("gap2:" + gap2);
		header.setSessionKey2(gap2);

		int m_dataSize = byteBuffer.getInt();
		// System.out.println("field_28:" + field_28);
		header.setGroupBaseSeq(m_dataSize);

		// readTmp = new byte[4];
		// readLen = inputStream.read(readTmp, 0, 4);
		// int m_field_32 = HNumberUtils.intFromByteArray(readTmp);
		// //System.out.println("f32:" + f32);
		// header.setF32(m_field_32);

		short field_36 = byteBuffer.getShort();
		// System.out.println("field_36:" + field_36);
		header.setGroupBufferCount(field_36);

		// 4个byte对齐，这两个byte忽略
		byteBuffer.getShort();

		int field_40 = byteBuffer.getInt();
		// System.out.println("f40:" + f40);
		header.setGroupSeq(field_40);

		int field_44 =  byteBuffer.getInt();
		// System.out.println("f44:" + f44);
		header.setBackup1(field_44);

		int m_PacketTotalCount = byteBuffer.getInt();
		// System.out.println("nCount:" + nCount);
		header.setCount(m_PacketTotalCount);

		int m_nPacketSeq = byteBuffer.getInt();
		// System.out.println("nSeq:" + nSeq);
		header.setSeq(m_nPacketSeq);

		byte[] payload = null;
		if (header.getVersion() == LivePacketHeader.VERSION_1) {
			int payloadLen = bytes.length - RelayLivePacketHeader.HEADER_LENGTH_VER1;
			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, RelayLivePacketHeader.HEADER_LENGTH_VER1, payload, 0, payloadLen);
			}
		} else if (header.getVersion() == LivePacketHeader.VERSION_2 ||
				header.getVersion() == LivePacketHeader.VERSION_3) {
			int requestSeq = byteBuffer.getInt();
			header.setReqSeq(requestSeq);

			int payloadLen = bytes.length - RelayLivePacketHeader.HEADER_LENGTH_VER2;

			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, RelayLivePacketHeader.HEADER_LENGTH_VER2, payload, 0, payloadLen);
			}
		} else
			throw new IllegalArgumentException(
					String.format("Live packet version incorrect , get %d", header.getVersion()));

		LivePacket packet = new LivePacket();
		packet.setHeader(header);
		packet.setData(payload);
		packet.setRelayMode(true);
		if (payload != null) {
			packet.setDataSize(payload.length);
		}

		return packet;
	}
	
	
	public static LivePacket parseLivePacket(byte[] bytes, Performance performance) {
		if (performance != null)
			performance.begin("parseLivePacket");
		if (bytes.length < LivePacketHeader.HEADER_LENGTH_VER1) {
			if (performance != null)
				performance.end("parseLivePacket", "parseLivePacket_lt" + LivePacketHeader.HEADER_LENGTH_VER1,
						bytes.length);
			throw new IllegalArgumentException(String.format("Live packet len should >= %d, but get %d",
					LivePacketHeader.HEADER_LENGTH_VER1, bytes.length));
		}
		ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);

		LivePacketHeader header = new LivePacketHeader();
		int readLen = -1;
		byte[] readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int protocolVer = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("protocolVer:" + protocolVer);

		if (protocolVer == RelayLivePacketHeader.PROTOCOL_RELAY) {
			return parseRelayLivePacket(inputStream, bytes);
		}

		header.setProtocol(protocolVer);

		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);
		short version = HNumberUtils.shortFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setVersion(version);

		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);
		short encryptType = HNumberUtils.shortFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setEncryptType(encryptType);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int nMsgCode = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nMsgCode:" + nMsgCode);
		header.setMsgCode(nMsgCode);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int m_field_8 = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nChannelId:" + nChannelId);
		header.setChannelId(m_field_8);

		readTmp = new byte[8];
		readLen = inputStream.read(readTmp, 0, 8);
		long gap1 = HNumberUtils.longFromByteArray(readTmp);
		// System.out.println("gap1:" + gap1);
		header.setSessionKey1(gap1);

		readTmp = new byte[8];
		readLen = inputStream.read(readTmp, 0, 8);
		long gap2 = HNumberUtils.longFromByteArray(readTmp);
		// System.out.println("gap2:" + gap2);
		header.setSessionKey2(gap2);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int m_dataSize = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("field_28:" + field_28);
		header.setGroupBaseSeq(m_dataSize);

		// readTmp = new byte[4];
		// readLen = inputStream.read(readTmp, 0, 4);
		// int m_field_32 = HNumberUtils.intFromByteArray(readTmp);
		// //System.out.println("f32:" + f32);
		// header.setF32(m_field_32);

		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);
		short field_36 = HNumberUtils.shortFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setGroupBufferCount(field_36);

		// 4个byte对齐，这两个byte忽略
		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int field_40 = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("f40:" + f40);
		header.setGroupSeq(field_40);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int field_44 = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("f44:" + f44);
		header.setBackup1(field_44);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int m_PacketTotalCount = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nCount:" + nCount);
		header.setCount(m_PacketTotalCount);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		// if(readLen != 4){
		// System.out.println("nPacketSeq bytes len " + readLen);
		// }
		int m_nPacketSeq = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nSeq:" + nSeq);
		header.setSeq(m_nPacketSeq);

		byte[] payload = null;
		if (header.getVersion() == LivePacketHeader.VERSION_1) {
			int payloadLen = bytes.length - LivePacketHeader.HEADER_LENGTH_VER1;

			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, LivePacketHeader.HEADER_LENGTH_VER1, payload, 0, payloadLen);
			}
		} else if (header.getVersion() == LivePacketHeader.VERSION_2 ||
				header.getVersion() == LivePacketHeader.VERSION_3) {
			readTmp = new byte[4];
			readLen = inputStream.read(readTmp, 0, 4);
			int requestSeq = HNumberUtils.intFromByteArray(readTmp);
			header.setReqSeq(requestSeq);

			int payloadLen = bytes.length - LivePacketHeader.HEADER_LENGTH_VER2;

			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, LivePacketHeader.HEADER_LENGTH_VER2, payload, 0, payloadLen);
			}
		} else
			throw new IllegalArgumentException(
					String.format("Live packet version incorrect , get %d", header.getVersion()));
		LivePacket packet = new LivePacket();
		packet.setHeader(header);
		packet.setData(payload);
		packet.setRelayMode(false);
		if (payload != null) {
			packet.setDataSize(payload.length);
		}

		String result;
		switch (nMsgCode) {
		case MsgCodeDef.MSGCODE_PingRequest54171:
			result = "receive_MSGCODE_PingRequest54171";
			break;

		case MsgCodeDef.MSGCODE_PingResponse54172:
			result = "receive_MSGCODE_PingResponse54172";
			break;

		case MsgCodeDef.MSGCODE_ServerAuthOK54180:
			result = "receive_MSGCODE_AuthOK54180";
			break;

		case MsgCodeDef.MSGCODE_ClientDataRequest54181:
			result = "receive_MSGCODE_ClientDataRequest54181";
			break;

		case MsgCodeDef.MSGCODE_ServerHBRequest54186:
			result = "receive_MSGCODE_ServerHBReq54185";
			break;

		case MsgCodeDef.MSGCODE_ClientHBReqResponse54187:
			result = "receive_MSGCODE_ClientHBReqResponse54187";
			break;

		case MsgCodeDef.MSGCODE_ServerDataResponse54188:
			result = "receive_MSGCODE_ServerDataResponse54188";
			break;

		case MsgCodeDef.MSGCODE_ClientDataResendRequest54189:
			result = "receive_MSGCODE_ClientDataResendReq54189";
			break;

		case MsgCodeDef.MSGCODE_ServerDataResendResponse54190:
			result = "receive_MSGCODE_ServerDataResendRes54190";
			break;

		case MsgCodeDef.MSGCODE_ClientFinishRequest54183:
			result = "receive_MSGCODE_ClientFinishRequest54183";
			break;

		case MsgCodeDef.MSGCODE_ClientHeartbeatRequest54193:
			result = "receive_MSGCODE_ClientHeartbeatRequest54193";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeDisconnected62250:
			result = "receive_MSGCODE_ServerErrCodeDisconnected62250";
			break;

		case MsgCodeDef.MSGCODE_ClientAuthRequest57343:
			result = "receive_MSGCODE_ClientAuthRequest57343";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeGetDataFailed62256:
			result = "receive_MSGCODE_ServerErrCodeGetDataFailed62256";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeAuthExpired62258:
			result = "receive_MSGCODE_ServerErrCodeAuthExpired62258";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeResourceNotFound62262:
			result = "receive_MSGCODE_ServerErrCodeResourceNotFound62262";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeStreamDiscontinueServer62264:
			result = "receive_MSGCODE_ServerErrCodeStreamDiscontinueServer62264";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeUnauthorised62260:
			result = "receive_MSGCODE_ServerErrCodeUnauthorised62260";
			break;

		case MsgCodeDef.MSGCODE_ServerErrCodeSessionExpired62266:
			result = "receive_ServerErrCodeSessionExpired62266";
			break;

		default:
			result = "receive_MSGCODE_" + nMsgCode;
		}
		if (performance != null)
			performance.end("parseLivePacket", result, bytes.length);
		return packet;
	}

	private static LivePacket parseRelayLivePacket(ByteArrayInputStream inputStream, byte[] bytes) {
		RelayLivePacketHeader header = new RelayLivePacketHeader();

		header.setRelayProtocol(RelayLivePacketHeader.PROTOCOL_RELAY);

		int readLen = -1;
		byte[] readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int sourceIP = HNumberUtils.intFromByteArray(readTmp);
		header.setSourceIP(sourceIP);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 2);
		int sourcePort = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setSourcePort(sourcePort);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 2);
		int relayPort = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setRelayPort(relayPort);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 2);
		int checksum = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setChecksum(checksum);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int protocolVer = HNumberUtils.intFromByteArray(readTmp);
		header.setProtocol(protocolVer);

		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);
		short version = HNumberUtils.shortFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setVersion(version);

		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);
		short encryptType = HNumberUtils.shortFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setEncryptType(encryptType);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int nMsgCode = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nMsgCode:" + nMsgCode);
		header.setMsgCode(nMsgCode);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int m_field_8 = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nChannelId:" + nChannelId);
		header.setChannelId(m_field_8);

		readTmp = new byte[8];
		readLen = inputStream.read(readTmp, 0, 8);
		long gap1 = HNumberUtils.longFromByteArray(readTmp);
		// System.out.println("gap1:" + gap1);
		header.setSessionKey1(gap1);

		readTmp = new byte[8];
		readLen = inputStream.read(readTmp, 0, 8);
		long gap2 = HNumberUtils.longFromByteArray(readTmp);
		// System.out.println("gap2:" + gap2);
		header.setSessionKey2(gap2);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int m_dataSize = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("field_28:" + field_28);
		header.setGroupBaseSeq(m_dataSize);

		// readTmp = new byte[4];
		// readLen = inputStream.read(readTmp, 0, 4);
		// int m_field_32 = HNumberUtils.intFromByteArray(readTmp);
		// //System.out.println("f32:" + f32);
		// header.setF32(m_field_32);

		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);
		short field_36 = HNumberUtils.shortFromByteArray(readTmp);
		// System.out.println("field_36:" + field_36);
		header.setGroupBufferCount(field_36);

		// 4个byte对齐，这两个byte忽略
		readTmp = new byte[2];
		readLen = inputStream.read(readTmp, 0, 2);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int field_40 = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("f40:" + f40);
		header.setGroupSeq(field_40);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int field_44 = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("f44:" + f44);
		header.setBackup1(field_44);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		int m_PacketTotalCount = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nCount:" + nCount);
		header.setCount(m_PacketTotalCount);

		readTmp = new byte[4];
		readLen = inputStream.read(readTmp, 0, 4);
		// if(readLen != 4){
		// System.out.println("nPacketSeq bytes len " + readLen);
		// }
		int m_nPacketSeq = HNumberUtils.intFromByteArray(readTmp);
		// System.out.println("nSeq:" + nSeq);
		header.setSeq(m_nPacketSeq);

		byte[] payload = null;
		if (header.getVersion() == LivePacketHeader.VERSION_1) {
			int payloadLen = bytes.length - RelayLivePacketHeader.HEADER_LENGTH_VER1;
			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, RelayLivePacketHeader.HEADER_LENGTH_VER1, payload, 0, payloadLen);
			}
		} else if (header.getVersion() == LivePacketHeader.VERSION_2 ||
				header.getVersion() == LivePacketHeader.VERSION_3) {

			readTmp = new byte[4];
			readLen = inputStream.read(readTmp, 0, 4);
			int requestSeq = HNumberUtils.intFromByteArray(readTmp);
			header.setReqSeq(requestSeq);

			int payloadLen = bytes.length - RelayLivePacketHeader.HEADER_LENGTH_VER2;

			if (payloadLen > 0) {
				payload = new byte[payloadLen];
				System.arraycopy(bytes, RelayLivePacketHeader.HEADER_LENGTH_VER2, payload, 0, payloadLen);
			}
		} else
			throw new IllegalArgumentException(
					String.format("Live packet version incorrect , get %d", header.getVersion()));

		LivePacket packet = new LivePacket();
		packet.setHeader(header);
		packet.setData(payload);
		packet.setRelayMode(true);
		if (payload != null) {
			packet.setDataSize(payload.length);
		}

		return packet;
	}

	public static byte[] serializeLivePacket(LivePacket packet) {
		ByteArrayOutputStream stream = new ByteArrayOutputStream();

		LivePacketHeader header = packet.getHeader();
		byte[] readTmp = HNumberUtils.toByteArray(header.getProtocol());
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getVersion());
		stream.write(readTmp, 0, 2);

		readTmp = HNumberUtils.toByteArray(header.getEncryptType());
		stream.write(readTmp, 0, 2);

		readTmp = HNumberUtils.toByteArray(header.getMsgCode());
		// System.out.println("nMsgCode:" + nMsgCode);
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getChannelId());
		// System.out.println("nChannelId:" + nChannelId);
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getSessionKey1());
		stream.write(readTmp, 0, 8);

		readTmp = HNumberUtils.toByteArray(header.getSessionKey2());
		stream.write(readTmp, 0, 8);

		readTmp = HNumberUtils.toByteArray(header.getGroupBaseSeq());
		stream.write(readTmp, 0, 4);

		// readTmp = HNumberUtils.toByteArray(header.getF32());
		// //System.out.println("f32:" + f32);
		// stream.write(readTmp, 0, 4);

		// short类型，也当成int处理，空出的2个字节填0。 4个byte对齐，这两个byte忽略
		readTmp = HNumberUtils.toByteArray((int)header.getGroupBufferCount());
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getGroupSeq());
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getBackup1());
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getCount());
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getSeq());
		stream.write(readTmp, 0, 4);

		readTmp = HNumberUtils.toByteArray(header.getReqSeq());
		stream.write(readTmp, 0, 4);

		if (packet.getData() != null) {
			stream.write(packet.getData(), 0, packet.getData().length);
		}
		byte[] streamBytes = stream.toByteArray();
		return streamBytes;
	}

	public static byte[] serializeLivePacketEx(LivePacket packet) {
		boolean ver1 = packet.getHeader().getVersion() == LivePacketHeader.VERSION_1;
		int packetLen = ver1 ? LivePacketHeader.HEADER_LENGTH_VER1 : LivePacketHeader.HEADER_LENGTH_VER2;
		if (packet.getData() != null) {
			packetLen += packet.getData().length;
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(packetLen).order(ByteOrder.LITTLE_ENDIAN);
		// ByteArrayOutputStream stream = new ByteArrayOutputStream(packetLen);

		LivePacketHeader header = packet.getHeader();
		// byte[] readTmp = HNumberUtils.toByteArray(header.getProtocol());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getProtocol());

		// readTmp = HNumberUtils.toByteArray(header.getVersion());
		// stream.write(readTmp, 0, 2);
		byteBuffer.putShort(header.getVersion());

		// readTmp = HNumberUtils.toByteArray(header.getEncryptType());
		// stream.write(readTmp, 0, 2);
		byteBuffer.putShort(header.getEncryptType());

		// readTmp = HNumberUtils.toByteArray(header.getMsgCode());
		// System.out.println("nMsgCode:" + nMsgCode);
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getMsgCode());

		// readTmp = HNumberUtils.toByteArray(header.getChannelId());
		// System.out.println("nChannelId:" + nChannelId);
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getChannelId());

		// readTmp = HNumberUtils.toByteArray(header.getSessionKey1());
		// stream.write(readTmp, 0, 8);
		byteBuffer.putLong(header.getSessionKey1());

		// readTmp = HNumberUtils.toByteArray(header.getSessionKey2());
		// stream.write(readTmp, 0, 8);
		byteBuffer.putLong(header.getSessionKey2());

		// readTmp = HNumberUtils.toByteArray(header.getGroupBaseSeq());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getGroupBaseSeq());

		// readTmp = HNumberUtils.toByteArray(header.getF32());
		// //System.out.println("f32:" + f32);
		// stream.write(readTmp, 0, 4);

		// short类型，也当成int处理，空出的2个字节填0。 4个byte对齐，这两个byte忽略
		// readTmp = HNumberUtils.toByteArray(header.getGroupBufferCount());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getGroupBufferCount());

		// readTmp = HNumberUtils.toByteArray(header.getGroupSeq());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getGroupSeq());

		// readTmp = HNumberUtils.toByteArray(header.getBackup1());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getBackup1());

		// readTmp = HNumberUtils.toByteArray(header.getCount());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getCount());

		// readTmp = HNumberUtils.toByteArray(header.getSeq());
		// stream.write(readTmp, 0, 4);
		byteBuffer.putInt(header.getSeq());

		if (!ver1)
			byteBuffer.putInt(header.getReqSeq());

		if (packet.getData() != null) {
			// stream.write(packet.getSlices(), 0, packet.getSlices().length);
			byteBuffer.put(packet.getData());
		}
		byte[] streamBytes = byteBuffer.array();
		return streamBytes;
	}

	public static byte[] serializeLivePacketHeader(LivePacket packet, ByteBuffer byteBuffer) {
		LivePacketHeader header = packet.getHeader();
		byteBuffer.position(0);

		byteBuffer.putInt(header.getProtocol());

		byteBuffer.putShort(header.getVersion());

		byteBuffer.putShort(header.getEncryptType());

		byteBuffer.putInt(header.getMsgCode());

		byteBuffer.putInt(header.getChannelId());

		byteBuffer.putLong(header.getSessionKey1());

		byteBuffer.putLong(header.getSessionKey2());

		byteBuffer.putInt(header.getGroupBaseSeq());

		byteBuffer.putInt(header.getGroupBufferCount());

		byteBuffer.putInt(header.getGroupSeq());

		byteBuffer.putInt(header.getBackup1());

		byteBuffer.putInt(header.getCount());

		byteBuffer.putInt(header.getSeq());

		if (header.getVersion() != LivePacketHeader.VERSION_1)
			byteBuffer.putInt(header.getReqSeq());

		byte[] streamBytes = byteBuffer.array();
		return streamBytes;
	}

	public static ByteBuffer buildByteBuffer(byte[] data, short protocolVersion) {
		int headerLen = protocolVersion == LivePacketHeader.VERSION_1 ? LivePacketHeader.HEADER_LENGTH_VER1
				: LivePacketHeader.HEADER_LENGTH_VER2;
		int packetLen = headerLen;
		if (data != null) {
			packetLen += data.length;
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(packetLen).order(ByteOrder.LITTLE_ENDIAN);

		if (data != null) {
			byteBuffer.position(headerLen);
			byteBuffer.put(data);
		}
		return byteBuffer;
	}
}
