package com.stream.brt.engine;

//import com.runjva.sourceforge.jsocks.protocol.Socks5DatagramSocket;
//import com.runjva.sourceforge.jsocks.protocol.Socks5Proxy;
//import com.runjva.sourceforge.jsocks.protocol.SocksException;

import com.stream.brt.engine.model.BrtUrlInfo;
import com.stream.brt.prot.LivePacket;
import com.stream.brt.prot.MsgCodeDef;
import com.stream.brt.prot.ProtocolBuilder;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import java.io.IOException;
import java.net.*;
import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 */
public class UDPSocket {
    private static Logger logger = LoggerFactory.getLogger(UDPSocket.class.getSimpleName());
    //private String m_ServerAddress;
    //private String m_LocalAddress;

    //private int m_nPort;
    private List<InetSocketAddress> remoteSocketAddresses;

    private long m_nReceivedPacketCount;
    private long m_nDroppedPacketCount;
    private long m_nSentPacketCount;
    private long m_PortUseBeginTime;
    //boolean m_bIsAvailable;
    //private int soTimeout = 500;
    private DatagramSocket socket;
    private ArrayBlockingQueue<DatagramPacket> readBufferQ;
    private LinkedBlockingQueue<LivePacket> sendQueue;
    private int bufferQueueSize;
    private boolean stoppedFlag = false;
    //private Performance performance;
    private AtomicInteger requestSeq;
    private Map<InetSocketAddress, AtomicInteger> receiveFromServerCntMap = new HashMap<>();
    private Map<InetSocketAddress, Integer> unreceiveHosts = new HashMap<>();
    private int unreceivedHostIndex = 0;

    public UDPSocket(List<BrtUrlInfo.ServerInfo> serverInfoList, String localAddress, int bufferQueueSize, Performance performance) throws IOException {
        this(serverInfoList, localAddress, bufferQueueSize, 1048576, performance); // 1048576 = 1024 * 024
    }

    public UDPSocket(List<BrtUrlInfo.ServerInfo> serverInfoList, String localAddress, int bufferQueueSize, int receiveBufferSize, Performance performance) throws IOException {
//        this.m_nPort = nRemotePort;
//        this.m_ServerAddress = remoteHost;
//        this.m_LocalAddress = localAddress;
        //this.performance = performance;
        //remoteSocketAddress = new InetSocketAddress(InetAddress.getByName(m_ServerAddress), nRemotePort);
        remoteSocketAddresses = new ArrayList<>();
        for (BrtUrlInfo.ServerInfo serverInfo : serverInfoList) {
            int port = serverInfo.port <= 0 ? BrtEngineConfig.serverPort : serverInfo.port;
            InetSocketAddress address = new InetSocketAddress(InetAddress.getByName(serverInfo.host), port);
            remoteSocketAddresses.add(address);
            unreceiveHosts.put(address, octetsToInt(address.getAddress().getAddress()));
        }

        if (localAddress != null) {
            //specific local ip and any local port
            InetSocketAddress address = new InetSocketAddress(localAddress, 0);
            socket = new DatagramSocket(address);
        } else {
            socket = new DatagramSocket();
        }
        socket.setSoTimeout(BrtEngineConfig.brtSoTimeout);
        //1mb
        this.bufferQueueSize = bufferQueueSize;
        readBufferQ = new ArrayBlockingQueue<>(bufferQueueSize);
        sendQueue = new LinkedBlockingQueue<>(bufferQueueSize);
        if (receiveBufferSize > 0) {
            socket.setReceiveBufferSize(receiveBufferSize);
            logger.info("UDP Socket Set ReceiveBufferSize %s, get size %s", receiveBufferSize, socket.getReceiveBufferSize());
        } else {
            logger.info("UDP Socket Default ReceiveBufferSize %s", socket.getReceiveBufferSize());
        }
        m_PortUseBeginTime = System.currentTimeMillis();

        Thread t = new Thread(new Runnable() {
            @Override
            public void run() {
                receiveLoop();
            }
        });
        t.setPriority(Thread.MAX_PRIORITY);
        t.start();

//        Thread s = new Thread(new Runnable() {
//            @Override
//            public void run() {
//                sendLoop();
//            }
//        });
//        s.setPriority(Thread.MAX_PRIORITY);
//        s.start();

//        if (logger.isDebug())
//            logger.debug("Init socket: serverAddress[%s:%s] localAddress[%s]", remoteHost, nRemotePort, localAddress);
    }

    public void setRequestSeq(int requestSeq) {
        this.requestSeq = new AtomicInteger(requestSeq);
    }

    public InetAddress getRemoteAddress() {
        return socket.getInetAddress();
    }

    public int getRemotePort() {
        return socket.getPort();
    }

    public int getReceiveWaterline() {
        return (int) (readBufferQ.size() * 100.0 / bufferQueueSize);
    }

    public int getSendWaterline() {
        return (int) (sendQueue.size() * 100.0 / bufferQueueSize);
    }

    public int getReceiveFromServerCnt() {
        return receiveFromServerCntMap.size();
    }

    public Collection<InetSocketAddress> getUnreceiveHosts() {
        return unreceiveHosts.keySet();
    }

    public List<InetSocketAddress> getReceivedHosts() {
        //return new ArrayList<>(receiveFromServerCntMap.keySet());
        return null;
    }

    public int getNextUnreceiveHost() {
        if (unreceiveHosts.isEmpty()) {
            return 0;
        }

        int index = unreceivedHostIndex;
        if (index >= remoteSocketAddresses.size())
            index = 0;

        int ret = 0;
        while (index < remoteSocketAddresses.size()) {
            InetSocketAddress address = remoteSocketAddresses.get(index);
            Integer host = unreceiveHosts.get(address);
            if (host != null) {
                ret = host;
                break;
            }

            index++;
        }

        unreceivedHostIndex = index;
        return ret;
    }

    public static final int octetsToInt(byte[] octets) {
        return octetsToInt(octets, 0);
    }

    public static final int octetsToInt(byte[] octets, int offset) {
        return (((octets[offset] & 0xff) << 24) |
                ((octets[offset + 1] & 0xff) << 16) |
                ((octets[offset + 2] & 0xff) << 8) |
                (octets[offset + 3] & 0xff));
    }

    public void send(LivePacket packet) throws IOException {
        //long begin = System.currentTimeMillis();
        if (remoteSocketAddresses.isEmpty()) {
            throw new IOException("UDP Socket server mode doesn't support send LivePacket directly");
        }

//        if (sendQueue.remainingCapacity() == 0) {
//            logger.warn("UDP socket send queue is full %s, drop this packet", sendQueue.size());
//            return;
//        }
//        sendQueue.offer(packet);

        asyncSend(packet);
        //long dur = System.currentTimeMillis() - begin;
        //logger.info("UDP send dur %s ms", dur);
    }

//    public void send(DatagramPacket datagramPacket) throws IOException {
//        socket.send(datagramPacket);
//    }

    //    public DatagramPacket receive() throws IOException {
//        byte[] buf = new byte[LivePacket.MAX_PACKET_LENGTH];
//        DatagramPacket receivePacket = new DatagramPacket(buf, buf.length, remoteSocketAddress);
//        try {
//            socket.receive(receivePacket);
//            return receivePacket;
//        } catch (SocketTimeoutException socketTimeoutException) {
//            return null;
//        } catch (SocketException se) {
//            throw se;
//        }
//    }
    public DatagramPacket receive() throws IOException {
        if (stoppedFlag) {
            throw new IOException("UDP Socket closed");
        }
        try {
            DatagramPacket packet = readBufferQ.poll(BrtEngineConfig.brtSoTimeout, TimeUnit.MILLISECONDS);
            return packet;
        } catch (InterruptedException e) {
            throw new IOException(e);
        }
    }

    protected DatagramPacket doReceive(byte[] receiveBuffer) throws IOException {
        DatagramPacket receivePacket = new DatagramPacket(receiveBuffer, receiveBuffer.length);
        try {
            //performance.begin("receive", Performance.Level.IMPORTANT);
            socket.receive(receivePacket);
//            if (receivePacket.getLength() == 0)
//                performance.end("receive", "receive_0");
//            else
//                performance.end("receive", "receive_OK", receivePacket.getLength());
            return receivePacket;
        } catch (SocketTimeoutException socketTimeoutException) {
            //performance.end("receive", Performance.Level.IMPORTANT,"receive_timeout");
            return null;
        } catch (SocketException se) {
            //performance.end("receive", Performance.Level.IMPORTANT, "receive_"+se.getMessage());
            throw se;
        }
    }

    protected void receiveLoop() {
        while (!stoppedFlag) {
            byte[] buf = new byte[LivePacket.MAX_CLIENT_PACKET_LENGTH_VER3];
            try {
                DatagramPacket packet = doReceive(buf);
                if (packet != null) {
                    /*InetSocketAddress address = (InetSocketAddress) packet.getSocketAddress();
                    AtomicInteger cnt = receiveFromServerCntMap.get(address);
                    if (cnt != null) {
                        cnt.incrementAndGet();
                    } else {
                        cnt = new AtomicInteger(1);
                        receiveFromServerCntMap.put(address, cnt);
                        unreceiveHosts.remove(address);
                    }*/

                    if (readBufferQ.remainingCapacity() == 0) {
                        //logger.warn("UDP socket readBufferQ is full %s, drop incoming packets", readBufferQ.size());
                        //m_nDroppedPacketCount++;
                        continue;
                    }
                    readBufferQ.offer(packet);
                    //m_nReceivedPacketCount++;
                } else {
                    try {
                        Thread.sleep(1);
                    } catch (Exception e) {
                    }
                }
            } catch (SocketException e) {
                //java.net.SocketException: Socket closed
                String msg = StringUtils.trimToEmpty(e.getMessage());
                if (msg.contains("Socket closed") || msg.contains("Socket is closed")) {
                    stoppedFlag = true;
                    break;
                }
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
                stoppedFlag = true;
                break;
            }
        }
        //stoppedFlag = true;
    }

    protected void sendLoop() {
        while (!stoppedFlag) {
            try {
                LivePacket packet = sendQueue.poll(50, TimeUnit.MILLISECONDS);
                if (packet != null) {
                    asyncSend(packet);
                }
            } catch (IOException e) {
                logger.error(e, e.getMessage());
                break;
            } catch (InterruptedException e) {
                break;
            }
        }
        stoppedFlag = true;
    }

    protected void asyncSend(LivePacket packet) throws IOException {
        //todo 这里竟然拖慢速度inetSocketAddress.getHostName很慢
        if (logger.isDebug()) {
            logger.debug("Client send data: %s",
                    ToStringBuilder.reflectionToString(packet.getHeader(), ToStringStyle.JSON_STYLE));
        }

        if (requestSeq != null) {
            packet.getHeader().setReqSeq(requestSeq.incrementAndGet());
        }

        String task;
        switch (packet.getHeader().getMsgCode()) {
            case MsgCodeDef.MSGCODE_PingRequest54171:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_PingRequest54171-1" : "send_MSGCODE_PingRequest54171";
                break;

            case MsgCodeDef.MSGCODE_PingResponse54172:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_PingResponse54172-1" : "send_MSGCODE_PingResponse54172";
                break;

            case MsgCodeDef.MSGCODE_ServerAuthOK54180:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerAuthOK54180-1" : "send_MSGCODE_ServerAuthOK54180";
                break;

            case MsgCodeDef.MSGCODE_ServerHBRequest54186:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerHBRequest54186-1" : "send_MSGCODE_ServerHBRequest54186";
                break;

            case MsgCodeDef.MSGCODE_ClientHBReqResponse54187:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ClientHBReqResponse54187-1" : "send_MSGCODE_ClientHBReqResponse54187";
                break;

            case MsgCodeDef.MSGCODE_ServerDataResponse54188:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerDataResponse54188-1" : "send_MSGCODE_ServerDataResponse54188";
                break;

            case MsgCodeDef.MSGCODE_ClientDataResendRequest54189:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ClientDataResendRequest54189-1" : "send_MSGCODE_ClientDataResendRequest54189";
                break;

            case MsgCodeDef.MSGCODE_ServerDataResendResponse54190:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerDataResendResponse54190-1" : "send_MSGCODE_ServerDataResendResponse54190";
                break;

            case MsgCodeDef.MSGCODE_ClientFinishRequest54183:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ClientFinishRequest54183-1" : "send_MSGCODE_ClientFinishRequest54183";
                break;

            case MsgCodeDef.MSGCODE_ClientHeartbeatRequest54193:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ClientHeartbeatRequest54193-1" : "send_MSGCODE_ClientHeartbeatRequest54193";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeDisconnected62250:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeDisconnected62250-1" : "send_MSGCODE_ServerErrCodeDisconnected62250";
                break;

            case MsgCodeDef.MSGCODE_ClientAuthRequest57343:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ClientAuthRequest57343-1" : "send_MSGCODE_ClientAuthRequest57343";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeGetDataFailed62256:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeGetDataFailed62256-1" : "send_MSGCODE_ServerErrCodeGetDataFailed62256";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeAuthExpired62258:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeAuthExpired62258-1" : "send_MSGCODE_ServerErrCodeAuthExpired62258";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeResourceNotFound62262:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeResourceNotFound62262-1" : "send_MSGCODE_ServerErrCodeResourceNotFound62262";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeStreamDiscontinueServer62264:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeStreamDiscontinueServer62264-1" : "send_MSGCODE_ServerErrCodeStreamDiscontinueServer62264";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeUnauthorised62260:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeUnauthorised62260-1" : "send_MSGCODE_ServerErrCodeUnauthorised62260";
                break;

            case MsgCodeDef.MSGCODE_ServerErrCodeSessionExpired62266:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_ServerErrCodeSessionExpired62266-1" : "send_MSGCODE_ServerErrCodeSessionExpired62266";
                break;

            default:
                task = packet.getHeader().getGroupDataSeq() == -1 ? "send_MSGCODE_" + packet.getHeader().getMsgCode() + "-1" : "send_MSGCODE_" + packet.getHeader().getMsgCode();
        }

        //performance.begin("send");
        //performance.begin(task);
        byte[] streamBytes = ProtocolBuilder.serializeLivePacket(packet);
        for (InetSocketAddress remoteSocketAddress : remoteSocketAddresses) {
            DatagramPacket sendPacket = new DatagramPacket(streamBytes, streamBytes.length, remoteSocketAddress);
            socket.send(sendPacket);
        }
        //performance.end("send", 0L, streamBytes.length);
        //performance.end(task, 0L, streamBytes.length);
    }

    public void close() {
        stoppedFlag = true;
        if (socket != null) {
            socket.close();
        }
    }
}
