package com.stream.brt.engine.model;

import com.backblaze.erasure.ReedSolomon;
import com.stream.brt.engine.ResendStat;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.*;

import static com.stream.brt.prot.RSConstant.BYTES_IN_INT;
import static com.stream.brt.prot.RSConstant.DATA_SHARDS;
import static com.stream.brt.prot.RSConstant.PARITY_SHARDS;

/**
 *
 */
public class LiveSliceGroup {
    private static Logger logger = LoggerFactory.getLogger(LiveSliceGroup.class.getSimpleName());

    public enum Status {
        DOWNLOADING,
        COMPLETED,
        READ,
        DROPPED
    }

    private int groupBaseSeq;
    private int count;
    private int f32;
    private short groupBaseCount;
    private int exSeq;
    private int f44;
    private long gap1;
    private long gap2;

    private long beginTime;
    private long decodeBeginTime;
    private long decodeEndTime;
    private long sendBeginTime;
    private long sendEndTime;
    private long lastCheckTime;
    private long duration;
    private int lostCount;
    private int firstLostCount;
    private int resendRounds;
    private int resendCount;
    private int lostPercent;
    private int firstLostPercent;
    private int rtt;

    private int subGroupSeq;
    private int subGroupDataShards;
    private int subGroupParityShards;
    private int subGroupShards;
    private int statSeq;

    private LiveSlice[] slices;
    final private TreeMap<Integer, LiveSliceSubGroup> subGroupTreeMap;
    private int completedCount;
    private Set<Integer> uncompletedSet;
    //初始数据，是否需要加快resend速度
    private boolean initStatus = false;

    private Status status = Status.DOWNLOADING;
    private String connectionId;

    private int totalShards = 0;
    private Set<Integer> uncompletedSubGroups = new HashSet<>();
    private ReedSolomon reedSolomon;

    private boolean isSlicePushMode = false;
    private int maxContinuousSeq = -1;
    private int currConsumedSeq = -1;

    public LiveSliceGroup(int count) {
        this(count, true, DATA_SHARDS, PARITY_SHARDS, false);
    }

    public LiveSliceGroup(int count, boolean preCalcLeft, int subGroupDataShards, int subGroupParityShards, boolean isSlicePushMode) {
        this.count = count;
        if (preCalcLeft) {
            uncompletedSet = new HashSet<>(count);
            for (int i = 0; i < count; i++) {
                uncompletedSet.add(i);
            }
        }

        this.subGroupDataShards = subGroupDataShards;
        this.subGroupParityShards = subGroupParityShards;
        subGroupShards = subGroupDataShards + subGroupParityShards;
        subGroupTreeMap = new TreeMap<>();

        slices = new LiveSlice[count];
        // without rs
        if (subGroupDataShards > 0) {
//            slices = new LiveSlice[count];
            reedSolomon = ReedSolomon.create(subGroupDataShards, subGroupParityShards);
        }

        this.beginTime = System.currentTimeMillis();
        this.isSlicePushMode = isSlicePushMode;
        if (subGroupDataShards > 0) this.isSlicePushMode = false;
    }

    public int getGroupBaseSeq() {
        return groupBaseSeq;
    }

    public void setGroupBaseSeq(int groupBaseSeq) {
        this.groupBaseSeq = groupBaseSeq;
    }

    public int getCount() {
        return count;
    }

    public long getBeginTime() {
        return beginTime;
    }

    public void setBeginTime(long beginTime) {
        this.beginTime = beginTime;
    }

    public long getLastCheckTime() {
        return lastCheckTime;
    }

    public void setLastCheckTime(long lastCheckTime) {
        this.lastCheckTime = lastCheckTime;
    }

    public long getDuration() {
        return duration;
    }

    public boolean addSlice(LiveSlice slice) {
        if (slice.seq >= slices.length) {
            return false;
        }
        if (slices[slice.seq] != null) {
            return false;
        }
        slices[slice.seq] = slice;

        if (subGroupDataShards == 0) {
            completedCount++;
            if (uncompletedSet != null) {
                uncompletedSet.remove(slice.seq);
            }
            if (isSlicePushMode && maxContinuousSeq < count - 1) {
                while (slices[maxContinuousSeq + 1] != null) {
                    maxContinuousSeq++;
                    if (maxContinuousSeq == count - 1)
                        break;
                }
            }
            if (isCompleted()) {
                duration = System.currentTimeMillis() - beginTime;
                if (logger.isDebug()) {
                    logger.debug("Completed group[%s] rec dur[%s]ms", getExSeq(), getDuration());
                }
            }
            return true;
        } else {
            LiveSliceSubGroup subGroup = subGroupTreeMap.get(slice.subGroupSeq);
            if (subGroup == null) {
                int subGroupBaseSeq = slice.subGroupSeq * subGroupShards;
                subGroup = new LiveSliceSubGroup(subGroupBaseSeq, slice.subGroupDataShards, slice.subGroupParityShards);
                subGroup.setSugGroupSeq(slice.subGroupSeq);
                subGroup.setDataShards(slice.subGroupDataShards);
                subGroup.setParityShards(slice.subGroupParityShards);
                totalShards += subGroupShards;
                uncompletedSubGroups.add(slice.subGroupSeq);
                subGroupTreeMap.put(slice.subGroupSeq, subGroup);
            }

            boolean ret = subGroup.addSlice(slice);
            if (ret) {
                completedCount++;
                if (subGroup.isComplete()) {
                    uncompletedSubGroups.remove(slice.subGroupSeq);
                }
                uncompletedSet.remove(slice.seq);
            }

            return ret;
        }
    }

    public int getAvailableCount() {
        return completedCount;
    }

    public boolean isCompleted() {
        if (subGroupDataShards == 0) {
            return completedCount == count;
        }

        if (totalShards < count) {
            return false;
        }

        return uncompletedSubGroups.size() <= 0;
    }

    public boolean isDropped() {
        return status == Status.DROPPED;
    }

    public void detail(String tag) {
        StringBuilder builder = new StringBuilder();
        for (LiveSliceSubGroup subGroup : getSubGroups()) {
            builder.append("subGroup[").append(subGroup.getSugGroupSeq()).append("]: slices[").append(subGroup.getSlices().length).append("], ");
            builder.append(" isComplete[").append(subGroup.isComplete()).append("], time[").append(System.currentTimeMillis() - subGroup.getBeginTime()).append("ms], ");
            builder.append("dataShards[").append(subGroup.getDataShards()).append("], parityShards[").append(subGroup.getParityShards()).append("], ");

            LiveSlice slices[] = subGroup.getSlices();
            int dataShards = subGroup.getDataShards();

            List<Integer> missingData = new ArrayList<>();
            List<Integer> missingParity = new ArrayList<>();
            for (int i = 0; i < subGroupShards; i++) {
                if (slices[i] == null) {
                    if (i < dataShards)
                        missingData.add(i);
                    else
                        missingParity.add(i);
                }
            }

            builder.append("Miss Data[").append(missingData.size()).append("], ");
            builder.append("Parity[").append(missingParity.size()).append("], ");
            builder.append("[ ").append(missingData.size() + missingParity.size()).append("/").append(subGroupShards);
            builder.append(" (").append((int) (100.0 * (missingData.size() + missingParity.size()) / subGroupShards)).append("%)]\n");
        }

        logger.error("%s [grp#%d]: %d/%d, resendCnt[%d], isComplete[%s], uncompleted:%d/%d, detail:\n%s\nuncompleteSlices[%d]:\n", tag, exSeq,
                subGroupShards, count, lostCount, isCompleted(), uncompletedSubGroups.size(), subGroupTreeMap.size(), builder.toString(), uncompletedSet.size());
    }

    public LiveSlice[] getSliceTreeMap() {
        return slices;
    }

    public TreeMap<Integer, LiveSliceSubGroup> getSubGroupTreeMap() {
        return subGroupTreeMap;
    }

    public void readSliceTreeMap(String mediaCode, OutputStream outputStream) throws IOException {
        if (subGroupDataShards == 0) {
            sendBeginTime = System.currentTimeMillis();
            for (LiveSlice slice : slices) {
                byte[] sliceData = slice.data;
                if (sliceData != null) {
                    outputStream.write(sliceData);
                } else {
                    logger.error("[%s] Error slice without data.group:%s, seq:%s", mediaCode, slice.groupSeq, slice.seq);
                }
            }
            sendEndTime = System.currentTimeMillis();
        } else {
            readWithRSSliceTreeMap(outputStream);
        }
    }

    public int writeSlice(OutputStream outputStream) throws IOException {
        if (!isSlicePushMode) {
            logger.error("writeSlice only available in slice push mode!");
            throw new IOException("writeSlice only available in slice push mode!");
        }
        if (subGroupDataShards > 0) {
            logger.error("writeSlice only work without fec!");
            throw new IOException("writeSlice only work without fec!");
        }
        int tmpMaxContinuousSeq = maxContinuousSeq;
        int tmpCurrConsumedSeq = currConsumedSeq + 1;
        if (tmpMaxContinuousSeq > currConsumedSeq) {
            for (int i = tmpCurrConsumedSeq; i <= tmpMaxContinuousSeq; i++) {
                byte[] sliceData = slices[i].data;
                if (sliceData != null) {
                    outputStream.write(sliceData);
                }
            }
            int tmpCount = tmpMaxContinuousSeq - tmpCurrConsumedSeq + 1;
            currConsumedSeq = tmpMaxContinuousSeq;
            logger.info("brt-engine, push group[%d] all[%d-%d] slice[%d,%d-%d]", exSeq, count, currConsumedSeq + 1, tmpCount, tmpCurrConsumedSeq, tmpMaxContinuousSeq);
            return tmpCount;
        } else {
            // logger.info("brt-engine, push group[%d] no available slice, all[%d] info[%d-%d]", exSeq, count, tmpCurrConsumedSeq, tmpMaxContinuousSeq);
            return 0;
        }
    }

    public int getF32() {
        return f32;
    }

    public void setF32(int f32) {
        this.f32 = f32;
    }

    public short getGroupBaseCount() {
        return groupBaseCount;
    }

    public void setGroupBaseCount(short groupBaseCount) {
        this.groupBaseCount = groupBaseCount;
    }

    public int getExSeq() {
        return exSeq;
    }

    public void setExSeq(int exSeq) {
        this.exSeq = exSeq;
    }

    public int getF44() {
        return f44;
    }

    public void setF44(int f44) {
        this.f44 = f44;
    }

    public long getGap1() {
        return gap1;
    }

    public void setGap1(long gap1) {
        this.gap1 = gap1;
    }

    public long getGap2() {
        return gap2;
    }

    public void setGap2(long gap2) {
        this.gap2 = gap2;
    }

    public int getLostCount() {
        return lostCount;
    }

    public void setLostCount(int lostCount) {
        this.lostCount = lostCount;
    }

    public Status getStatus() {
        return status;
    }

    public void setStatus(Status status) {
        this.status = status;
    }

    public String getConnectionId() {
        return connectionId;
    }

    public void setConnectionId(String connectionId) {
        this.connectionId = connectionId;
    }

    public Set<Integer> getUncompletedSet() {
        return uncompletedSet;
    }

    public boolean isInitStatus() {
        return initStatus;
    }

    public void setInitStatus(boolean initStatus) {
        this.initStatus = initStatus;
    }

    public int getSubGroupSeq() {
        return subGroupSeq;
    }

    public void setSubGroupSeq(int subGroupSeq) {
        this.subGroupSeq = subGroupSeq;
    }

    public int getSubGroupDataShards() {
        return subGroupDataShards;
    }

    public void setSubGroupDataShards(int subGroupDataShards) {
        this.subGroupDataShards = subGroupDataShards;
    }

    public int getSubGroupParityShards() {
        return subGroupParityShards;
    }

    public int getSubGroupShards() {
        return subGroupShards;
    }

    public void setSubGroupParityShards(int subGroupParityShards) {
        this.subGroupParityShards = subGroupParityShards;
    }

    private List<LiveSliceSubGroup> getSubGroups() {
        synchronized (subGroupTreeMap) {
            return new ArrayList<>(subGroupTreeMap.values());
        }
    }

    public long getDecodeBeginTime() {
        return decodeBeginTime;
    }

    public void setDecodeBeginTime(long decodeBeginTime) {
        this.decodeBeginTime = decodeBeginTime;
    }

    public long getDecodeEndTime() {
        return decodeEndTime;
    }

    public void setDecodeEndTime(long decodeEndTime) {
        this.decodeEndTime = decodeEndTime;
    }

    public long getSendBeginTime() {
        return sendBeginTime;
    }

    public void setSendBeginTime(long sendBeginTime) {
        this.sendBeginTime = sendBeginTime;
    }

    public long getSendEndTime() {
        return sendEndTime;
    }

    public void setSendEndTime(long sendEndTime) {
        this.sendEndTime = sendEndTime;
    }

    public int getLostPercent() {
        return lostPercent;
    }

    public void setLostPercent(int lostPercent) {
        this.lostPercent = lostPercent;
    }

    public int getRtt() {
        return rtt;
    }

    public void setRtt(int rtt) {
        this.rtt = rtt;
    }

    public int getStatSeq() {
        return statSeq;
    }

    public void setStatSeq(int statSeq) {
        this.statSeq = statSeq;
    }

    public int getResendRounds() {
        return resendRounds;
    }

    public void setResendRounds(int resendRounds) {
        this.resendRounds = resendRounds;
    }

    public int getFirstLostCount() {
        return firstLostCount;
    }

    public void setFirstLostCount(int firstLostCount) {
        this.firstLostCount = firstLostCount;
    }

    public int getFirstLostPercent() {
        return firstLostPercent;
    }

    public void setFirstLostPercent(int firstLostPercent) {
        this.firstLostPercent = firstLostPercent;
    }

    public int getResendCount() {
        return resendCount;
    }

    public void setResendCount(int resendCount) {
        this.resendCount = resendCount;
    }

    private void readWithRSSliceTreeMap(OutputStream outputStream) throws IOException {
        int realDataSize = 0;
        int writtenBytes = 0;
        int dataShards = 0;
        int parityShards = 0;
        int shardsDataSize = 0;
        int shardsParitySize = 0;
        int paddingBytes = 0;
        long startTime = 0;
        long decodeTime = 0;
        long sendTime = 0;
        int lostCnt = 0;

        decodeBeginTime = System.currentTimeMillis();
        //StringBuilder builder = new StringBuilder();
        for (LiveSliceSubGroup subGroup : getSubGroups()) {
            LiveSlice slices[] = subGroup.getSlices();
            byte[][] shards = new byte[subGroupShards][];
            boolean[] shardPresent = new boolean[subGroupShards];
            int shardSize = 0;
            int subGrpRealDataSize = 0;
            int subGrpWrittenBytes = 0;
            int subGrpShardsDataSize = 0;
            int subGrpShardsParitySize = 0;
            int subGrpPaddingBytes = 0;
            long subGrpDecodeTime = 0;

            long decodeStartTime = System.currentTimeMillis();
            for (int i = 0; i < subGroupShards; i++) {
                if (slices[i] != null) {
                    shards[i] = slices[i].data;
                    if (shardSize == 0)
                        shardSize = slices[i].data.length;
                    shardPresent[i] = true;
                }
            }

            for (int i = 0; i < subGroupShards; i++) {
                if (!shardPresent[i]) {
                    shards[i] = new byte[shardSize];
                    lostCnt++;
                }
            }

            reedSolomon.decodeMissing(shards, shardPresent, 0, shardSize);
            subGrpDecodeTime = System.currentTimeMillis() - decodeStartTime;

            if (startTime == 0) {
                startTime = System.currentTimeMillis() - decodeBeginTime;
            }
            subGrpShardsDataSize = shardSize * subGroupDataShards;
            subGrpShardsParitySize = shardSize * subGroupParityShards;

            long subGrpSendStartTime = System.currentTimeMillis();
            for (int i = 0; i < subGroupDataShards; i++) {
                byte[] data = shards[i];
                if (subGrpRealDataSize == 0) {
                    byte[] temp = new byte[BYTES_IN_INT];
                    System.arraycopy(data, 0, temp, 0, BYTES_IN_INT);
                    subGrpRealDataSize = ByteBuffer.wrap(temp).getInt();
                    if (subGrpRealDataSize <= 0 || subGrpRealDataSize > subGrpShardsDataSize - 4) {
                        throw new IndexOutOfBoundsException();
                    }

                    subGrpWrittenBytes = data.length - temp.length;
                    outputStream.write(data, temp.length, subGrpWrittenBytes);
                } else {
                    if (subGrpWrittenBytes + data.length <= subGrpRealDataSize) {
                        outputStream.write(data);
                        subGrpWrittenBytes += data.length;
                    } else if (subGrpWrittenBytes < subGrpRealDataSize) {
                        int len = subGrpRealDataSize - subGrpWrittenBytes;
                        outputStream.write(data, 0, len);
                        subGrpWrittenBytes += len;
                        subGrpPaddingBytes += (data.length - len);
                    } else {
                        subGrpPaddingBytes += data.length;
                    }
                }
            }
            sendTime += (System.currentTimeMillis() - subGrpSendStartTime);

//            builder.append(String.format("\ngroup[%d], subGrp[%d], decodeTime[%d], shardSize[%d], size[%d], shardsDataSize[%d], shardsParitySize[%d], realDataSize[%d], writtenBytes[%d], paddingBytes[%d]",
//                    exSeq, subGroup.getSugGroupSeq(), subGrpDecodeTime, shardSize, subGrpShardsDataSize + subGrpShardsParitySize,
//                    subGrpShardsDataSize , subGrpShardsParitySize,
//                    subGrpRealDataSize, subGrpWrittenBytes, subGrpPaddingBytes));

            realDataSize += subGrpRealDataSize;
            writtenBytes += subGrpWrittenBytes;
            dataShards += subGroupDataShards;
            parityShards += subGroupParityShards;
            shardsDataSize += subGrpShardsDataSize;
            shardsParitySize += subGrpShardsParitySize;
            paddingBytes += subGrpPaddingBytes;
            decodeTime += subGrpDecodeTime;
        }

        decodeEndTime = decodeBeginTime + decodeTime;
        sendBeginTime = decodeEndTime;
        sendEndTime = sendBeginTime + sendTime;

        this.lostCount = lostCnt;
        this.lostPercent = 100 * lostCnt / (dataShards + parityShards);
        if (this.firstLostCount == 0) {
            this.firstLostCount = lostCount;
        }
        if (this.firstLostPercent == 0) {
            this.firstLostPercent = lostPercent;
        }
        this.rtt = ResendStat.getRtt();

//        builder.append(String.format("\ngroup[%d], startTime[%d], decodeTime[%d], dataShards[%d], parityShards[%d], size[%d], shardsDataSize[%d], shardsParitySize[%d], realDataSize[%d], writtenBytes[%d], paddingBytes[%d]",
//                exSeq, startTime, decodeTime, dataShards, parityShards, shardsDataSize + shardsParitySize,
//                shardsDataSize , shardsParitySize,
//                realDataSize, writtenBytes, paddingBytes));
//
//        logger.error(builder.toString());
    }
}
