package com.stream.brt.engine;

import com.stream.brt.engine.model.LiveSlice;
import com.stream.brt.engine.model.LiveSliceGroup;
import com.stream.brt.prot.LivePacketHeader;

import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;

/**
 *
 */
public class BrtConnectChannelData {
    private int channelId;
    private String mediaCode;
    private TreeMap<Integer, LiveSliceGroup> groupTreeMap;
    private long startTime;
    private long lastUpdate;
    private int initGroupCount;
    private int keepOverdueSize;
    private TreeSet<Integer> overdueGroups;

    public BrtConnectChannelData(int keepOverdueSize) {
        groupTreeMap = new TreeMap<>();
        overdueGroups = new TreeSet<>();
        this.keepOverdueSize = keepOverdueSize;
    }

    public boolean addGroup(LiveSliceGroup group) {
        if (groupTreeMap.containsKey(group.getExSeq())) {
            return false;
        }
        groupTreeMap.put(group.getExSeq(), group);
        return true;
    }

    public boolean removeGroup(int groupSeq) {
        if (!groupTreeMap.containsKey(groupSeq)) {
            return false;
        }
        groupTreeMap.remove(groupSeq);
        if (overdueGroups.size() >= keepOverdueSize)
            overdueGroups.pollFirst();
        overdueGroups.add(groupSeq);
        return true;
    }

    public boolean addSlice(final LiveSlice slice, final LivePacketHeader packetHeader, String cnnId, Performance performance, boolean isSlicePushMode) {
        if (overdueGroups.contains(slice.groupSeq))
            return false;
        LiveSliceGroup group = groupTreeMap.get(slice.groupSeq);
        if (group == null) {
            group = new LiveSliceGroup(slice.groupCount, true, slice.subGroupDataShards, slice.subGroupParityShards, isSlicePushMode);
            group.setExSeq(slice.groupSeq);
            group.setGroupBaseSeq(packetHeader.getGroupBaseSeq());
            //group.setGroupDataCount(slice.groupCount);
            //group.setF32(packetHeader.getF32());
            group.setGroupBaseCount(packetHeader.getGroupBufferCount());
            group.setF44(packetHeader.getBackup1());
            group.setGap1(packetHeader.getSessionKey1());
            group.setGap2(packetHeader.getSessionKey2());
            long now = System.currentTimeMillis();
            group.setBeginTime(now);
            group.setLastCheckTime(now);
            group.setConnectionId(cnnId);
            group.setStatus(LiveSliceGroup.Status.DOWNLOADING);
            group.setStatSeq(slice.statSeq);
            boolean initStatus = initGroupCount < 2;
            if (initStatus) {
                group.setInitStatus(true);
                initGroupCount++;
            }
            groupTreeMap.put(slice.groupSeq, group);
            performance.begin("LiveSliceGroup", slice.groupSeq, Performance.Level.IMPORTANT);
        }

        boolean added = group.addSlice(slice);
        if (added) {
            //设置最后更新时间
            group.setLastCheckTime(System.currentTimeMillis());
        }
        return added;
    }

    public Collection<LiveSliceGroup> getGroupList() {
        if (groupTreeMap.isEmpty()) {
            return Collections.emptyList();
        }
        return groupTreeMap.values();
    }

    public LiveSliceGroup getGroup(int groupSeq) {
        return groupTreeMap.get(groupSeq);
    }

    public int getGroupCount() {
        return groupTreeMap.size();
    }

    public LiveSliceGroup getFirstGroup() {
        if (groupTreeMap.size() == 0) {
            return null;
        }
        return groupTreeMap.firstEntry().getValue();
    }

    public int getFirstGroupSeq() {
        if (groupTreeMap.size() == 0) {
            return -1;
        }
        return groupTreeMap.firstKey();
    }

    public int getChannelId() {
        return channelId;
    }

    public void setChannelId(int channelId) {
        this.channelId = channelId;
    }

    public String getMediaCode() {
        return mediaCode;
    }

    public void setMediaCode(String mediaCode) {
        this.mediaCode = mediaCode;
    }

    public long getStartTime() {
        return startTime;
    }

    public void setStartTime(long startTime) {
        this.startTime = startTime;
    }

    public long getLastUpdate() {
        return lastUpdate;
    }

    public void setLastUpdate(long lastUpdate) {
        this.lastUpdate = lastUpdate;
    }

}
