package com.stream.brt.engine;

import com.stream.brt.engine.base.BaseEngine;
import com.stream.brt.engine.base.BaseEngineListener;
import com.stream.brt.engine.metric.BrtEngineMetric;
import com.stream.brt.engine.metric.BrtMetricStorage;
import com.stream.brt.engine.metric.BrtMetricStorageDefault;
import com.stream.brt.engine.model.LiveSliceGroup;
import com.stream.brt.prot.LivePacketHeader;
import com.stream.brt.tool.io.PipelineOutputStream;
import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;

import java.io.*;
import java.util.Map;

/**
 *
 */
public class BrtEngine extends BaseEngine implements ValueProvider.RefreshListener {
    private final static Logger logger = LoggerFactory.getLogger(BrtEngine.class.getSimpleName());
    private BaseEngineListener engineListener;
    private BrtEngineChannelData channelData;

    private BrtConnect connect;
    private String mediaCode;
    private Performance performance;
    private BrtEngineMetric engineMetric;
    private boolean isFirstData = true;
    private long lastWriteClientTime = 0;

    public void initModule(String did, final String brtUrl, String platform, String appVer, BaseEngineListener playerReader, ValueProvider provider) throws Exception {
        initModule(did, brtUrl, platform, appVer, playerReader, provider, new BrtMetricStorageDefault());
    }

    //brt://10.9.8.7:21000,10.9.8.88:21000/live1/foxtv1?token=xxyyzz&time=1500000&valid=60
    @Override
    public void initModule(String did, final String brtUrl, String platform, String appVer, BaseEngineListener playerReader, ValueProvider provider, BrtMetricStorage brtMetricStorage) throws Exception {
        //todo test
        //private int completedGroupCount = 0;
        logger.info("brt-engine, %s initModule()", BrtEngine.class.getSimpleName());
        int statSeq = BrtStat.getSeq();
        BrtStat.put(statSeq, "initBrt");

        provider.addRefreshListener(this);
        initCfg(provider);
        // performance stat init
        performance = new Performance(BrtEngineConfig.isPerformanceStat);
        engineMetric = new BrtEngineMetric(BrtEngineConfig.isMetricOn, brtMetricStorage);
        this.engineListener = playerReader;
        this.channelData = new BrtEngineChannelData();
        this.connect = new BrtConnect("1", provider, statSeq);
        this.connect.initModule(did, brtUrl, platform, appVer, new ClientConnectListener(), performance, engineMetric);
        this.mediaCode = this.connect.getMediaCode();

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    runPlayerTask();
                } catch (Exception e) {
                    logger.error(e, "[%s] Error in player run()", mediaCode);
                }
            }
        }).start();
    }

    public void initCfg(ValueProvider provider) {
        //Engine config begin
        BrtEngineConfig.serverTimeout =
                provider.val(BrtEngineConfig.SERVER_TIMEOUT, BrtEngineConfig.DFT_SERVER_TIMEOUT);

        BrtEngineConfig.receiveGroupTimeout =
                provider.val(BrtEngineConfig.RECEIVE_GROUP_TIMEOUT, BrtEngineConfig.DFT_RECEIVE_GROUP_TIMEOUT);

        BrtEngineConfig.loginTimeout =
                provider.val(BrtEngineConfig.LOGIN_TIMEOUT, BrtEngineConfig.DFT_LOGIN_TIMEOUT);

        if (BrtEngineConfig.loginTimeout <= BrtEngineConfig.DFT_AUTH_MAX_RTT)
            BrtEngineConfig.loginTimeout = BrtEngineConfig.DFT_LOGIN_TIMEOUT;

        BrtEngineConfig.pingTimeout =
                provider.val(BrtEngineConfig.PING_TIMEOUT, BrtEngineConfig.DFT_PING_TIMEOUT);

        BrtEngineConfig.pingInterval =
                provider.val(BrtEngineConfig.PING_INTERVAL, BrtEngineConfig.DFT_PING_INTERVAL);

        BrtEngineConfig.groupResendDuration =
                provider.val(BrtEngineConfig.GROUP_RESEND_DURATION, BrtEngineConfig.DFT_GROUP_RESEND_DURATION);
        if (BrtEngineConfig.groupResendDuration <= 100)
            BrtEngineConfig.groupResendDuration = BrtEngineConfig.DFT_GROUP_RESEND_DURATION;

        BrtEngineConfig.isPerformanceStat = provider.val(BrtEngineConfig.IS_PERFORMANCE_STAT, false);

        BrtEngineConfig.performanceStatDur = provider.val(BrtEngineConfig.PERFORMANCE_STAT_DURATION, 600_000);
        if (BrtEngineConfig.performanceStatDur < 1000) {
            BrtEngineConfig.performanceStatDur = 0;
            BrtEngineConfig.isPerformanceStat = false;
        }

        BrtEngineConfig.receiveUdpTimeout =
                provider.val(BrtEngineConfig.RECEIVE_UDP_TIMEOUT, BrtEngineConfig.DFT_RECEIVE_GROUP_TIMEOUT);
        if (BrtEngineConfig.receiveUdpTimeout < 5000)
            BrtEngineConfig.receiveUdpTimeout = 19_000;

        BrtEngineConfig.groupCacheCount =
                provider.val(BrtEngineConfig.GROUP_CACHE_COUNT, BrtEngineConfig.DFT_GROUP_CACHE_CNT);

        BrtEngineConfig.serverPort =
                provider.val(BrtEngineConfig.SERVER_PORT, BrtEngineConfig.DFT_SERVER_PORT);

        BrtEngineConfig.resendOneByOneRatio =
                provider.val(BrtEngineConfig.RESEND_ONE_BY_ONE_RATIO, BrtEngineConfig.DFT_RESEND_ONE_BY_ONE_RATIO);

        BrtEngineConfig.groupChkResendDuration =
                provider.val(BrtEngineConfig.GROUP_CHK_RESEND_DURATION, BrtEngineConfig.DFT_GROUP_CHK_RESEND_DURATION);

        BrtEngineConfig.maxRTO =
                provider.val(BrtEngineConfig.MAX_RTO, BrtEngineConfig.DFT_MAX_RTO);

        BrtEngineConfig.minRTO =
                provider.val(BrtEngineConfig.MIN_RTO, BrtEngineConfig.DFT_MIN_RTO);

        BrtEngineConfig.rtoMultiFactor =
                provider.val(BrtEngineConfig.RTO_MULTI_FACTOR, BrtEngineConfig.DFT_RTO_MULTI_FACTOR);

        BrtEngineConfig.authMaxRTT =
                provider.val(BrtEngineConfig.AUTH_MAX_RTT, BrtEngineConfig.DFT_AUTH_MAX_RTT);

        if (BrtEngineConfig.authMaxRTT < 500 || BrtEngineConfig.authMaxRTT >= BrtEngineConfig.DFT_LOGIN_TIMEOUT)
            BrtEngineConfig.authMaxRTT = BrtEngineConfig.DFT_AUTH_MAX_RTT;

        BrtEngineConfig.maxThrottleSendBwLevel =
                provider.val(BrtEngineConfig.MAX_THROTTLE_SEND_BW_LEVEL, BrtEngineConfig.DFT_MAX_THROTTLE_SEND_BW_LEVEL);

        BrtEngineConfig.minThrottleSendBwLevel =
                provider.val(BrtEngineConfig.MIN_THROTTLE_SEND_BW_LEVEL, BrtEngineConfig.DFT_MIN_THROTTLE_SEND_BW_LEVEL);

        BrtEngineConfig.initRtoMultiFactor =
                provider.val(BrtEngineConfig.INIT_RTO_MULTI_FACTOR, BrtEngineConfig.DFT_INIT_RTO_MULTI_FACTOR);

        BrtEngineConfig.brtCheckResendStrategy =
                provider.val(BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY, BrtEngineConfig.DFT_BRT_CHECK_RESEND_STRATEGY);

        BrtEngineConfig.brtResendStrategy =
                provider.val(BrtEngineConfig.BRT_RESEND_STRATEGY, BrtEngineConfig.DFT_BRT_RESEND_STRATEGY);

        BrtEngineConfig.brtThrottleDuration =
                provider.val(BrtEngineConfig.THROTTLE_DURATION, BrtEngineConfig.DFT_THROTTLE_DURATION);
        if (BrtEngineConfig.brtThrottleDuration <= 0)
            BrtEngineConfig.brtThrottleDuration = BrtEngineConfig.DFT_THROTTLE_DURATION;

        BrtEngineConfig.brtDropTest =
                provider.val(BrtEngineConfig.BRT_DROP_TEST, BrtEngineConfig.DFT_BRT_DROP_TEST);

        BrtEngineConfig.brtDropRatio =
                provider.val(BrtEngineConfig.BRT_DROP_RATIO, BrtEngineConfig.DFT_BRT_DROP_RATIO);
        if (BrtEngineConfig.brtDropRatio < 0) {
            BrtEngineConfig.brtDropRatio = 0;
        } else if (BrtEngineConfig.brtDropRatio > 100) {
            BrtEngineConfig.brtDropRatio = 100;
        }

        if (BrtEngineConfig.brtDropRatio == 0) {
            BrtEngineConfig.brtDropTest = false;
        }

        BrtEngineConfig.brtResendStrategyGroupCnt =
                provider.val(BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY_GROUP_CNT, BrtEngineConfig.DFT_BRT_CHECK_RESEND_STRATEGY_GROUP_CNT);

        BrtEngineConfig.brtSelServerStrategy =
                provider.val(BrtEngineConfig.BRT_SELECT_SERVER_STRATEGY, BrtEngineConfig.DFT_BRT_SELECT_SERVER_STRATEGY);

        // 如果配置了短key，则用短key的配置，通常在brt url中配置
        BrtEngineConfig.brtSelServerStrategy =
                provider.val(BrtEngineConfig.BRT_SELECT_SERVER_STRATEGY_SHORT, BrtEngineConfig.brtSelServerStrategy);

        BrtEngineConfig.brtSoTimeout =
                provider.val(BrtEngineConfig.BRT_SO_TIMEOUT, BrtEngineConfig.DFT_BRT_SO_TIMEOUT);

        BrtEngineConfig.isMetricOn = provider.val(BrtEngineConfig.BRT_METRIC_STAT, true);

        BrtEngineConfig.brtKeepOverdueSize = provider.val(BrtEngineConfig.BRT_KEEP_OVERDUE_SIZE, BrtEngineConfig.DFT_KEEP_OVER_DUE_SIZE);

        BrtEngineConfig.brtHBInterval = provider.val(BrtEngineConfig.BRT_HB_INTERVAL, BrtEngineConfig.DFT_BRT_HB_INTERVAL);

        BrtEngineConfig.brtVer = provider.val(BrtEngineConfig.BRT_VER, BrtEngineConfig.DFT_BRT_VER);

        BrtEngineConfig.brtDRInterval = provider.val(BrtEngineConfig.BRT_DR_INTERVAL, BrtEngineConfig.DFT_BRT_DR_INTERVAL);

        BrtEngineConfig.dataPushMode = provider.val(BrtEngineConfig.DATA_PUSH_MODE, BrtEngineConfig.DFT_DATA_PUSH_MODE);

        if (BrtEngineConfig.brtVer == LivePacketHeader.VERSION_3) {
            BrtEngineConfig.dataPushMode = BrtEngineConfig.DATA_PUSH_MODE_BLOCK;
        }

        //Engine config end
    }

    @Override
    public void shutdown() {
        if (this.connect != null) {
            this.connect.shutdown();
        }
        performance.stop();
    }

    @Override
    public boolean isRunning() {
        if (this.connect != null) {
            return this.connect.isConnected();
        }
        return false;
    }

    @Override
    public void onAppEvent(int event, Map<String, Object> info) {

    }

    // 从LiveModule的本地缓冲读取数据，通过TcpClientSocket将数据发送给Video Player
    private void runPlayerTask() throws IOException {
        //logger.info("[%s] Started runPlayerTask()", this.mediaCode, playerReceiveMode);
        //1024k 临时缓冲
        PipelineOutputStream outputStream = new PipelineOutputStream(1024 * 1024);
        final InputStream playerInputStream = outputStream.getInputStream();

        Runnable onDataReceivedAction = new Runnable() {
            @Override
            public void run() {
                engineListener.onDataReceiveStart(playerInputStream);
            }
        };

        boolean pushBySlice = (BrtEngineConfig.dataPushMode == BrtEngineConfig.DATA_PUSH_MODE_SLICE);
        long sleepTime = 5;
        if (pushBySlice) {
            sleepTime = 100;
        }

        while (true) {
            //判断系统状态，决定是否退出
            if (!connect.isConnected()) {
                break;
            }
            try {
                while (connect.isConnected()) {
                    boolean gotData = false;
                    if (pushBySlice) {
                        gotData = processAvailableSliceData2(outputStream, onDataReceivedAction);
                    } else {
                        gotData = processAvailableSliceData(outputStream, onDataReceivedAction);
                    }
                    //如果上一次读取到了数据，则再次尝试读取下一个，否则等一会再尝试
                    if (!gotData) {
                        break;
                    }
                }
                //TODO 测试代码
                // 默认设置是200ms，这里为了方便调试，设置为500
                Thread.sleep(sleepTime);
            } catch (Exception e) {
                if (e instanceof IOException && e.getMessage().equalsIgnoreCase("Pipe closed")) {
                    logger.info("[%s] Stopped runPlayerTask() by engine closed", this.mediaCode);
                } else {
                    logger.error(e, "[%s] Error in runPlayerTask for channel", this.mediaCode);
                }
            }
        }
        try {
            outputStream.close();
        } catch (Exception e) {
            logger.error(e, "Error in close playerOutputStream, ignore it");
        }

        logger.info("[%s] Ended runPlayerTask()", this.mediaCode);
    }

    private boolean processAvailableSliceData(OutputStream outputStream, Runnable dataReceivedAction) throws IOException, InterruptedException {
        LiveSliceGroup firstGroup = channelData.getFirstGroup();
        //todo c++代码没有找到超时逻辑。如果一个group长时间没有数据，则可以跳过
        if (firstGroup == null) {
            //第一个数据group是null
//            if (logger.isDebug())
//                logger.debug("[%s] Player: check top group is null. Ignore it", this.mediaCode);
            return false;
        }
        if (logger.isDebug())
            logger.debug("[%s] Player: check top group. group[%s], count %s, completed[%s]",
                    this.mediaCode, firstGroup.getExSeq(), firstGroup.getCount(), firstGroup.isCompleted());

        //删除已经写完的group
        if (!firstGroup.isCompleted()) {
            return false;
        }

        // only for local test
        //lost(firstGroup);

        firstGroup.setRtt(ResendStat.getRtt());
        // 数据已经到达，回调调用者，做好数据接收准备
        if (isFirstData && dataReceivedAction != null) {
            isFirstData = false;
            dataReceivedAction.run();
            logger.info("brt-engine, dataReceivedAction run");
        }

        // todo:往output stream写数据, 线程可能会被挂住, output stream的write方法一直不返回，导致播放器卡住不播放，数据一直从缓冲区溢出丢弃
        try {
            logger.info("brt-engine, push data to player-[%s]", firstGroup.getExSeq());
            firstGroup.readSliceTreeMap(mediaCode, outputStream);
            //completedGroupCount++;
        } catch (Exception e) {
            logger.error("processAvailableSliceData [%s] Error in flushing data group", this.mediaCode);
        }
        BrtStat.put(firstGroup, connect.getResendStat(), BrtConnect.RemoveGrpEvent.COMPLETE);

        channelData.removeGroup(firstGroup);
        lastWriteClientTime = System.currentTimeMillis();
        return true;
    }

    private boolean processAvailableSliceData2(OutputStream outputStream, Runnable dataReceivedAction) {
        LiveSliceGroup firstGroup = channelData.getFirstGroup2();
        if (firstGroup == null) {
            return false;
        } else {
            // 数据已经到达，回调调用者，做好数据接收准备
            if (isFirstData && dataReceivedAction != null) {
                isFirstData = false;
                dataReceivedAction.run();
                logger.info("brt-engine, dataReceivedAction run");
            }
        }

        // todo:往output stream写数据, 线程可能会被挂住, output stream的write方法一直不返回，导致播放器卡住不播放，数据一直从缓冲区溢出丢弃
        try {
            int writeSliceCount = firstGroup.writeSlice(outputStream);
            return writeSliceCount > 0;
        } catch (Exception e) {
            logger.error("processAvailableSliceData2 [%s] Error in flushing data group", this.mediaCode);
        } finally {
            //删除已经写完的group
            if (firstGroup.isCompleted()) {
                firstGroup.setRtt(ResendStat.getRtt());
                logger.info("brt-engine, block completed[%d]", firstGroup.getExSeq());
                BrtStat.put(firstGroup, connect.getResendStat(), BrtConnect.RemoveGrpEvent.COMPLETE);
                channelData.removeGroup(firstGroup);
            }
            lastWriteClientTime = System.currentTimeMillis();
        }
        return true;
    }

// only for local test
//    private int lostCnt = -1;
//    private void lost(LiveSliceGroup group) {
//        for (Map.Entry<Integer, LiveSliceSubGroup> entry : group.getSubGroupTreeMap().entrySet()) {
//            LiveSliceSubGroup subGroup = entry.getValue();
//            Set<Integer> lost = new HashSet<>();
//            LiveSlice slices[] = subGroup.getSlices();
//            if (lostCnt < 0) {
//                lostCnt = group.getSubGroupParityShards();
//            }
//
//            while (lost.size() < lostCnt) {
//                int i = RandomUtils.nextInt(0, slices.length);
//                if (lost.contains(i))
//                    continue;
//
//                slices[i] = null;
//                lost.add(i);
//            }
//        }
//        lostCnt--;
//    }

    private void handleGroupCacheOverflow() {
        //保存数据成功
        //如果缓存超限，则删除旧数据。因为这里是BrtConnect同步逐个packet处理，所以最多只会多出一个数据
        int curGroupCount = channelData.getGroupCount();
        performance.begin("GroupCache", Performance.Level.IMPORTANT);
        if (curGroupCount > BrtEngineConfig.groupCacheCount) {
            performance.end("GroupCache", Performance.Level.IMPORTANT, "GroupCache.Overflow", curGroupCount);
            engineListener.onDataOverflow();

            do {
                LiveSliceGroup sliceGroup = channelData.getFirstGroup();
                if (sliceGroup == null)
                    break;

                channelData.removeGroup(sliceGroup);
                BrtStat.put(sliceGroup, connect.getResendStat(), BrtConnect.RemoveGrpEvent.DROP_OVERCACHE);
                curGroupCount = channelData.getGroupCount();
                if (logger.isInfo())
                    logger.info("[%s] Current group count[%s] is morn than limit %s, drop older group seq[%s], lastWriteClient:%dms ago",
                            mediaCode, curGroupCount, BrtEngineConfig.groupCacheCount, sliceGroup.getExSeq(), System.currentTimeMillis() - lastWriteClientTime);
            } while (curGroupCount > BrtEngineConfig.groupCacheCount);
        } else {
            performance.end("GroupCache", "GroupCache.normal", curGroupCount);
        }
    }

    @Override
    public void onValueRefresh(ValueProvider provider) {
        initCfg(provider);
    }

    private class ClientConnectListener implements BrtConnectListener {

        @Override
        public void onStart() {
            engineListener.onStart();
        }

        @Override
        public void onAuthPassed(Map info) {
            isFirstData = true;
            engineListener.onAuthPassed();
        }

        @Override
        public void onAuthFailed(int code, String msg) {
            engineListener.onAuthFailed(code, msg);
        }

        @Override
        public void onDataBegin(LiveSliceGroup sliceGroup) {
            boolean added = channelData.addGroup(sliceGroup);
            if (!added) {
                if (logger.isDebug())
                    logger.debug("[%s] duplicated slice group %s, ignore it", mediaCode, sliceGroup.getExSeq());
            }
        }

        @Override
        public void onDataCompleted(LiveSliceGroup sliceGroup) {
            handleGroupCacheOverflow();
        }

        @Override
        public void onDataDropped(LiveSliceGroup sliceGroup, String msg) {
            channelData.removeGroup(sliceGroup);
        }

        @Override
        public void onInfo(int code, String msg, Map info) {

        }

        @Override
        public void onDisconnected(boolean serverReason, int lastMsgCode, int innerErrCode) {
            engineListener.onStop(serverReason, lastMsgCode, innerErrCode);
        }

        @Override
        public void onDataReady(int chanid, int port) {
            engineListener.onDataReady(chanid, port);
        }
    }
}