package com.stream.brt.engine;


import com.stream.brt.engine.metric.BrtEngineMetric;
import com.stream.brt.engine.model.*;
import com.stream.brt.prot.*;
import com.stream.brt.tool.DESCipherUtils;
import com.stream.brt.tool.HConstants;
import com.stream.brt.tool.MD5CustomUtils;
import com.stream.brt.tool.log.Logger;
import com.stream.brt.tool.log.LoggerFactory;

import org.apache.commons.io.Charsets;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetSocketAddress;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static com.stream.brt.engine.BrtConstant.LIVE_MODULE_VERSION;

/**
 *
 */
public class BrtConnect implements UdpServerProvider.Ping {
    private static Logger logger = LoggerFactory.getLogger(BrtConnect.class.getSimpleName());

    public static enum RemoveGrpEvent {
        COMPLETE,
        DROP_TIMEOUT,
        DROP_OVERDUE,
        DROP_OVERCACHE
    }

    private static final int SOCKET_RECEIVE_BATCH_PACKET_COUNT = 30000;

    //数据模式：收取数据
    private static final int DATA_MODE_PULL = 1;
    //数据模式：推送数据
    private static final int DATA_MODE_PUSH = 2;

    private String mediaCode;
    private String token;
    private int serverTime;
    private int validMinutes;
    private String did;
    private String platform;
    private String appVer;

    //订阅数据计算方式：
    //数据默认被分为32组，对应int(4bytes)的32二进制位
    //最右位有值，代表余数0，右1位代表余数1，右2位代表余数2，以此类推，右31位代表余数31
    //这样的方式，可以让一个订阅者同时订阅多组

    //订阅数据的分组值,默认订阅所有
    private int subDataValue = 0xffffffff;

    private List<UDPSocket> udpSockets;
    private int currentUDPSocketIndex;

    private LivePacket currentMsgPacket;
    private int currentChannelId;
    private boolean m_bLiveServerDisconnected;
    private long m_nReceivedDataSize;
    private boolean m_bLivePacketStopped;
    private long m_LastServerActiveTime;
    private ConcurrentMap<Integer, BrtConnectChannelData> liveChannelDataMap;

    private long m_LastCheckDroppedPacketTime;
    private long m_LastProcessStatAndHBTime;
    private long m_LastProcessPacketDataTime;
    private int m_nPacket54190Received_Count;
    private int m_nSentPacketCount54189;
    private boolean m_bIsAvailable = true;
    private boolean m_bClientRequestStop = false;
    private boolean m_bServerTimeoutStop = false;
    private int m_LastMsgCode = 0;

    private BrtConnectListener connectListener;
    private int dataMode = DATA_MODE_PULL;

    private String localIp;

    private String connectionId;
    //private CRC crcTool;

    private int requestSeq = -1;

    private UdpServerProvider serverProvider;
    private Performance performance;
    private BrtEngineMetric brtEngineMetric;
    private ResendStat resendStat;
    private ValueProvider provider;
    private int statSeq;
    private long sessionKey1;
    private long sessionKey2;
    private short sev = 0;

    private int lastErrorCode = LiveInnerErrorCodeDef.INNER_CODE_UNKNOWN;

    private boolean isFirstSlice = true;
    private boolean isSlicePushMode = false;

    public BrtConnect(String connectionId, ValueProvider provider, int statSeq) {
        this.connectionId = connectionId;
        this.provider = provider;
        this.statSeq = statSeq;
    }

    /**
     * BRT格式
     * time:服务器时间，精度到秒，java time stamp/1000
     * valid:有效期，单位分钟
     * token:签名，签名原文: did#t15000000#v60
     * brt://10.9.8.7:21000/live1/foxtv1?token=xxyyzz&time=1500000&valid=60
     */
    public void initModule(String did, String brtUrl, String platform, String appVer, BrtConnectListener connectListener, Performance performance, BrtEngineMetric engineMetric) throws Exception {
        this.performance = performance;
        this.brtEngineMetric = engineMetric;
        serverProvider = new DefaultUdpServerProvider(brtUrl, this, performance);
        BrtUrlInfo brtUrlInfo = serverProvider.getBrtUrlInfo();
        String scheme = brtUrlInfo.getScheme();
        if (brtUrlInfo.getServerInfos().size() < 1) {
            throw new Exception("Invalid server number " + brtUrlInfo.getServerInfos().size());
        }
        String mediaCode = brtUrlInfo.getPath();

        Map<String, String> paramMap = brtUrlInfo.getParamMap();
        String token = paramMap.get("token");
        if (token == null) {
            //为了兼容nimble的sign的命名
            token = paramMap.get("wmsAuthSign");
        }
        String tokenInProvider = provider.val("eng_token", "");
        if (StringUtils.isNotEmpty(tokenInProvider)) {
            token = tokenInProvider;
        }
        int serverTime = 0;
        String strServerTime = paramMap.get("time");
        if (strServerTime != null) {
            serverTime = Integer.parseInt(StringUtils.trimToNull(strServerTime));
        }
        int validMinutes = 120;
        String strValidMinutes = paramMap.get("valid");
        if (strValidMinutes != null) {
            validMinutes = Integer.parseInt(StringUtils.trimToNull(strValidMinutes));
        }

        performance.setMediaCode(mediaCode);

        // add parameters in url to engine cfg
        provider.merge(paramMap);

        initModule(serverProvider, did, token, serverTime, validMinutes, mediaCode, platform, appVer, connectListener);
    }

    //init with any local ip address
    public void initModule(UdpServerProvider provider,
                           String did, String token, int serverTime, int validMinutes, String mediaCode, String platform,
                           String appVer, BrtConnectListener connectListener) throws IOException {
        initModule(provider, did, token, serverTime, validMinutes, mediaCode, platform, appVer, connectListener, null);
    }

    //init with specific local ip address
    public void initModule(final UdpServerProvider provider,
                           String did, String token, int serverTime, int validMinutes, String mediaCode, String platform,
                           String appVer, BrtConnectListener connectListener, String localIp) throws IOException {
        this.serverProvider = provider;
        this.mediaCode = mediaCode;
        this.token = token;
        this.did = did;
        this.serverTime = serverTime;
        this.validMinutes = validMinutes;
        this.platform = platform;
        this.connectListener = connectListener;
        this.localIp = localIp;
        this.appVer = appVer;

        this.sev = this.provider.val(BrtEngineConfig.BRT_KEY_SEV, this.sev);

        udpSockets = new ArrayList<>();
        currentUDPSocketIndex = 0;

        //crcTool = new CRC(CRC.Parameters.CRC16);

        liveChannelDataMap = new ConcurrentHashMap<>();
        this.isSlicePushMode = (BrtEngineConfig.dataPushMode == BrtEngineConfig.DATA_PUSH_MODE_SLICE);

        resendStat = new ResendStat(mediaCode, brtEngineMetric);
        new Thread(new Runnable() {
            @Override
            public void run() {
                Run();
            }
        }).start();
    }

    public String getMediaCode() {
        return mediaCode;
    }

    public boolean isConnected() {
        return m_bIsAvailable;
    }

    @Override
    public long ping(List<BrtUrlInfo.ServerInfo> serverInfos) throws Exception {
        UDPPingSocket socket = null;
        try {
            socket = createPingSocket(serverInfos);
            LivePacket resPacket = ping(socket);
            if (resPacket == null)
                return 0;

            return 112; //56B*2
        } catch (Exception e) {
            throw e;
        } finally {
            if (socket != null)
                socket.close();
        }
    }

    private UDPPingSocket createPingSocket(List<BrtUrlInfo.ServerInfo> serverInfos) throws IOException {
        //int port = serverInfo.port <= 0 ? BrtEngineConfig.serverPort : serverInfo.port;
        UDPPingSocket udpSocket = new UDPPingSocket(serverInfos, localIp, 0);
        return udpSocket;
    }

    private UDPSocket createSocket(List<BrtUrlInfo.ServerInfo> serverInfos) throws IOException {
        //int port = serverInfo.port <= 0 ? BrtEngineConfig.serverPort : serverInfo.port;
        UDPSocket udpSocket = BrtEngineConfig.brtDropTest ? new UDPDroppableSocket(serverInfos, localIp, 5000, performance) :
                new UDPSocket(serverInfos, localIp, 5000, performance);
        return udpSocket;
    }

    private void addSockets(UDPSocket udpSocket) throws IOException {
        clearSockets();
        udpSockets.add(udpSocket);
    }

    public void Run() {
//        if (logger.isDebug())
//            logger.debug("[%s] Started Run()", this.mediaCode);
        logger.info("brt-engine, brt connect start run");
        connectListener.onStart();
        brtEngineMetric.start();
        this.m_LastServerActiveTime = System.currentTimeMillis();

        try {
            runDownLoadTask();
        } catch (Exception e) {
            logger.error(e, "[%s] Error in Run()", this.mediaCode);
        }
        if (m_bIsAvailable) {
            m_bIsAvailable = false;
            //logger.info("[%s] Forced to set Available=false", this.mediaCode);
        }
        int innerErrCode = getLastInnerErrCode();
        connectListener.onDisconnected(!m_bClientRequestStop, m_LastMsgCode, innerErrCode);
        brtEngineMetric.stop();
        // 使用过后重置，防止干扰下次的判断
        lastErrorCode = LiveInnerErrorCodeDef.INNER_CODE_UNKNOWN;
//        if (logger.isDebug())
//            logger.debug("[%s] Ended Run()", this.mediaCode);
    }

    private int getLastInnerErrCode() {
        if (lastErrorCode != LiveInnerErrorCodeDef.INNER_CODE_UNKNOWN) {
            return lastErrorCode;
        }
        int innerErrCode = LiveInnerErrorCodeDef.INNER_CODE_OK;
        if (m_bServerTimeoutStop) {
            innerErrCode = LiveInnerErrorCodeDef.INNER_CODE_TIMEOUT;
        }
        return innerErrCode;
    }

    private LivePacket ping(UDPPingSocket udpSocket) throws Exception {
        LivePacketHeader packetHeader = initLivePacketHeader();
        // ping
        packetHeader.setMsgCode(MsgCodeDef.MSGCODE_PingRequest54171);
        packetHeader.setSessionKey1(System.currentTimeMillis());
        LivePacket reqPacket = new LivePacket();
        reqPacket.setHeader(packetHeader);

//        if (logger.isDebug())
//            logger.debug("[%s] Ping request is sending first message...", this.mediaCode);
        udpSocket.send(reqPacket);

        long pingStartTime = System.currentTimeMillis();

        while (System.currentTimeMillis() - pingStartTime < BrtEngineConfig.pingTimeout) {
            DatagramPacket datagramPacket = udpSocket.receive();
            if (datagramPacket != null) {
                int receiveLen = datagramPacket.getLength();
                //运行到这里，接收数据成功
                //校验消息格式
                byte[] msgBytes = null;
                int dataRecSize = datagramPacket.getLength();
                if (dataRecSize < datagramPacket.getData().length) {
                    byte[] newRecBuffer = new byte[dataRecSize];
                    System.arraycopy(datagramPacket.getData(), 0, newRecBuffer, 0, dataRecSize);
                    msgBytes = newRecBuffer;
                } else {
                    msgBytes = datagramPacket.getData();
                }
                //System.arraycopy(datagramPacket.getData(), 0, msgBytes, 0, receiveLen);
                LivePacket resPacket = ProtocolBuilder.parseLivePacket(msgBytes, performance);
                if (resPacket.getHeader().getProtocol() == LivePacketHeader.PROTOCOL) {
                    if (resPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_PingResponse54172) {
                        return resPacket;
                    }
                } else {
                    //无效的消息格式，忽略不处理
                    //logger.warn("[%s] Invalid msg format: %s, ignore it", this.mediaCode, resPacket.getHeader().toString());
                }

                resendStat.setRtt((int) (System.currentTimeMillis() - pingStartTime));
                break;
            } else {
                logger.error("[%s] Ping pending by not receive res msg, try again", this.mediaCode);
            }
        }

        return null;
    }

    public boolean auth() throws Exception {
        byte[] loginReqData = ClientUtils.buildLoginReqData(did, token, mediaCode, serverTime, validMinutes, platform, appVer, provider);
        LivePacketHeader packetHeader = initLivePacketHeader();
        //客户端请求服务器端的默认ret code号码0xE000/57344
        packetHeader.setMsgCode(MsgCodeDef.MSGCODE_ClientAuthRequest57343);
        //订阅数据
        packetHeader.setBackup1(subDataValue);
        LivePacket reqPacket = new LivePacket();
        reqPacket.setHeader(packetHeader);
        reqPacket.setData(loginReqData);

        UDPSocket udpSocket = udpSockets.get(currentUDPSocketIndex);
//        if (logger.isDebug())
//            logger.debug("[%s]S[%s] Auth request is sending first message...", this.mediaCode, currentUDPSocketIndex);

        int tryCnt = BrtEngineConfig.loginTimeout / BrtEngineConfig.authMaxRTT;
        int tries = 0;
        //验证标识，初始为false
        boolean bSessionAuthPassed = false;
        int nSessionAuthResCode = 0;
        boolean bSessionResTimeout = true;
        //performance.begin("Auth", Performance.Level.IMPORTANT);
        int authSeq = RandomUtils.nextInt(1000, 10000);

        do {
            reqPacket.getHeader().setReqSeq(authSeq++);
            udpSocket.send(reqPacket);
            //停顿一下，提升发送成功率
            Thread.sleep(1);
            udpSocket.send(reqPacket);
            //System.out.println(reqPacket.getHeader().toString());
//            if (logger.isDebug())
//                logger.debug("[%s]S[%s] Auth request sent 2 messages", this.mediaCode, currentUDPSocketIndex);

            // 如果接收失败，可能的原因是客户端和服务器端的路由还没有建立，或者session req的udp数据包丢失。
            // 这时就需要根据概率在后面重试重新发送session req消息。
            //byte[] receiveBuffer = new byte[4096];
            long loginStartTime = System.currentTimeMillis();
            int loginRtt = 0;
            while (System.currentTimeMillis() - loginStartTime < BrtEngineConfig.authMaxRTT) {
                DatagramPacket datagramPacket = udpSocket.receive();
                if (datagramPacket != null) {
                    if (loginRtt == 0) {
                        loginRtt = (int) (System.currentTimeMillis() - loginStartTime) >> 1;
                        ResendStat.setRtt(loginRtt);
                    }

                    BrtStat.put(statSeq, "ACS");
                    //运行到这里，接收数据成功
                    //校验消息格式
                    byte[] msgBytes = null;
                    int dataRecSize = datagramPacket.getLength();
                    if (dataRecSize < datagramPacket.getData().length) {
                        byte[] newRecBuffer = new byte[dataRecSize];
                        System.arraycopy(datagramPacket.getData(), 0, newRecBuffer, 0, dataRecSize);
                        msgBytes = newRecBuffer;
                        BrtStat.put(statSeq, "ACE");
                    } else {
                        msgBytes = datagramPacket.getData();
                        BrtStat.put(statSeq, "ANE");
                        //System.arraycopy(datagramPacket.getData(), 0, msgBytes, 0, dataRecSize);
                    }
                    try {
                        LivePacket resPacket = ProtocolBuilder.parseLivePacket(msgBytes, performance);
                        if (resPacket.getHeader().getProtocol() == LivePacketHeader.PROTOCOL) {
                            //忽略可能先于认证应答到达的消息
                            if (//resPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_ServerDataResponse54188
                                    resPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_ServerHBRequest54186
                                            || resPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_PingResponse54172) {
                                continue;
                            }
                            //消息协议合法
                            boolean sessionProcessResult = processSessionPacket(resPacket.getHeader());
                            if (sessionProcessResult) {
                                //远程校验成功
                                bSessionAuthPassed = true;
                                this.currentMsgPacket = resPacket;
                                // set request seq base value from server
                                udpSocket.setRequestSeq(resPacket.getHeader().getReqSeq());
                                sessionKey1 = reqPacket.getHeader().getSessionKey1();
                                sessionKey2 = reqPacket.getHeader().getSessionKey2();
                            } else if (m_bLiveServerDisconnected || m_bLivePacketStopped) {
                                //远程校验失败，并且服务器已经断开，则设置当前server不可用，并退出
                                this.m_bIsAvailable = false;
                                nSessionAuthResCode = resPacket.getHeader().getMsgCode();
                            } else {
                                nSessionAuthResCode = resPacket.getHeader().getMsgCode();
                            }
                        } else {
                            //无效的消息格式，忽略不处理
                            //logger.warn("[%s] Invalid msg format: %s, ignore it", this.mediaCode, resPacket.getHeader().toString());
                        }
                    } catch (Exception e) {
                        // 有些vpn会发送长度为12的无效报文，会导致异常，这里直接忽略之 by Elegant@2019年3月26日22:09:47
                        // logger.error(e, "error when process auth response");
                        continue;
                    }

                    //接收到数据，不论返回成功或是失败，都退出等待循环
                    //接收到应答，就不是timeout了
                    bSessionResTimeout = false;
                    break;
                } else {
                    //logger.error("[%s]S[%s] Auth pending by not receive res msg, try again", this.mediaCode, currentUDPSocketIndex);
                }
            }


//            if (bSessionResTimeout && logger.isInfo())
//                logger.info("Auth receive data timeout, tries[%d/%d], loginTimeout[%d], autMaxRTT[%d], host[%s]",
//                        tries+1, tryCnt, BrtEngineConfig.loginTimeout, BrtEngineConfig.authMaxRTT, udpSocket.getRemoteAddress().getHostAddress());
        } while (!bSessionAuthPassed && bSessionResTimeout && ++tries < tryCnt);

        if (!bSessionAuthPassed) {
            //没有认证通过，则退出
            m_LastMsgCode = nSessionAuthResCode;
            int errorCode = m_LastMsgCode;
            String errorMsg = "";
            if (bSessionResTimeout) {
                errorMsg = "AuthTimeout";
                errorCode = BrtConstant.INFO_CODE_TIME_OUT;
            } else if (nSessionAuthResCode == MsgCodeDef.MSGCODE_ServerErrCodeDisconnected62250) {
                errorMsg = "AuthServerDisconnected-" + m_LastMsgCode;
            } else {
                //服务器返回错误
                errorMsg = "AuthRetFailed-" + m_LastMsgCode;
            }
            //logger.error("[%s] Auth failed, err code[%s], msg[%s]. exit", this.mediaCode, m_LastMsgCode, errorMsg);
            connectListener.onAuthFailed(errorCode, errorMsg);
            //performance.end("Auth", Performance.Level.IMPORTANT, errorMsg, tries);
            BrtStat.put(statSeq, "authE");
            return false;
        } else {
            //performance.end("Auth", "Auth.OK", tries);
            connectListener.onAuthPassed(Collections.EMPTY_MAP);
            m_bIsAvailable = true;
            m_bLiveServerDisconnected = false;
            m_bServerTimeoutStop = false;
            m_bLivePacketStopped = false;
            BrtStat.put(statSeq, "authE");
            return true;
        }
    }

    public void runDownLoadTask() throws Exception {
        m_LastProcessPacketDataTime = System.currentTimeMillis();
        this.m_LastCheckDroppedPacketTime = System.currentTimeMillis();
        m_LastProcessStatAndHBTime = System.currentTimeMillis();

        List<List<BrtUrlInfo.ServerInfo>> exclude = new ArrayList<>();
        while (!m_bClientRequestStop) {
            List<BrtUrlInfo.ServerInfo> serverInfos = null;
            try {
                serverInfos = serverProvider.best(exclude);
                if (serverInfos.isEmpty()) {
                    logger.error("All %d groups of servers are failed", exclude.size());
                    return;
                }
                runTask(serverInfos);
                if (lastErrorCode == LiveInnerErrorCodeDef.INNER_CODE_KICK_OFF) {
                    logger.error("last error code:%d no need try more server!", lastErrorCode);
                    break;
                }
            } catch (Exception e) {
                logger.error(e, "Error in run a server. mediaCode[%s]", mediaCode);
            }

            if (serverInfos != null) {
                exclude.add(serverInfos);
            }
        }
    }

    private void runTask(List<BrtUrlInfo.ServerInfo> serverInfos) throws Exception {
        BrtStat.put(statSeq, "authS");
        UDPSocket udpSocket = createSocket(serverInfos);
        addSockets(udpSocket);

        //认证没有通过，则退出
        brtEngineMetric.authStart(mediaCode);
        if (!auth()) {
            brtEngineMetric.authFailure(serverInfos);
            return;
        } else {
            brtEngineMetric.authSuccess(serverInfos);
            logger.info("brt-engine, auth success");
        }

        // 服务器是否收到数据请求的标记, 如果没有收到，需要给服务器发送MsgCodeDef.MSGCODE_ClientDataRequest54181请求，触发服务器下发数据
        boolean isDataRequestReceivedByServer = false;
        long lastDataRequestSendTime = 0l;
        while (true) {
            if (!m_bIsAvailable) {
                //logger.info("[%s] Got exit signal from system, exit while loop", this.mediaCode);
                brtEngineMetric.tryServer(serverInfos, "m_bIsAvailable false");
                return;
            }

            //反复循环接收所有的socket
            for (UDPSocket curUDPSocket : udpSockets) {
                // 循环500次，接收数据。可以理解为，一个数据组最多包含500个packet
                //todo 当ts片段有6s的时候，4k的视频实际数据包一般都超过500个包，目前是单个udp socket运行，所以没有影响
                int dataReceiveRoundCount = 0;
                while (++dataReceiveRoundCount - 1 <= SOCKET_RECEIVE_BATCH_PACKET_COUNT) {
                    // while循环的开始位置====================================
                    if (!m_bIsAvailable) {
                        //logger.info("[%s] Got exit signal from system, exit for loop", this.mediaCode);
                        brtEngineMetric.tryServer(serverInfos, "m_bIsAvailable false");
                        return;
                    }

                    if (!isDataRequestReceivedByServer && System.currentTimeMillis() - lastDataRequestSendTime >= BrtEngineConfig.brtDRInterval) {
                        sendDataRequest(udpSocket);
                        lastDataRequestSendTime = System.currentTimeMillis();
                    }

                    performance.begin("receiveSocketData", Performance.Level.IMPORTANT);
                    boolean hasData = receiveSocketData(curUDPSocket);
                    if (!hasData) {
                        //处理时间超过19s就退出
                        long timeout = System.currentTimeMillis() - m_LastProcessPacketDataTime;
                        if (timeout > BrtEngineConfig.receiveUdpTimeout) {
                            this.m_bIsAvailable = false;
                            this.m_bServerTimeoutStop = true;
                            //logger.error("[%s] Receive timeout after %s millis", this.mediaCode, timeout);
                            //playerReader.onDataReceiveStop(MsgCodeDef.MSGCODE_TIME_OUT, "DataTimeout");
                            performance.end("receiveSocketData", Performance.Level.IMPORTANT, "receiveSocketData.timeout");
                            brtEngineMetric.tryServer(serverInfos, "ReceiveDataTimeout");
                            return;
                        }
                        //没有接收到数据，则退出当前socket的接收循环
                        break;
                    }
                    performance.end("receiveSocketData", "receiveSocketData.OK");

                    if (!isDataRequestReceivedByServer) {
                        isDataRequestReceivedByServer = true;
                    }

                    if (m_bLiveServerDisconnected) {
                        this.m_bIsAvailable = false;
                        sendClientFinish(curUDPSocket);
                        brtEngineMetric.tryServer(serverInfos, "ServerDisconnected");
                        return;
                    } else if (m_bLivePacketStopped) {
                        //todo 原始c++代码中没有这个逻辑，是后加的
                        //通讯中断，退出。与上面的差异是不用向服务器发送通知
                        this.m_bIsAvailable = false;
                        brtEngineMetric.tryServer(serverInfos, "PacketStopped");
                        return;
                    }
                    // while循环的结束位置
                    // ====================================
                }
                //开始使用下一个socket
            }
        }
    }

    private void sendClientFinish(UDPSocket udpSocket) throws IOException {
        if (currentMsgPacket == null)
            return;

        LivePacket feedbackPacket = new LivePacket();
        feedbackPacket.setHeader(currentMsgPacket.getHeader().copy());
        //通知服务器终止
        feedbackPacket.getHeader().setMsgCode(MsgCodeDef.MSGCODE_ClientFinishRequest54183);
        //发送两次,提高成功率，服务器端需要兼容重复的消息
        udpSocket.send(feedbackPacket);
        udpSocket.send(feedbackPacket);
        this.currentMsgPacket = feedbackPacket;
//        if(logger.isInfo()){
//            logger.info("[%s] Sent logout message to server", this.mediaCode);
//        }
    }

    private void sendDataRequest(UDPSocket udpSocket) throws IOException {
        LivePacket dataRequestPacket = new LivePacket();

        dataRequestPacket.setHeader(currentMsgPacket.getHeader().copy());
        dataRequestPacket.getHeader().setMsgCode(MsgCodeDef.MSGCODE_ClientDataRequest54181);
        //System.out.println(String.format("%s", dataRequestPacket.getHeader().toString()));
        String reqData = null;
        if (udpSocket.getReceivedHosts() != null && udpSocket.getReceivedHosts().size() > 0) {
            reqData = appendReceivedHost("", udpSocket.getReceivedHosts());
        }
        if (!StringUtils.isEmpty(reqData)) {
            byte data[] = encryptRequestData(reqData);
            dataRequestPacket.setData(data);
        }

        udpSocket.send(dataRequestPacket);
        this.currentMsgPacket = dataRequestPacket;
    }

    //return value: true: 有数据， false:无数据
    private boolean receiveSocketData(UDPSocket udpSocket) throws IOException, CloneNotSupportedException {
        try {
            //logger.debug("[%s] S[%s] Receive start...", this.mediaCode, currentUDPSocketIndex);
            //byte[] dataRecBuffer = new byte[2048];
            performance.begin("receiveSocketDataDetail", Performance.Level.IMPORTANT);
            byte[] dataRecBuffer = null;
            DatagramPacket datagramPacket = udpSocket.receive();

            if (datagramPacket == null) {
                //没有接收到数据，则退出当前socket的接收循环
                performance.end("receiveSocketDataDetail", "receiveSocketDataDetail.null");
                return false;
            }
            int dataRecSize = datagramPacket.getLength();

            performance.begin("dataRecBuffer");
            if (dataRecSize < datagramPacket.getData().length) {
                byte[] newRecBuffer = new byte[dataRecSize];
                System.arraycopy(datagramPacket.getData(), 0, newRecBuffer, 0, dataRecSize);
                dataRecBuffer = newRecBuffer;
                performance.end("dataRecBuffer", "dataRecBuffer.arraycopy");
            } else {
                dataRecBuffer = datagramPacket.getData();
                performance.end("dataRecBuffer", "dataRecBuffer.assign");
            }
            LivePacket dataPacket = null;
            try {
                dataPacket = ProtocolBuilder.parseLivePacket(dataRecBuffer, performance);
            } catch (Exception e) {
                // logger.error(e, "error when process data packet");
                return false;
            }
//        if(logger.isDebug())
//            logger.debug("[%s] received data: %s", mediaCode, ToStringBuilder.reflectionToString(dataPacket.getHeader(), ToStringStyle.JSON_STYLE));
            //累加接收数据量，这里包含了header的数据
            this.m_nReceivedDataSize += dataRecSize;

            //记录当前收到的msgcode,为了可能的错误退出做记录
            m_LastMsgCode = dataPacket.getHeader().getMsgCode();

            //检查数据格式是否合法
            if (dataPacket.getHeader().getProtocol() != LivePacketHeader.PROTOCOL) {
//            logger.error("[%s] Received wrong msg protocol: %s, msg header: %s, data len %s,packet len %s",
//                    this.mediaCode, dataPacket.getHeader().getProtocol(), dataPacket.getHeader().toString(), dataPacket.getDataSize(), dataRecSize);
//            performance.end("receiveSocketDataDetail", Performance.Level.IMPORTANT, "receiveSocketDataDetail.wrongProtocol");
                return false;
            } else if (dataRecSize > LivePacket.MAX_CLIENT_PACKET_LENGTH_VER3) {
//            logger.error("[%s] Received err size msg, want < %d but get: %s", this.mediaCode, LivePacket.MAX_PACKET_LENGTH_VER2, dataRecSize);
//            performance.end("receiveSocketDataDetail", Performance.Level.IMPORTANT,"receiveSocketDataDetail.gtmax");
                return false;
            }
            //运行到这里，接收数据合法，则开始处理

            // 服务器心跳请求54185或者视频数据54187类型，记录信息
            if (dataPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_ServerHBRequest54186
                    || dataPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_ServerDataResponse54188) {
                this.currentMsgPacket = dataPacket;
            }

            this.m_LastProcessPacketDataTime = System.currentTimeMillis();

            performance.begin("processDataPacket");
            processDataPacket(udpSocket, dataPacket);
            performance.end("processDataPacket");

            if (m_bLivePacketStopped) {
                //logger.info("receiveSocketData, m_bLivePacketStopped[%s]", m_bLivePacketStopped);
                return true;
            }

            //只有PULL模式才检查收取数据
            if (dataMode == DATA_MODE_PULL) {
                //检查丢包情况
                checkDroppedPacket(udpSocket);
            }

            performance.end("receiveSocketDataDetail", "receiveSocketDataDetail.OK");
            return dataPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_ServerDataResponse54188 ||
                    dataPacket.getHeader().getMsgCode() == MsgCodeDef.MSGCODE_ServerDataResendResponse54190;
        } finally {
            //无论是否收到数据都做心跳
            //统计和发送心跳
            //long processDataPacketBegin = System.currentTimeMillis();
            statisticsAndHBPacket(udpSocket);
            //logger.info("statisticsAndHBPacket dur %s ms", System.currentTimeMillis() - processDataPacketBegin);
        }

    }
    /* use ClientUtils instead
    //todo: add app version, engine version, brt url request params...
    private byte[] buildLoginReqData() throws Exception {
        //生成请求数据
        int _nRandNumLess30 = ClientUtils.lrand48() % 30;
        byte[] randBytes1 = ClientUtils.genarateRandomData(_nRandNumLess30 + 6);
        String strRand1 = new String(randBytes1);
        //strRand1 = "7Ft1PA0RqJGiR088vfGnr46Y4B-";

        int _nRandNumLess57 = ClientUtils.lrand48() % 57;
        byte[] randBytes2 = ClientUtils.genarateRandomData(_nRandNumLess57 + 6);
        String strRand2 = new String(randBytes2);

        // M=1 控制服务端多IP，区分(兼容)旧版本
        String reqTemplate = "RANDSTRING1=%s&CODE=%s&RANDSTRING2=%s&TOKEN=%s&TIME=%s&VALID=%s&APKSN=%s&NATIVESN=%s&PLATFORMINFO=%s&VERSION=%s&MD5SN=%s&APPVER=%s&M=1&CHN=%s&SID=%s";
        String md5String = MD5CustomUtils.genMdd5String(did);
        String chnVal = provider.val(BrtEngineConfig.BRT_KEY_CHN, "");
        String nativesnVal = provider.val(BrtEngineConfig.BRT_KEY_NATIVESN, "");
        String sid = provider.val(BrtEngineConfig.BRT_KEY_SID, "");
        String reqData = String.format(reqTemplate, strRand1, mediaCode, strRand2,
                StringUtils.trimToEmpty(token), serverTime, validMinutes,
                did, nativesnVal, StringUtils.trimToEmpty(platform), LIVE_MODULE_VERSION, md5String,
                StringUtils.trimToEmpty(appVer), chnVal, sid);
//        if (logger.isDebug())
//            logger.debug("[%s] Req plain: %s", this.mediaCode, reqData);

        byte encrypedData[] = encryptRequestData(reqData);

        String cs = provider.val(BrtEngineConfig.BRT_KEY_CS, "");
        if (!StringUtils.isEmpty(cs)) {
            byte[] csBytes = cs.getBytes(Charsets.UTF_8);
            short loginLen = (short) encrypedData.length;
            short csLen = (short) csBytes.length;
            short totalDataLen = (short) (2 + csLen + 2 + loginLen); // 没有判断越界，一般也不会发生
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            stream.write(HNumberUtils.toByteArray(totalDataLen));
            stream.write(HNumberUtils.toByteArray(csLen));
            stream.write(csBytes);
            stream.write(HNumberUtils.toByteArray(loginLen));
            stream.write(encrypedData);

            encrypedData = stream.toByteArray();
        }

        return encrypedData;
    }*/

    private String appendReceivedHost(String originalData, List<InetSocketAddress> hosts) {
        if (hosts == null || hosts.isEmpty())
            return originalData;

        StringBuilder builder = new StringBuilder(originalData);
        if (builder.length() > 0)
            builder.append("&");
        builder.append("RCVH=");
        for (InetSocketAddress address : hosts) {
            builder.append(address.getAddress().getHostAddress()).append(":").append(address.getPort()).append(",");
        }
        builder.setLength(builder.length() - 1);

        return builder.toString();
    }

    private byte[] encryptRequestData(String reqData) {
        byte[] reqBytes = reqData.getBytes();
        byte[] fixedLenReqBytes = ClientUtils.fixedByteArray(reqBytes, 8);
        byte[] encryptReqBytes = DESCipherUtils.encrypt(fixedLenReqBytes, DESCipherUtils.arrayIntToByte(HConstants.live_download_des_key471));
//        if (logger.isDebug())
//            logger.debug("[%s] Req encrypt: %s", this.mediaCode, HexUtils.byte2hex(encryptReqBytes).toLowerCase());
        return encryptReqBytes;
    }

    /**
     * @return true 需要进一步处理，比如首个数据或最后一个， false 不许进一步处理
     */
    private boolean processDataPacket(UDPSocket udpSocket, LivePacket packet) throws IOException {
        LivePacketHeader header = packet.getHeader();
        int msgCode = header.getMsgCode();

        String _strMsg;

        if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeDisconnected62250) {
            m_bLivePacketStopped = true;
//            _strMsg = "[%s] Server disconnected:%s";
//            logger.error(_strMsg, this.mediaCode, msgCode);
            return false;
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeAuthExpired62258) {
            m_bLivePacketStopped = true;
//            _strMsg = "[%s] Auth expired:%s";
//            logger.error(_strMsg, this.mediaCode, msgCode);
            return false;
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeStreamDiscontinueServer62264) {
            //TODO 服务器已经通知不再继续提供服务，这里需要怎么处理呢？现在是继续不断请求服务器，好像没有太大用处
            //c++原始逻辑：this.m_LastServerActiveTime = System.currentTimeMillis();
            //TODO 暂时修改为，遇到这种情况，不更新服务器的活动时间，会导致客户端超时退出
            //TODO 超时退出的方式太慢，无端浪费，现在修改为直接退出
            m_bLivePacketStopped = true;
//            logger.error("[%s] stream discontinue server:%s", this.mediaCode, udpSocket.getRemoteAddress());
            return false;
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeResourceNotFound62262) {
            //resource not found错误
//            _strMsg = "[%s] Resource not found:%s";
            m_bLivePacketStopped = true;
//            logger.error(_strMsg, this.mediaCode, msgCode);
            return false;
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeGetDataFailed62256) {
            this.m_bLivePacketStopped = true;
            //          _strMsg = "[%s] GetDataFailed:%s";
//            logger.error(_strMsg, this.mediaCode, msgCode);
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeUnauthorised62260) {
//            _strMsg = "[%s] Unauthorised:%s";
            this.m_bLivePacketStopped = true;
//            logger.error(_strMsg, this.mediaCode, msgCode);
            return false;
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeSessionExpired62266) {
//            _strMsg = "[%s] Session expired:%s";
            this.m_bLivePacketStopped = true;
//            logger.error(_strMsg, this.mediaCode, msgCode);
            return false;
        } else if (msgCode == MsgCodeDef.MSGCODE_ServerErrCodeKickoff62270) {
            this.m_bLivePacketStopped = true;
            lastErrorCode = LiveInnerErrorCodeDef.INNER_CODE_KICK_OFF;
            return false;
        }
        // 收到54185的消息，直接将原始消息的code改为54187,发送回去。平均每个socket每秒收到一个消息
        // 作用：服务器探测客户端的心跳
        if (msgCode == MsgCodeDef.MSGCODE_ServerHBRequest54186) {
            LivePacketHeader headerCopy = packet.getHeader().copy();

            int throttleSendBwLevel = headerCopy.getGroupDataCount();
            if (throttleSendBwLevel < BrtEngineConfig.minThrottleSendBwLevel)
                throttleSendBwLevel = BrtEngineConfig.minThrottleSendBwLevel;
            else if (throttleSendBwLevel > BrtEngineConfig.maxThrottleSendBwLevel)
                throttleSendBwLevel = BrtEngineConfig.maxThrottleSendBwLevel;
            resendStat.setThrottleSendBwLevel(throttleSendBwLevel);

            headerCopy.setMsgCode(MsgCodeDef.MSGCODE_ClientHBReqResponse54187);
            LivePacket feedbackPacket = new LivePacket();
            feedbackPacket.setHeader(headerCopy);
            udpSocket.send(feedbackPacket);
//            if (logger.isDebug()) {
//                logger.info("[%s] Got server HB and sent back", this.mediaCode);
//            }
            return false;
        }
        if (msgCode == MsgCodeDef.MSGCODE_ClientHeartbeatResponse54194) {
            int rtt = (int) (System.currentTimeMillis() % Short.MAX_VALUE - header.getGroupBufferCount());
            if (rtt > 0)
                resendStat.setHBRtt(rtt);

//            if (logger.isDebug()) {
//                logger.info("[%s] Got client HB response from server", this.mediaCode);
//            }
            return false;
        }
        if (msgCode <= MsgCodeDef.MSGCODE_ServerHBRequest54186) {
//            if (msgCode == 54183) {
//                // todo
//                // 间隔小于20s为何会忽略？
//                if (System.currentTimeMillis() - this.m_DownloadTaskStartTime <= 20 * 1000) {
//                    logger.error("[%s] Server msg code[%s] within DownloadTaskStartTime 20s", this.mediaCode, msgCode);
//                    return false;
//                }
//                logger.error("[%s] Server msg code[%s] more than DownloadTaskStartTime 20s", this.mediaCode, msgCode);
//                //todo 大于20s, 记录错误v9 = 328447;返回
//                return false;
//            }
//            if (msgCode != 54184)
//                return false;
            // 54184消息，按照50%概率直接返回，50%继续处理
            //todo 这里继续处理的逻辑不清楚，先直接返回
//            int randNum = ClientUtils.lrand48() % 100;
//            if (randNum > 49)
//                return false;
            return false;
        }
        if (msgCode != MsgCodeDef.MSGCODE_ServerDataResponse54188
                && msgCode != MsgCodeDef.MSGCODE_ServerDataResendResponse54190) {
            //不是数据包，不用继续处理
            return false;
        }
        if (msgCode == MsgCodeDef.MSGCODE_ServerDataResendResponse54190) {
            //统计重发数据的数量
            this.m_nPacket54190Received_Count++;
            int sendTime = header.getBackup1();
            if (sendTime > 0 && sendTime < Short.MAX_VALUE) {
                int current = (int) (System.currentTimeMillis() % Short.MAX_VALUE);
                int rtt = current - sendTime;
                if (rtt > 0) {
                    resendStat.setRtt(rtt);
                }
            }
        }

        //接收到了视频数据。数据header
        //记录最新的收到有效数据的时间，供drop check判断时间
        this.m_LastServerActiveTime = System.currentTimeMillis();

        int _nGroupBaseSeq = header.getGroupBaseSeq();
        int _nGroupSeq = header.getGroupSeq();
        //todo
        if (_nGroupSeq <= header.getGroupBufferCount() + _nGroupBaseSeq && _nGroupSeq >= _nGroupBaseSeq) {
            LiveSlice slice = new LiveSlice();
            if (header.getVersion() <= LivePacketHeader.VERSION_2) {
                slice.seq = header.getSeq();
                slice.groupCount = header.getCount();
            } else {
                slice.seq = header.getGroupDataSeq();
                slice.subGroupSeq = header.getSubGroupSeq();
                slice.groupCount = header.getGroupDataCount();
                slice.subGroupDataShards = header.getSubGroupDataShards();
                slice.subGroupParityShards = header.getSubGroupParityShards();
            }
            slice.data = packet.getData();
            slice.channelId = header.getChannelId();
            slice.groupSeq = header.getGroupSeq();
            slice.statSeq = statSeq;

            //todo 测试代码,验证crc16
//            int crc16Val = (int)crcTool.calculateCRC(slice.data);
//            if(crc16Val != header.getBackup1()){
//                logger.error("Packet data verify failed. group[%s] seq[%s] crc16[%s] actual[%s]",
//                        header.getGroupSeq(), header.getSugGroupSeq(), header.getBackup1(), crc16Val);
//            }
            if (!assembleNewSlice(slice, header)) {
                if (header.getGroupDataSeq() == 0)
                    return true;
            }
            if (header.getGroupDataSeq() == header.getGroupDataCount() - 1) {
                // 判断接收到的最后一个packet
                return true;
            }
        } else {
            //不符合接收条件，丢弃数据并记录日志
//            logger.info("[%s] Dropped packet group[%s] count[%s] seq[%s] GroupSeq[%s] GroupBufferCount[%s]",
//                    header.getChannelId(), _nGroupBaseSeq, header.getGroupDataCount(),
//                    header.getSugGroupSeq(), header.getGroupSeq(), header.getGroupBufferCount());
        }
        return false;
    }

    private boolean processSessionPacket(LivePacketHeader resHeader) {
        int _nServerMsgCode = resHeader.getMsgCode();
        String _strMsg;
        if (_nServerMsgCode == MsgCodeDef.MSGCODE_ServerErrCodeGetDataFailed62256) {
            this.m_bLivePacketStopped = true;
//            _strMsg = "[%s] GetDataFailed:%s";
//            logger.error(_strMsg, this.mediaCode, _nServerMsgCode);
            return false;
        }
        if (_nServerMsgCode > MsgCodeDef.MSGCODE_ServerErrCodeGetDataFailed62256) {
            if (_nServerMsgCode == MsgCodeDef.MSGCODE_ServerErrCodeAuthExpired62258) {
                this.m_bLivePacketStopped = true;
//                _strMsg = "[%s] Auth expired:%s";
//                logger.error(_strMsg, _nServerMsgCode);
            } else if (_nServerMsgCode < MsgCodeDef.MSGCODE_ServerErrCodeAuthExpired62258) {
//                _strMsg = "[%s] Unauthorised:%s";
//                logger.error(_strMsg, this.mediaCode, _nServerMsgCode);
            } else {
                if (_nServerMsgCode != MsgCodeDef.MSGCODE_ServerErrCodeResourceNotFound62262) {
                    return false;
                }
//                _strMsg = "[%s] Resource not found:%s";
//                logger.error(_strMsg, this.mediaCode, _nServerMsgCode);
            }
            return false;
        }
        if (_nServerMsgCode == MsgCodeDef.MSGCODE_ServerErrCodeDisconnected62250) {
            this.m_bLiveServerDisconnected = true;
//            logger.error("[%s] Live server disconnected:%s", this.mediaCode, _nServerMsgCode);
            return false;
        }
        if (_nServerMsgCode != MsgCodeDef.MSGCODE_ServerAuthOK54180 && _nServerMsgCode != MsgCodeDef.MSGCODE_ServerDataResponse54188)
            return false;
        //忽略channel id=0的消息
        if (resHeader.getChannelId() != 0 && this.currentChannelId != resHeader.getChannelId()) {
            int oldChannelId = this.currentChannelId;
            this.currentChannelId = resHeader.getChannelId();
            //初始化channel data
            getOrInitLiveChannelData(resHeader.getChannelId());
//            logger.info("[%s] Changed channel id from %s to %s", this.mediaCode, oldChannelId, this.currentChannelId);
        }

//        logger.info("[%s] Login success to channel %s", this.mediaCode, this.currentChannelId);
        //认证通过
        return true;
    }

    private void statisticsAndHBPacket(UDPSocket udpSocket) throws IOException, CloneNotSupportedException {
        long now = System.currentTimeMillis();
        long checkIntervalMillis = now - this.m_LastProcessStatAndHBTime;
        if (checkIntervalMillis < BrtEngineConfig.brtHBInterval) {
            //logger.debug("m_LastProcessStatAndHBTime < 500, ignore");
            return;
        }

        //todo 统计下载速度，暂略
        long nBandwidthSpeed = this.m_nReceivedDataSize * 1000 / checkIntervalMillis;

        // 估计是计数器，当计数超过上限后，重置
        //m_nSentPacketCount54189是持续累加，直到换台
        if (this.m_nSentPacketCount54189 >= Integer.MAX_VALUE - 1000) {
            this.m_nSentPacketCount54189 = 0;
            this.m_nPacket54190Received_Count = 0;
        }

        // 如果发送的packet数量是接收到的54190 packet数量的5倍，则认为上传带宽有问题
        if (this.m_nSentPacketCount54189 > 2000) {
            double uploadDownloadRatio = this.m_nSentPacketCount54189 / (this.m_nPacket54190Received_Count + 1);
            //performance.begin("Upload-Download", Performance.Level.IMPORTANT);
            if (uploadDownloadRatio > 5) {
                //logger.error("[%s] Upload-Download bandwidth ratio error: %s", this.mediaCode, (int) (uploadDownloadRatio * 100));
                //已经出现问题，重置统计数据重新计算，否则持续报相同的错误
                this.m_nSentPacketCount54189 = 0;
                this.m_nPacket54190Received_Count = 0;
                //performance.end("Upload-Download", Performance.Level.IMPORTANT, "Upload-Download.ratio>5");
            }
            //performance.end("Upload-Download", "Upload-Download.normal");
        }
        //数据接收统计置0
        this.m_nReceivedDataSize = 0;

        //向所有已经连接的socket发送统计心跳消息
        //msgcode=54193, count=本地发送和上次发送的时间间隔ms，比如501
        // GroupSeq值含义是发送的54189消息的统计个数
        // GroupBaseSeq是接收到的59190消息的统计个数
        // seq含义是平均网速，单位bytes/s
        LivePacketHeader heartbeatHeader = currentMsgPacket.getHeader().copy();
        heartbeatHeader.setMsgCode(MsgCodeDef.MSGCODE_ClientHeartbeatRequest54193);
        heartbeatHeader.setGroupDataCount((int) checkIntervalMillis);
        heartbeatHeader.setGroupSeq(this.m_nSentPacketCount54189);
        heartbeatHeader.setGroupBaseSeq(this.m_nPacket54190Received_Count);
        heartbeatHeader.setGroupDataSeq((int) nBandwidthSpeed);
        //订阅数据
        heartbeatHeader.setBackup1(subDataValue);
        heartbeatHeader.setGroupBufferCount((short) (System.currentTimeMillis() % Short.MAX_VALUE));

        LivePacket heartbeatMsg = new LivePacket();
        heartbeatMsg.setHeader(heartbeatHeader);

        udpSocket.send(heartbeatMsg);

//        long dur = System.currentTimeMillis() - m_LastProcessStatAndHBTime;
//        if (logger.isInfo())
//            logger.info("[%s] Sent client HB to server dur[%s]ms", this.mediaCode, dur);
        m_LastProcessStatAndHBTime = System.currentTimeMillis();
    }

    private void checkDroppedPacket(UDPSocket udpSocket) throws IOException, CloneNotSupportedException {
        long now = System.currentTimeMillis();
        long checkIntervalMillis = now - this.m_LastCheckDroppedPacketTime;
        if (checkIntervalMillis < 500) {
            //logger.debug("m_LastCheckDroppedPacketTime < 500, ignore");
            return;
        }

        //performance.begin("checkDroppedPacket", Performance.Level.IMPORTANT);
        long beginCheckDroppedPacket = System.currentTimeMillis();
        // 超过30s没有数据或没有通讯流量通讯则认为与服务器连接中断了。
        long noDataTime = System.currentTimeMillis() - this.m_LastServerActiveTime;
        if (noDataTime > BrtEngineConfig.serverTimeout) {
//            logger.error("[%s] disconnected from server caused by time out [%d]s: [%s]",
//                    this.mediaCode, noDataTime, udpSocket.getRemoteAddress());
            this.m_bServerTimeoutStop = true;
            this.m_bLiveServerDisconnected = true;
            //performance.end("checkDroppedPacket", Performance.Level.IMPORTANT, "checkDroppedPacket.serverTimeout");
            return;
        }
        BrtConnectChannelData channelData = liveChannelDataMap.get(currentChannelId);
        if (channelData == null) {
            //logger.error("[%s] Cannot find channel data by id %s", this.mediaCode, currentChannelId);
            return;
        }
//        long begin = System.currentTimeMillis();
        checkDownloadBufferComplete(channelData);
//        long checkDownloadBufferCompleteDur = System.currentTimeMillis() - beginCheckDroppedPacket;
//        if (logger.isInfo()) {
//            logger.info("checkDroppedPacket.checkDownloadBufferCompleteDur[%d]", checkDownloadBufferCompleteDur);
//        }
//        if (logger.isInfo())
//            logger.info("checkDownloadBufferComplete dur %s ms", System.currentTimeMillis() - begin);

        //循环检查每个sliceGroup是否有缺数据
        List<LiveSliceGroup> sliceGroups = new ArrayList<>(channelData.getGroupList());
        int totalResendGroups = 0;
        int totalResendPackages = 0;
        int totalResendSlices = 0;
        for (LiveSliceGroup sliceGroup : sliceGroups) {
            long sliceGroupNow = System.currentTimeMillis();
            long sliceGroupLifeDur = sliceGroupNow - sliceGroup.getBeginTime();
            long sliceGroupLastCheckDur = sliceGroupNow - sliceGroup.getLastCheckTime();

            //todo 自行添加的逻辑:如果一个group从开始接收到现在，超过 x s没有完成，则放弃并移除
            //todo 这个数据最好根据实际数据的长度来判断比较好，一刀切会有问题
            if (!sliceGroup.isCompleted() && sliceGroupLifeDur > BrtEngineConfig.receiveGroupTimeout) {
                removeGroup(channelData, sliceGroup.getExSeq(), RemoveGrpEvent.DROP_TIMEOUT, "ReceiveGrp.SliceTimeout");
//                if (logger.isInfo())
//                    logger.info("[%s] Dropped timeout[cfg:%s ms] group %s, count %s, available %s, life time %s ms, last resent dur %s ms",
//                            this.mediaCode, BrtEngineConfig.receiveGroupTimeout, sliceGroup.getExSeq(), sliceGroup.getGroupDataCount(), sliceGroup.getAvailableCount(), sliceGroupLifeDur, sliceGroupLastCheckDur);
                continue;
            }

            // 799999us=0.8秒, 如果当前的group在这个规定的时间内还没有接收完成，则向服务器发送补发数据请求54189
            //消息格式与54187/54190一样，只是msgcode不同
            //TODO 现在使用的方式是，最后group的被更新或检查时间超过0.8s，才发送补发数据请求
            //TODO 调优参数，使用group的生存时间，还是最后更新时间？
            boolean isInitStatus = sliceGroup.isInitStatus();
            int resendCfgDur;
            int resendChkCfgDur;
            long groupLastResendCheckDur;
            String resendStrategy = BrtEngineConfig.brtCheckResendStrategy;

            // if group count is grater than 1000, use resend slow start strategy
            if (sliceGroup.getCount() > BrtEngineConfig.brtResendStrategyGroupCnt) {
                resendStrategy = BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY_SLOW;
            }

            if (BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY_FAST.equalsIgnoreCase(resendStrategy)) {
                resendCfgDur = BrtEngineConfig.groupResendDuration;//  provider.val(BrtEngineConfig.GROUP_RESEND_DURATION, BrtEngineConfig.DFT_GROUP_RESEND_DURATION);//BrtEngineConfig.groupResendDuration;
                resendChkCfgDur = BrtEngineConfig.groupChkResendDuration;// provider.val(BrtEngineConfig.GROUP_CHK_RESEND_DURATION, BrtEngineConfig.DFT_GROUP_CHK_RESEND_DURATION);//500;
                groupLastResendCheckDur = sliceGroupLastCheckDur;
            } else {
                resendCfgDur = resendStat.getRto(sliceGroup.getExSeq(), isInitStatus, sliceGroup.getCount());
                resendChkCfgDur = resendStat.getRto(sliceGroup.getExSeq(), false, sliceGroup.getCount());
                groupLastResendCheckDur = sliceGroupNow - resendStat.getLastResendTime(sliceGroup.getExSeq(), sliceGroup.getBeginTime());
            }

            if (!sliceGroup.isCompleted()
                    && ((isInitStatus && sliceGroupLifeDur >= resendCfgDur)
                    || (!isInitStatus && groupLastResendCheckDur >= resendChkCfgDur))) {
                int availableCount = sliceGroup.getAvailableCount();
                int availableRatio = (int) (availableCount * 1.0 / (sliceGroup.getCount() + 1) * 100);
                boolean isResendOneByOne = availableRatio > BrtEngineConfig.resendOneByOneRatio;
                totalResendGroups++;
//                if (logger.isInfo())
//                    logger.info("[%s] Prepare to request resend for channel[%s] group %s, begin dur %sms, check dur %sms, count %s, available %s, ratio %s%%",
//                            this.mediaCode, currentChannelId, sliceGroup.getExSeq(), sliceGroupLifeDur, groupLastResendCheckDur, sliceGroup.getGroupDataCount(), sliceGroup.getAvailableCount(), availableRatio);
                //如果已经下载的比例超过20%，则使用逐个重发的方式，否则直接全部重新发送
                if (isResendOneByOne) {
                    // 如果Group有数据，则发送剩余没有收到的重发请求回去
                    //LiveSlice[] slices = sliceGroup.getSliceTreeMap();
                    //for (int i = 0; i < slices.length; i++) {
                    Set<Integer> resendSet = sliceGroup.getUncompletedSet();
                    if (BrtEngineConfig.DFT_BRT_RESEND_STRATEGY.equalsIgnoreCase(BrtEngineConfig.brtResendStrategy)) {
                        for (int i : resendSet) {
                            //if (slices[i] == null) {
                            LivePacketHeader resendMsgHeader = initLivePacketHeader();
                            resendMsgHeader.setMsgCode(MsgCodeDef.MSGCODE_ClientDataResendRequest54189);
                            if (sliceGroup.getSubGroupDataShards() == 0) {
                                resendMsgHeader.setSeq(i);
                                resendMsgHeader.setCount(sliceGroup.getCount());
                            } else {
                                resendMsgHeader.setGroupDataCount(sliceGroup.getCount());
                                resendMsgHeader.setSubGroupDataShards(sliceGroup.getSubGroupDataShards());
                                resendMsgHeader.setSubGroupParityShards(sliceGroup.getSubGroupParityShards());
                                resendMsgHeader.setGroupDataSeq(i);
                                resendMsgHeader.setSubGroupSeq(sliceGroup.getSubGroupSeq());
                            }
                            resendMsgHeader.setChannelId(channelData.getChannelId());
                            resendMsgHeader.setGroupBaseSeq(sliceGroup.getGroupBaseSeq());
                            //resendMsgHeader.setF32(sliceGroup.getF32());
                            resendMsgHeader.setGroupBufferCount(sliceGroup.getGroupBaseCount());
                            resendMsgHeader.setGroupSeq(sliceGroup.getExSeq());
                            //int time = (int)(System.currentTimeMillis() % Short.MAX_VALUE);
                            //resendMsgHeader.setBackup1(time);
                            //resendMsgHeader.setBackup1(sliceGroup.getF44());
                            resendMsgHeader.setBackup1(udpSockets.get(0).getNextUnreceiveHost());
                            resendMsgHeader.setSessionKey1(sliceGroup.getGap1());
                            resendMsgHeader.setSessionKey2(sliceGroup.getGap2());
                            LivePacket resendMsg = new LivePacket();
                            resendMsg.setHeader(resendMsgHeader);

                            udpSocket.send(resendMsg);
                            this.m_nSentPacketCount54189++;
                            totalResendPackages++;
                            // }
                        }
                    } else {
                        LivePacketHeader resendMsgHeader = initLivePacketHeader();
                        resendMsgHeader.setMsgCode(MsgCodeDef.MSGCODE_ClientDataResendRequest54189);
                        if (sliceGroup.getSubGroupDataShards() == 0) {
                            resendMsgHeader.setSeq(-2);
                            resendMsgHeader.setCount(sliceGroup.getCount());
                        } else {
                            resendMsgHeader.setGroupDataCount(sliceGroup.getCount());
                            resendMsgHeader.setSubGroupDataShards(sliceGroup.getSubGroupDataShards());
                            resendMsgHeader.setSubGroupParityShards(sliceGroup.getSubGroupParityShards());
                            resendMsgHeader.setGroupDataSeq(-2);
                            resendMsgHeader.setSubGroupSeq(sliceGroup.getSubGroupSeq());
                        }
                        resendMsgHeader.setChannelId(channelData.getChannelId());
                        resendMsgHeader.setGroupBaseSeq(sliceGroup.getGroupBaseSeq());
                        //resendMsgHeader.setF32(sliceGroup.getF32());
                        resendMsgHeader.setGroupBufferCount(sliceGroup.getGroupBaseCount());
                        resendMsgHeader.setGroupSeq(sliceGroup.getExSeq());
                        //int time = (int) (System.currentTimeMillis() % Short.MAX_VALUE);
                        //resendMsgHeader.setBackup1(time);
                        //resendMsgHeader.setBackup1(sliceGroup.getF44());
                        resendMsgHeader.setBackup1(udpSockets.get(0).getNextUnreceiveHost());
                        resendMsgHeader.setSessionKey1(sliceGroup.getGap1());
                        resendMsgHeader.setSessionKey2(sliceGroup.getGap2());
                        LivePacket resendMsg = new LivePacket();
                        resendMsg.setHeader(resendMsgHeader);

                        resendMsg.setData(generateResendData(sliceGroup));
                        udpSocket.send(resendMsg);

                        this.m_nSentPacketCount54189++;
                        totalResendPackages++;
                    }

                    if (sliceGroup.isInitStatus()) {
                        sliceGroup.setInitStatus(false);
                    }
                    //设置已经处理的时间，防止频繁刷新
                    sliceGroup.setLastCheckTime(System.currentTimeMillis());
                } else {
                    // 如果Group一个数据都没有接收到，则发送首个消息(-1)的重发消息回去。
                    LivePacketHeader resendMsgHeader = initLivePacketHeader();
                    resendMsgHeader.setMsgCode(MsgCodeDef.MSGCODE_ClientDataResendRequest54189);
                    if (sliceGroup.getSubGroupDataShards() == 0) {
                        resendMsgHeader.setSeq(-1);
                        resendMsgHeader.setCount(sliceGroup.getCount());
                    } else {
                        resendMsgHeader.setGroupDataCount(sliceGroup.getCount());
                        resendMsgHeader.setSubGroupDataShards(sliceGroup.getSubGroupDataShards());
                        resendMsgHeader.setSubGroupParityShards(sliceGroup.getSubGroupParityShards());
                        resendMsgHeader.setGroupDataSeq(-1);
                        resendMsgHeader.setSubGroupSeq(sliceGroup.getSubGroupSeq());
                    }
                    resendMsgHeader.setChannelId(channelData.getChannelId());
                    resendMsgHeader.setGroupBaseSeq(sliceGroup.getGroupBaseSeq());
                    //int time = (int)(System.currentTimeMillis() % Short.MAX_VALUE);
                    //resendMsgHeader.setBackup1(time);
                    //resendMsgHeader.setF32(sliceGroup.getF32());
                    resendMsgHeader.setGroupBufferCount(sliceGroup.getGroupBaseCount());
                    resendMsgHeader.setGroupSeq(sliceGroup.getExSeq());
                    //resendMsgHeader.setBackup1(sliceGroup.getF44());
                    resendMsgHeader.setBackup1(udpSockets.get(0).getNextUnreceiveHost());
                    resendMsgHeader.setSessionKey1(sliceGroup.getGap1());
                    resendMsgHeader.setSessionKey2(sliceGroup.getGap2());
                    LivePacket resendMsg = new LivePacket();
                    resendMsg.setHeader(resendMsgHeader);

                    udpSocket.send(resendMsg);
                    this.m_nSentPacketCount54189++;
                    //设置已经处理的时间，防止频繁刷新
                    sliceGroup.setLastCheckTime(System.currentTimeMillis());
                    totalResendPackages++;

                    if (sliceGroup.isInitStatus()) {
                        sliceGroup.setInitStatus(false);
                    }
                }

                int resendSlices = sliceGroup.getCount() - availableCount;
                totalResendSlices += resendSlices;
                int rounds = resendStat.recordResend(sliceGroup.getExSeq(), sliceGroupNow, isInitStatus,
                        sliceGroupLifeDur, resendCfgDur, groupLastResendCheckDur, resendChkCfgDur,
                        isResendOneByOne, resendSlices, totalResendGroups, totalResendPackages, totalResendSlices,
                        sliceGroups.size());
                int lostPercent = resendSlices * 100 / sliceGroup.getCount();
                sliceGroup.setLostPercent(lostPercent);
                sliceGroup.setLostCount(resendSlices);
                sliceGroup.setResendRounds(rounds);
                sliceGroup.setResendCount(sliceGroup.getResendCount() + resendSlices);
                if (rounds == 1) {
                    sliceGroup.setFirstLostCount(resendSlices);
                    sliceGroup.setFirstLostPercent(lostPercent);
                }
            }
        }
        //更新上次处理时间，判断下次执行的时间
        this.m_LastCheckDroppedPacketTime = System.currentTimeMillis();

//        if (logger.isInfo()) {
//            logger.info("checkDroppedPacket.checkResendDur[%d]", System.currentTimeMillis() - beginCheckDroppedPacket - checkDownloadBufferCompleteDur);
//            logger.info("checkDroppedPacket dur %s ms", System.currentTimeMillis() - beginCheckDroppedPacket);
//        }
//        performance.end("checkDroppedPacket", "checkDroppedPacket.normal");
    }

    private byte[] generateResendData(LiveSliceGroup sliceGroup) {
        Set<Integer> resendSet = sliceGroup.getUncompletedSet();
        if (sliceGroup.getSubGroupDataShards() == 0) {
            return ResendUtils.resendSet2Bytes(resendSet);
        } else {
            int lostPercent = resendSet.size() * 100 / sliceGroup.getSubGroupShards();
            ResendStatInfo info = new ResendStatInfo();
            info.setRtt(ResendStat.getRtt());
            info.setLostPercent(lostPercent);
            info.setResendSet(resendSet);

            return ResendUtils.resendStatInfo2Bytes(info);
        }
    }

    private boolean assembleNewSlice(LiveSlice slice, LivePacketHeader packetHeader) {
        //todo 可能出现不同channle数据同时存在的情况，比如换台的时候，这里先假定不换台
        if (this.currentChannelId == 0) {
            this.currentChannelId = slice.channelId;
        }
        if (slice.channelId != currentChannelId) {
//            if (logger.isInfo())
//                logger.info("[%s] Current channel id is %s, ignore packet with channel id %s", this.mediaCode, currentChannelId, slice.channelId);
            return false;
        }
        if (isFirstSlice) {
            logger.info("brt-engine, receive first slice, group[%d]", slice.groupSeq);
            isFirstSlice = false;
        }
        BrtConnectChannelData channelData = liveChannelDataMap.get(currentChannelId);
        if (channelData == null) {
            channelData = getOrInitLiveChannelData(slice.channelId);
        }
        int groupCount = channelData.getGroupCount();
        if (groupCount > 0) {
            int firstSliceGroupSeq = channelData.getFirstGroupSeq();
            //如果服务器端缓存的最小group已经比本地的还大，本地没有完成的group则无法再得到新数据了，只能删除处理
            int serverGroupBaseSeq = packetHeader.getGroupBaseSeq();
            //firstSliceGroupSeq = channelData.getFirstGroupSeq();
            //一次处理一个，因为slice很多，group少，处理来的及
            if (firstSliceGroupSeq < serverGroupBaseSeq) {
                LiveSliceGroup sliceGroup = channelData.getGroup(firstSliceGroupSeq);
                if (sliceGroup != null && !sliceGroup.isCompleted()) {
                    logger.info("brt-engine, connect remove group because overdue-[%d] vs [%d]", firstSliceGroupSeq, serverGroupBaseSeq);
                    removeGroup(channelData, firstSliceGroupSeq, RemoveGrpEvent.DROP_OVERDUE, "ReceiveGrp.<ServerGroupBaseSeq");
//                    if (logger.isInfo())
//                        logger.info("[%s] Current oldest group seq[%s] is older than server group base seq [%s], drop client group seq[%s]",
//                                this.mediaCode, firstSliceGroupSeq, serverGroupBaseSeq, firstSliceGroupSeq);
                }
            }
        }
        if (!channelData.addSlice(slice, packetHeader, connectionId, performance, isSlicePushMode)) {
            resendStat.recordRepeat(slice.groupSeq, slice.seq);
            //记录重复数据包数量
//            if (logger.isDebug())
//                logger.debug("[%s] Duplicate msg: %s", this.mediaCode, ToStringBuilder.reflectionToString(packetHeader, ToStringStyle.JSON_STYLE));
            return false;
        } else {
            if (connectListener != null) {
                try {
                    connectListener.onDataBegin(channelData.getGroup(slice.groupSeq));
                } catch (Exception e) {
                    logger.error("Error on post data begin to connect listener. groupSeq:%d", slice.groupSeq);
                }
            }
        }
        return true;
    }

    private BrtConnectChannelData getOrInitLiveChannelData(final int channelId) {
        BrtConnectChannelData channelData = liveChannelDataMap.get(channelId);
        if (channelData == null) {
            synchronized (liveChannelDataMap) {
                channelData = liveChannelDataMap.get(channelId);
                if (channelData == null) {
                    channelData = new BrtConnectChannelData(BrtEngineConfig.brtKeepOverdueSize);
                    channelData.setChannelId(channelId);
                    channelData.setStartTime(System.currentTimeMillis());
                    liveChannelDataMap.put(channelId, channelData);
                }
            }
        }

        return channelData;
    }

    private LivePacketHeader initLivePacketHeader() {
        //初始化包头数据
        LivePacketHeader header = new LivePacketHeader();
        header.setMsgCode(0);
        header.setChannelId(0);
        header.setGroupDataSeq(0);
        header.setGroupDataCount(0);
        header.setGroupSeq(0);

        header.setSessionKey1(0);
        header.setSessionKey2(0);
        header.setGroupBaseSeq(0);
        header.setGroupBufferCount((short) 0);
        //订阅数据
        header.setBackup1(subDataValue);

        // SEV(StreamEngineValue)
        header.setEncryptType(sev);

        return header;
    }

    //检查每个group是否已经完成下载
    // 从LiveModule的本地缓冲读取数据，通过TcpClientSocket将数据发送给Video Player
    private void checkDownloadBufferComplete(BrtConnectChannelData channelData) {
        try {
            if (currentChannelId != 0) {
                processAvailableSliceData(channelData);
            }
        } catch (Exception e) {
            logger.error(e, "[%s] Error in check buffer complete for channel %s", this.mediaCode, currentChannelId);
        }
    }

    private boolean processAvailableSliceData(BrtConnectChannelData channelData) throws IOException, InterruptedException {
        if (channelData == null) {
            //没有数据
//            if (logger.isDebug())
//                logger.debug("[%s] Connect: check top group, channelData is null", mediaCode);
            return false;
        }
        if (channelData.getGroupCount() == 0) {
            //没有数据
//            if (logger.isInfo())
//                logger.info("[%s] Connect: check top group, channelData.getGroupDataCount()=0", mediaCode);
            return false;
        }
//        int socketWaterline = udpSockets.iterator().next().getReceiveWaterline();
//        int sendWaterline = udpSockets.iterator().next().getSendWaterline();
//        long sliceGroupBegin = System.currentTimeMillis();
        Collection<LiveSliceGroup> liveSliceGroups = new ArrayList<>(channelData.getGroupList());
        for (LiveSliceGroup sliceGroup : liveSliceGroups) {
//            long dur = sliceGroup.getDuration();
//            if (dur == 0) {
//                dur = System.currentTimeMillis() - sliceGroup.getBeginTime();
//            }
//            if (logger.isInfo())
//                logger.info("[%s] Connect: check top group. group[%s], count [%s], have[%s], dur[%s]ms, resend[%s], completed[%s], socket REC WL[%s%%], SEND WL[%s%%]",
//                        this.mediaCode, sliceGroup.getExSeq(), sliceGroup.getGroupDataCount(), sliceGroup.getAvailableCount(), dur, sliceGroup.getLostCount(), sliceGroup.isCompleted(), socketWaterline, sendWaterline);

            //删除已经写完的group
            if (sliceGroup.isCompleted()) {
                removeGroup(channelData, sliceGroup.getExSeq(), RemoveGrpEvent.COMPLETE, "ReceiveGrp.OK");
//                if (logger.isInfo())
//                    logger.info("[%s] Connect removed completed group[%s] rec dur[%s]ms", this.mediaCode, sliceGroup.getExSeq(), sliceGroup.getDuration());
            }
        }
//        long sliceGroupDur = System.currentTimeMillis() - sliceGroupBegin;
//        if (logger.isInfo())
//            logger.info("[%s] sliceGroupDur [%s]ms", this.mediaCode, sliceGroupDur);

//        long removeGroupCountBegin = System.currentTimeMillis();
        //todo 这个数字需要不断测试优化
        //限制缓存的group总数量不超过10个（根据配置），超过的部分，最早的废弃
        //重新计算剩余slice group数量
        int groupCount = channelData.getGroupCount();
        if (groupCount > BrtEngineConfig.groupCacheCount) {
            int removeGroupCount = groupCount - BrtEngineConfig.groupCacheCount;
            for (int i = 0; i < removeGroupCount; i++) {
                int removeFirstSliceGroupSeq = channelData.getFirstGroupSeq();
                if (removeFirstSliceGroupSeq > 0) {
                    removeGroup(channelData, removeFirstSliceGroupSeq, RemoveGrpEvent.DROP_OVERCACHE, "ReceiveGrp.exceededGroupCacheCountLimit");
//                    if (logger.isInfo())
//                        logger.info("[%s] Current group count[%s] is morn than limit %s, drop older group seq[%s]",
//                                this.mediaCode, groupCount, BrtEngineConfig.groupCacheCount, removeFirstSliceGroupSeq);
                }
            }
        }
//        long removeGroupCountDur = System.currentTimeMillis() - removeGroupCountBegin;
//        if (logger.isInfo())
//            logger.info("[%s] removeGroupCount [%s]ms", this.mediaCode, removeGroupCountDur);
        return true;
    }

    private void removeGroup(BrtConnectChannelData channelData, int groupSeq, RemoveGrpEvent event, String msg) {
        boolean isDropped = false;
        if (connectListener != null) {
            LiveSliceGroup sliceGroup = null;
            try {
                sliceGroup = channelData.getGroup(groupSeq);
                if (sliceGroup == null)
                    return;

                switch (event) {
                    case COMPLETE:
                        sliceGroup.setStatus(LiveSliceGroup.Status.COMPLETED);
                        connectListener.onDataCompleted(sliceGroup);
                        break;

                    case DROP_TIMEOUT:
                    case DROP_OVERDUE:
                    case DROP_OVERCACHE:
                        sliceGroup.setRtt(ResendStat.getRtt());
                        sliceGroup.setStatus(LiveSliceGroup.Status.DROPPED);
                        connectListener.onDataDropped(sliceGroup, msg);
                        isDropped = true;
                        BrtStat.put(sliceGroup, resendStat, event);
                        break;
                }
            } catch (Throwable e) {
                logger.error(e, "[%s] Error in post data to listener, data seq:%s, event:%s", this.mediaCode, groupSeq, event);
            } finally {
                if (sliceGroup != null) {
                    try {
                        resendStat.removeAndLogResendInfo(event, groupSeq, sliceGroup.getBeginTime(), sliceGroup.getAvailableCount(), sliceGroup.getCount(), msg);
                    } catch (Throwable ignored) {
                    }
                    //sliceGroup.detail(event.name());
                }
            }
        }

        if (isDropped)
            performance.end("LiveSliceGroup", groupSeq, Performance.Level.IMPORTANT, msg);
        else
            performance.end("LiveSliceGroup", msg);

        channelData.removeGroup(groupSeq);
    }

    ResendStat getResendStat() {
        return resendStat;
    }

    public void shutdown() {
        //todo 支持重复关闭操作
        m_bClientRequestStop = true;
        m_bIsAvailable = false;
        liveChannelDataMap.clear();
        clearSockets();
        serverProvider.shutdown();
        if (logger.isInfo())
            logger.info("[%s] shutdown end", this.mediaCode);
    }

    private void clearSockets() {
        if (udpSockets != null && !udpSockets.isEmpty()) {
            for (UDPSocket udpSocket : udpSockets) {
                try {
                    sendClientFinish(udpSocket);
                } catch (Exception e) {
                    logger.error(e, "[%s]Error in send udp socket, ignore", this.mediaCode);
                }

//                List<String> unreceivedHosts = new ArrayList<>();
//                for (InetSocketAddress address : udpSocket.getUnreceiveHosts()) {
//                    unreceivedHosts.add(address.getHostString());
//                }
//                if (!unreceivedHosts.isEmpty()) {
//                    logger.error("Unreceived Hosts: %s", unreceivedHosts);
//                }

                try {
                    udpSocket.close();
                } catch (Exception e) {
                    logger.error(e, "[%s]Error in closing udp socket, ignore", this.mediaCode);
                }
            }
            udpSockets.clear();
        }
    }
}
