package com.stream.brt.engine;

import com.stream.tool.log.Logger;
import com.stream.tool.log.LoggerFactory;

import java.text.SimpleDateFormat;
import java.util.*;

public class Performance {
    public interface Listener {
        void onStat(LinkedHashMap<String, Map<String,String>> performance);
    }

    enum Level {
        NORMAL,
        IMPORTANT
    }

    private static Logger logger = LoggerFactory.getLogger(Performance.class.getSimpleName());
    final private Map<String,Map<Long,Long>> runningTasks = Collections.synchronizedMap(new HashMap<String,Map<Long,Long>>());
    final private Map<String,TaskStat> taskStatMap = Collections.synchronizedMap(new HashMap<String,TaskStat>());
    private Timer timer;
    private long startTime = 0;
    private String startTimeReadable;
    private long lastRunAt = 0;
    private static Listener listener;
    private static BrtStatusListener statusListener;
    private String server;
    private String mediaCode;
    private boolean isPerformanceStat = false;

    Performance(boolean isPerformanceStat) {
        startTime = System.currentTimeMillis();
        lastRunAt = startTime;
        SimpleDateFormat dateFormat = new SimpleDateFormat("dd-MM-yyyy HH:mm:ss");
        startTimeReadable = dateFormat.format(new Date(startTime));
        runningTasks.clear();
        taskStatMap.clear();

        this.isPerformanceStat = isPerformanceStat;
        if (isPerformanceStat) {
            timer = new Timer();
            timer.schedule(new TimerTask() {
                @Override
                public void run() {
                    doTaskStat();
                }
            }, 10000, BrtEngineConfig.performanceStatDur);
        } else {
            if (timer != null) {
                timer.cancel();
                timer.purge();
                timer = null;
            }
        }
    }

    void setServer(String server) {
        this.server = server;
    }

    public void setMediaCode(String mediaCode) {
        this.mediaCode = mediaCode;
    }

    void stop() {
        doTaskStat();
        isPerformanceStat = false;
        if (timer != null) {
            timer.cancel();
            timer.purge();
            timer = null;
        }
    }

    public static void setListener(Listener listener) {
        Performance.listener = listener;
    }

    public static void setStatusListener(BrtStatusListener listener) {
        Performance.statusListener = listener;
    }

    public void begin(String task) {
        begin(task, 0L, Level.NORMAL);
    }

    public void begin(String task, Level level) {
        begin(task, 0L, level);
    }

    public void begin(String task, long id) {
        begin(task, id, Level.NORMAL);
    }

    public void begin(String task, long id, Level level) {
        if (!isPerformanceStat && level != Level.IMPORTANT && statusListener == null)
            return;

        Map<Long,Long> tasks = runningTasks.get(task);
        if (tasks == null) {
            synchronized (runningTasks) {
                tasks = runningTasks.get(task);
                if (tasks == null) {
                    tasks = Collections.synchronizedMap(new HashMap<Long,Long>());
                    tasks.put(id, System.currentTimeMillis());
                    runningTasks.put(task, tasks);
                }
            }
        } else {
            tasks.put(id, System.currentTimeMillis());
        }
    }

    public void end(String task) {
        end(task, 0L);
    }

    public void end(String task, Level level) {
        end(task, 0L, level);
    }

    public void end(String task, long id) {
        end(task, id, Level.NORMAL, null, 0);
    }

    public void end(String task, long id, Level level) {
        end(task, id, level, null, 0);
    }

    public void end(String task, String result) {
        end(task, 0L, result);
    }

    public void end(String task, Level level, String result) {
        end(task, 0L, level, result);
    }

    public void end(String task, long id, String result){
        end(task, id, Level.NORMAL, result, 0);
    }

    public void end(String task, long id, Level level, String result){
        end(task, id, level, result, 0);
    }

    public void end(String task, String result, long bytes) {
        end(task, 0L, Level.NORMAL, result, bytes);
    }

    public void end(String task, Level level, String result, long bytes) {
        end(task, 0L, level, result, bytes);
    }

    public void end(String task, long id, long bytes) {
        end(task, id, Level.NORMAL,null, bytes);
    }

    public void end(String task, long id, Level level, long bytes) {
        end(task, id, level,null, bytes);
    }

    public void end(String task, long id, Level level, String result, long bytes) {
        if (!isPerformanceStat && level != Level.IMPORTANT && statusListener == null)
            return;

        long current = System.currentTimeMillis();

        Map<Long, Long> map =  runningTasks.get(task);
        if (map == null)
            return;

        Long begin = map.remove(id);
        if (begin == null) {
            return;
        }

        long time = current - begin;
        String key = task;
        if (result != null && !result.trim().equals(""))
            key = result;

        TaskStat stat = taskStatMap.get(key);
        if (stat == null) {
            stat = new TaskStat();
            if (time>=0) {
                stat.maxRunTime = time;
                stat.minRunTime = time;
                stat.totalRunTime = time;
                stat.negativeCnt=0;
                stat.totalNegativeTime=0;
                stat.minNegativeTime=0;
            } else {
                stat.maxRunTime=0;
                stat.minRunTime=0;
                stat.totalRunTime = 0;
                stat.negativeCnt=1;
                stat.totalNegativeTime=time;
                stat.minNegativeTime=time;
            }

            stat.task = key;
            stat.totalRunCnt = 1;
            taskStatMap.put(key,stat);
        } else if (current > stat.lastTaskRunAt) {
            if (time >=0) {
                if (stat.minRunTime > time) {
                    stat.minRunTime = time;
                }
                if (stat.maxRunTime < time) {
                    stat.maxRunTime = time;
                }
                stat.totalRunTime += time;
            } else {
                stat.negativeCnt++;
                if (stat.minNegativeTime > time) {
                    stat.minNegativeTime = time;
                }
                stat.totalNegativeTime+=time;
            }
            stat.totalRunCnt++;
        }

        stat.totalBytes += bytes;
        stat.lastTaskRunAt = current;
        lastRunAt = current;
    }

    private synchronized void doTaskStat() {
        if (taskStatMap.isEmpty())
            return;

        if (!logger.isInfo() && listener == null && statusListener == null)
            return;

        List<TaskStat> stats = new ArrayList<>(taskStatMap.values());
        Collections.sort(stats);

        LinkedHashMap<String, Map<String,String>> linkedHashMap = new LinkedHashMap<>();
        Map<String,String> map = new HashMap<>();
        map.put("beginAt", Long.toString(startTime));
        map.put("lastTaskRunAt", Long.toString(lastRunAt));
        map.put("server", server);
        map.put("mediaCode", mediaCode);
        linkedHashMap.put("Runtime", map);

        for (TaskStat stat:stats) {
            map = new HashMap<>();
            map.put("totalRunCnt",            Long.toString(stat.totalRunCnt));
            map.put("totalRunTime",      Long.toString(stat.totalRunTime));
            map.put("totalNegativeTime", Long.toString(stat.totalNegativeTime));
            map.put("negativeCnt",       Long.toString(stat.negativeCnt));
            map.put("minNegativeTime",   Long.toString(stat.minNegativeTime));
            map.put("maxRunTime",        Long.toString(stat.maxRunTime));
            map.put("minRunTime",        Long.toString(stat.minRunTime));
            map.put("lastTaskRunAt",     Long.toString(stat.lastTaskRunAt));
            map.put("totalBytes",             Long.toString(stat.totalBytes));
            map.put("lastPrintTotalBytes",    Long.toString(stat.lastPrintTotalBytes));
            map.put("lastTotalRunTime",  Long.toString(stat.lastTotalRunTime));
            map.put("lastPrintTotalRunCnt",   Long.toString(stat.lastPrintTotalRunCnt));

            long runTime =  stat.totalRunTime  - stat.lastTotalRunTime;
            long avgRunTime = stat.totalRunTime / stat.totalRunCnt;

            double avgRunCntSpeed = stat.totalRunTime >0 ? stat.totalRunCnt * 1000.0 / stat.totalRunTime : -1;
            long runCnt = (stat.totalRunCnt - stat.lastPrintTotalRunCnt);
            double currentRunCntSpeed = runTime > 0 ? runCnt * 1000.0 / runTime : -1;

            long avgByteSpeed = stat.totalRunTime> 0 ? stat.totalBytes / stat.totalRunTime : -1;
            long bytes = (stat.totalBytes - stat.lastPrintTotalBytes);
            long currentByteSpeed = runTime > 0 ? bytes / runTime : -1;

            stat.lastTotalRunTime = stat.totalRunTime;
            stat.lastPrintTotalRunCnt = stat.totalRunCnt;
            stat.lastPrintTotalBytes = stat.totalBytes;

            map.put("runTime", Long.toString(runTime));
            map.put("runCnt", Long.toString(runCnt));
            map.put("bytes", Long.toString(bytes));
            map.put("avgRunTime", Long.toString(avgRunTime));
            map.put("avgRunCntSpeed", String.format(Locale.ENGLISH, "%.2f", avgRunCntSpeed));
            map.put("currentRunCntSpeed", String.format(Locale.ENGLISH, "%.2f", currentRunCntSpeed));
            map.put("avgByteSpeed", Long.toString(avgByteSpeed));
            map.put("currentByteSpeed", Long.toString(currentByteSpeed));

            linkedHashMap.put(stat.task, map);
        }

        if (listener != null) {
            listener.onStat(linkedHashMap);
        }

        if (logger.isInfo() || statusListener != null)
            printTaskStat(linkedHashMap);
    }

    public void printTaskStat(LinkedHashMap<String, Map<String,String>> linkedHashMap) {
        StringBuilder builder = new StringBuilder();
        builder.append("\nPerformance| ===== Begin ======================================================================\n");
        builder.append(String.format("Performance| BeginAt:%s RunningTime: %ds LastTaskRunAt:%ds ago\n",  startTimeReadable, (lastRunAt - startTime)/1000, (System.currentTimeMillis()-lastRunAt)/1000));
        String title = "Performance| AvgRunTime    AvgRunCntSpeed(CNT/S)  RunCntSpeed(CNT/S)  RunCnt        TotalRunTime  AvgSpeed(KB/S)  MinRunTime    MaxRunTime    -Cnt    -TotalTime  -minTime  Speed(KB/S)   Task\n";
        builder.append(title);
        for (Map.Entry<String, Map<String,String>> entry:linkedHashMap.entrySet()) {
            if (entry.getKey().equals("Runtime"))
                continue;

            Map<String,String> map = entry.getValue();
            builder.append(String.format("Performance| %8s      %16s         %12s          %8s      %8s      %8s        %8s      %8s      %4s    %8s    %6s    %8s      %s\n",
                    map.get("avgRunTime"),
                    map.get("avgRunCntSpeed"),
                    map.get("currentRunCntSpeed"),
                    map.get("totalRunCnt"),
                    map.get("totalRunTime"),
                    map.get("avgByteSpeed"),
                    map.get("minRunTime"),
                    map.get("maxRunTime"),
                    map.get("negativeCnt"),
                    map.get("totalNegativeTime"),
                    map.get("minNegativeTime"),
                    map.get("currentByteSpeed"),
                    entry.getKey()));
        }
        builder.append("Performance| ===== End ========================================================================\n");

        String msg = builder.toString();
        if (logger.isInfo())
            logger.info(msg);

        BrtStatusListener l = statusListener;
        if (l != null)
            l.onStatus(msg);
    }

    private static class TaskStat implements Comparable<TaskStat>{
        String task;
        long totalRunCnt;
        long totalRunTime;
        long totalNegativeTime;
        long negativeCnt;
        long minNegativeTime;
        long maxRunTime;
        long minRunTime;
        long lastTaskRunAt;
        long totalBytes;
        long lastPrintTotalBytes;
        long lastTotalRunTime;
        long lastPrintTotalRunCnt;

        @Override
        public int compareTo(TaskStat o) {
            return o.task.compareTo(task);
        }
    }
}
