package com.stream.brt.engine;

import com.stream.brt.engine.metric.BrtEngineMetric;
import com.stream.brt.engine.model.LiveSliceGroup;
import com.stream.brt.engine.model.ResendMetricInfo;
import com.stream.brt.prot.LivePacket;
import com.stream.brt.tool.log.Logger;
import com.stream.brt.tool.log.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ResendStat {
    private static Logger logger = LoggerFactory.getLogger(ResendStat.class.getSimpleName());
    private static final double ALPHA = 0.8;

    //记录重发信息
    private Map<Integer, List<ResendInfo>> resendInfos = new HashMap<>();

    private String mediaCode;
    private static int rtt = 300;
    private static int srtt = 300;
    private static double repeatRatio = 0;
    private int throttleSendBwLevel = BrtEngineConfig.brtThrottleDuration;
    private int sliceSendInterval = 1;
    private boolean isMultiFactor;
    private boolean isMultiRepeatRatio;
    private static BrtStatusListener statusListener;
    private static BrtResendStatListener resendStatListener;
    private BrtEngineMetric brtEngineMetric;
    private ResendMetricInfo resendMetricInfo;

    public static void setStatusListener(BrtStatusListener l) {
        statusListener = l;
    }

    public static void setBrtResendStatListener(BrtResendStatListener l) {
        resendStatListener = l;
    }

    public static int getRtt() {
        return rtt;
    }

    public static void setRtt(int rtt) {
        srtt = (int) ((ALPHA * srtt) + ((1 - ALPHA) * rtt));
        ResendStat.rtt = rtt;
    }

    ResendStat(String mediaCode, BrtEngineMetric engineMetric) {
        this.mediaCode = mediaCode;
        this.isMultiFactor = isMulti(BrtEngineConfig.rtoMultiFactor, 1.0D);
        this.isMultiRepeatRatio = isMulti(repeatRatio, 0D);
        brtEngineMetric = engineMetric;
        resendMetricInfo = new ResendMetricInfo();
    }

    void statSliceGroup(LiveSliceGroup sliceGroup, BrtConnect.RemoveGrpEvent event) {
        int lostPercent = sliceGroup.getFirstLostPercent();
        if (resendMetricInfo.min1stLostPercent > lostPercent) {
            resendMetricInfo.min1stLostPercent = lostPercent;
        }
        if (resendMetricInfo.max1stLostPercent < lostPercent) {
            resendMetricInfo.max1stLostPercent = lostPercent;
        }

        int lostCount = sliceGroup.getFirstLostCount();
        if (resendMetricInfo.min1stLostSlices > lostCount) {
            resendMetricInfo.min1stLostSlices = lostCount;
        }
        if (resendMetricInfo.max1stLostSlices < lostCount) {
            resendMetricInfo.max1stLostSlices = lostCount;
        }

        if (lostCount > 0) {
            if (resendMetricInfo.lost1stGroupCnt++ == 0) {
                resendMetricInfo.avg1stLostPercent = lostPercent;
                resendMetricInfo.avg1stLostSlices = lostCount;
            } else {
                resendMetricInfo.avg1stLostPercent = avg(resendMetricInfo.avg1stLostPercent, lostPercent, resendMetricInfo.lost1stGroupCnt);
                resendMetricInfo.avg1stLostSlices = avg(resendMetricInfo.avg1stLostSlices, lostCount, resendMetricInfo.lost1stGroupCnt);
            }
        }

        lostPercent = sliceGroup.getLostPercent();
        if (resendMetricInfo.min2ndLostPercent > lostPercent) {
            resendMetricInfo.min2ndLostPercent = lostPercent;
        }
        if (resendMetricInfo.max2ndLostPercent < lostPercent) {
            resendMetricInfo.max2ndLostPercent = lostPercent;
        }

        lostCount = sliceGroup.getLostCount();
        if (resendMetricInfo.min2ndLostSlices > lostCount) {
            resendMetricInfo.min2ndLostSlices = lostCount;
        }
        if (resendMetricInfo.max2ndLostSlices < lostCount) {
            resendMetricInfo.max2ndLostSlices = lostCount;
        }

        if (lostCount > 0) {
            if (resendMetricInfo.lost2ndGroupCnt++ == 0) {
                resendMetricInfo.avg2ndLostPercent = lostPercent;
                resendMetricInfo.avg2ndLostSlices = lostCount;
            } else {
                resendMetricInfo.avg2ndLostPercent = avg(resendMetricInfo.avg2ndLostPercent, lostPercent, resendMetricInfo.lost2ndGroupCnt);
                resendMetricInfo.avg2ndLostSlices = avg(resendMetricInfo.avg2ndLostSlices, lostCount, resendMetricInfo.lost2ndGroupCnt);
            }
        }

        switch (event) {
            case DROP_OVERDUE:
                resendMetricInfo.overdueGroupCnt++;
                break;
            case DROP_TIMEOUT:
                resendMetricInfo.timeoutGroupCnt++;
                break;
            case DROP_OVERCACHE:
                resendMetricInfo.overCacheGroupCnt++;
                break;
        }

        resendMetricInfo.rtt = rtt;

        BrtResendStatListener l = resendStatListener;
        if (l != null) {
            l.onStatUpdate(resendMetricInfo);
        }
    }

    void setHBRtt(int rtt) {
        //this.rtt = rtt;
    }

    void setThrottleSendBwLevel(int throttleSendBwLevel) {
        this.throttleSendBwLevel = throttleSendBwLevel;
        this.sliceSendInterval = BrtEngineConfig.brtThrottleDuration / throttleSendBwLevel;
        if (this.sliceSendInterval <= 0)
            this.sliceSendInterval = 1;
    }

    int getRto(int groupSeq, boolean isInit, int slices) {
        List<ResendInfo> list = resendInfos.get(groupSeq);
        int rto;
        if (list == null) {
            rto = calRto(slices);
            if (!isInit)
                rto += srtt;
            if (isMultiRepeatRatio)
                rto += (int) (rto * repeatRatio);
        } else {
            ResendInfo resendInfo = list.get(list.size() - 1);
            rto = resendInfo.rto;
            if (!isInit)
                rto += srtt;
            if (resendInfo.isMultiRepeatRatio)
                rto += (int) (rto * resendInfo.repeatRatio);
        }

        if (isMultiFactor)
            rto = (int) (rto * BrtEngineConfig.rtoMultiFactor);

        return rto < BrtEngineConfig.minRTO ? BrtEngineConfig.minRTO :
                (rto > BrtEngineConfig.maxRTO ? BrtEngineConfig.maxRTO : rto);
    }

    long getLastResendTime(int groupSeq, long groupBeginTime) {
        List<ResendInfo> list = resendInfos.get(groupSeq);
        if (list == null || list.isEmpty()) {
            return groupBeginTime;
        }

        ResendInfo resendInfo = list.get(list.size() - 1);
        return resendInfo.endTime;
    }

    void recordRepeat(int groupSeq, int slice) {
        List<ResendInfo> list = resendInfos.get(groupSeq);
        if (list == null || list.isEmpty()) {
            return;
        }

        ResendInfo resendInfo = list.get(list.size() - 1);
        resendInfo.repeatSlices++;
    }

    int recordResend(int groupSeq, long sendTime, boolean isInitial, long sliceGroupLifeDur, int resendCfgDur,
                      long sliceGroupLastCheckDur, int resendChkCfgDur, boolean is1by1, int resendSlices,
                      int totalResendGroups, int totalResendPackages, int totalResendSlices,
                      int groupSize) {
        long endTime = System.currentTimeMillis();
        ResendInfo resendInfo = new ResendInfo();
        resendInfo.isInitial = isInitial;
        resendInfo.sliceGroupLifeDur = sliceGroupLifeDur;
        resendInfo.resendCfgDur = resendCfgDur;
        resendInfo.sliceGroupLastCheckDur = sliceGroupLastCheckDur;
        resendInfo.resendChkCfgDur = resendChkCfgDur;
        resendInfo.endTime = endTime;
        resendInfo.sendTime = sendTime;
        resendInfo.is1by1 = is1by1;
        resendInfo.resendSlices = resendSlices;
        resendInfo.totalResendGroups = totalResendGroups;
        resendInfo.totalResendPackages = totalResendPackages;
        resendInfo.totalResendSlices = totalResendSlices;
        resendInfo.groupSize = groupSize;

        List<ResendInfo> list = resendInfos.get(groupSeq);
        if (list == null) {
            list = new ArrayList<>();
            resendInfos.put(groupSeq, list);
            resendInfo.repeatRatio = repeatRatio;
        } else {
            int repeatCnt = 0;
            int resendCnt = 0;
            for (ResendInfo info : list) {
                repeatCnt += info.repeatSlices;
                resendCnt += info.resendSlices;
            }

            if (resendCnt == 0 || repeatCnt == 0) {
                if (isMultiRepeatRatio) {
                    resendInfo.repeatRatio = repeatRatio * (1 - ALPHA);
                    resendInfo.isMultiRepeatRatio = isMulti(resendInfo.repeatRatio, 0D);
                }
            } else {
                resendInfo.repeatRatio = repeatRatio * (1 - ALPHA) + ALPHA * repeatCnt / resendCnt;
                resendInfo.isMultiRepeatRatio = isMulti(resendInfo.repeatRatio, 0D);
            }
        }

        resendInfo.rto = calRto(resendSlices);

        list.add(resendInfo);

        return list.size();
        // logger.error("group[%d] resendNum[%d] sliceGroupLastCheckDur:[%d] resendChkCfgDur:[%d]", groupSeq, list.size(), sliceGroupLastCheckDur, resendChkCfgDur);
    }

    // todo 是否考虑的全部重发的情况
    void removeAndLogResendInfo(BrtConnect.RemoveGrpEvent removeGrpEvent, int groupSeq, long groupBeginTime, int availableCnt, int slices, String result) {
        // logger.error("RemoveGroup[%d] Type[%s]", groupSeq, removeGrpEvent.name());
        List<ResendInfo> list = resendInfos.remove(groupSeq);
        int resendRounds = list == null ? 0 : list.size();
        long costTime = System.currentTimeMillis() - groupBeginTime;
        int totalRepeatSlices = 0;
        int totalResendSlices = 0;
        int firstResendSlices = 0;
        StringBuilder builder = null;
        if (list != null && list.size() > 0) firstResendSlices = list.get(0).resendSlices;
        switch (removeGrpEvent) {
            case COMPLETE:
                brtEngineMetric.finishGroup(costTime, slices, firstResendSlices, resendRounds);
                break;
            case DROP_TIMEOUT:
            case DROP_OVERDUE:
            case DROP_OVERCACHE:
                brtEngineMetric.dropGroup(removeGrpEvent, slices, firstResendSlices, resendRounds, costTime);
                break;
        }

        if (logger.isInfo() || statusListener != null)
            builder = new StringBuilder();

        if (list != null) {
            if (builder != null)
                builder.append("---RS   Seq    ResendTime    ResendCostTime   SinceLastResend     isOneByOne   ResendSlices    SliceRepeat    RepeatRatio    ReceivedSinceLastResend    RTT     SRTT    throttleSendBwLevel    RTO  isInitial     1stCondition         2ndCondition       groupSize  totalResendGroups  totalResendPackages  totalResendSlices\n");

            int lastResendSlices = slices;
            long lastResendTime = groupBeginTime;
            int i = 1;
            for (ResendInfo resendInfo : list) {
                if (builder != null)
                    builder.append(String.format("---RS   %3d %13d %17d %17d %14s %14d %14d %14.2f %12d %20d %8d %16d %12d %10s %20s %20s %10d %18d %20d %18d\n",
                            i++,
                            resendInfo.sendTime - groupBeginTime,
                            resendInfo.endTime - resendInfo.sendTime,
                            resendInfo.sendTime - lastResendTime,
                            resendInfo.is1by1,
                            resendInfo.resendSlices,
                            resendInfo.repeatSlices,
                            resendInfo.repeatRatio,
                            lastResendSlices - resendInfo.resendSlices,
                            rtt,
                            srtt,
                            throttleSendBwLevel,
                            resendInfo.rto,
                            resendInfo.isInitial,
                            String.format("%d(%d-%d)",
                                    resendInfo.sliceGroupLifeDur - resendInfo.resendCfgDur,
                                    resendInfo.sliceGroupLifeDur,
                                    resendInfo.resendCfgDur),
                            String.format("%d(%d-%d)", resendInfo.sliceGroupLastCheckDur - resendInfo.resendChkCfgDur,
                                    resendInfo.sliceGroupLastCheckDur,
                                    resendInfo.resendChkCfgDur),
                            resendInfo.groupSize,
                            resendInfo.totalResendGroups,
                            resendInfo.totalResendPackages,
                            resendInfo.totalResendSlices
                    ));

                lastResendSlices = resendInfo.resendSlices;
                lastResendTime = resendInfo.sendTime;
                totalRepeatSlices += resendInfo.repeatSlices;
                totalResendSlices += resendInfo.resendSlices;
            }
        } else if (builder != null) {
            builder.append("---RS   No resend.\n");
        }

        if (builder != null) {
            builder.append("---RS   \n");
            String msg = String.format("\n---RS   Channel[%s], Group[%d], Available/Size[%d/%d], ResendStrategy[cfg=%s(if %d>%d = %s),%s], Repeat/Resend[%d/%d], ResendCnt[%d], CostTime[%ds], RepeatRatio[%.2f], MaxRTO[%d], MinRTO[%d], RTOMulti[%.2f], Speed:{%.2fGroups/S, %.2fSlice/S, %dKB/S} %s\n%s",
                    mediaCode,
                    groupSeq,
                    availableCnt,
                    slices,
                    BrtEngineConfig.brtCheckResendStrategy,
                    slices,
                    BrtEngineConfig.brtResendStrategyGroupCnt,
                    BrtEngineConfig.BRT_CHECK_RESEND_STRATEGY_SLOW,
                    BrtEngineConfig.brtResendStrategy,
                    totalRepeatSlices,
                    totalResendSlices,
                    resendRounds,
                    costTime / 1000,
                    repeatRatio,
                    BrtEngineConfig.maxRTO,
                    BrtEngineConfig.minRTO,
                    BrtEngineConfig.rtoMultiFactor,
                    1000.0 / costTime,
                    availableCnt * 1000.0 / costTime,
                    availableCnt * LivePacket.MAX_PACKET_LENGTH_VER3 / costTime,
                    result, builder.toString());
            if (logger.isInfo())
                logger.info(msg);

            BrtStatusListener l = statusListener;
            if (l != null) {
                l.onStatus(msg);
            }
        }

        if (totalRepeatSlices > 0 && totalResendSlices > 0) {
            repeatRatio = repeatRatio * (1 - ALPHA) + ALPHA * totalRepeatSlices / totalResendSlices;
            isMultiRepeatRatio = isMulti(repeatRatio, 0);
        } else if (isMultiRepeatRatio) {
            repeatRatio = repeatRatio * (1 - ALPHA);
            isMultiRepeatRatio = isMulti(repeatRatio, 0);
        }

        if (resendMetricInfo.minResendRounds > resendRounds) {
            resendMetricInfo.minResendRounds = resendRounds;
        }
        if (resendMetricInfo.maxResendRounds < resendRounds) {
            resendMetricInfo.maxResendRounds = resendRounds;
        }

        if (resendMetricInfo.minResendSlices > totalResendSlices) {
            resendMetricInfo.minResendSlices = totalResendSlices;
        }
        if (resendMetricInfo.maxResendSlices < totalResendSlices) {
            resendMetricInfo.maxResendSlices = totalResendSlices;
        }

        if (resendMetricInfo.minRepeatSlices > totalRepeatSlices) {
            resendMetricInfo.minRepeatSlices = totalRepeatSlices;
        }
        if (resendMetricInfo.maxRepeatSlices < totalRepeatSlices) {
            resendMetricInfo.maxRepeatSlices = totalRepeatSlices;
        }

        if (resendRounds > 0)  {
            if (resendMetricInfo.resendGroupCnt++ == 0) {
                resendMetricInfo.avgResendRounds = resendRounds;
                resendMetricInfo.avgResendSlices = totalResendSlices;
            } else {
                resendMetricInfo.avgResendRounds = avg(resendMetricInfo.avgResendRounds, resendRounds, resendMetricInfo.resendGroupCnt);
                resendMetricInfo.avgResendSlices = avg(resendMetricInfo.avgResendSlices, totalResendSlices, resendMetricInfo.resendGroupCnt);
            }
        }

        if (totalRepeatSlices > 0) {
            if (resendMetricInfo.repeatGroupCnt++ == 0) {
                resendMetricInfo.avgRepeatSlices = totalRepeatSlices;
            } else {
                resendMetricInfo.avgRepeatSlices = avg(resendMetricInfo.avgRepeatSlices, totalRepeatSlices, resendMetricInfo.repeatGroupCnt);
            }
        }

        resendMetricInfo.groupCnt++;
    }

    private double avg(double avgVal, int newVal, int count) {
        return avgVal + (newVal - avgVal) / count;
    }

    private int calRto(int sendSlices) {
        return sliceSendInterval * sendSlices;
    }

    private boolean isMulti(double a, double b) {
        double diff = a - b;
        return diff > -0.001 || diff < 0.001;
    }

    private class ResendInfo {
        private long sendTime;
        private long endTime;
        private boolean isInitial;
        private long sliceGroupLifeDur;
        private long sliceGroupLastCheckDur;
        private int resendChkCfgDur;
        private int resendCfgDur;
        private boolean is1by1;
        private int resendSlices;
        private int groupSize;
        private int rto;
        private int repeatSlices;
        private double repeatRatio;
        private boolean isMultiRepeatRatio;
        private int totalResendGroups, totalResendPackages, totalResendSlices;
    }
}
