package com.ott.stream.rapid.agent;

import androidx.annotation.NonNull;

/**
 * Representation of a playback stream or a time shift stream
 */
public final class RapidAgentCatchupUri extends RapidAgentUri {
    /**
     * The start time of the epg in milliseconds since the Unix epoch.
     */
    private String epgStart;
    /**
     * The duration of the epg in seconds.
     */
    private String epgDuration;
    /**
     * Whether it is a time shift stream.
     */
    private boolean isTimeShift;
    /**
     * Offset seconds when time shift start. For example, it is 60 * 5 for 5 minutes ago.
     */
    private long timeShiftDelay;

    /**
     * Constructs a new instance
     *
     * @param streamId    the stream id
     * @param epgStart    the start time of the epg in milliseconds since the Unix epoch
     * @param epgDuration the duration of the epg in seconds
     */
    public RapidAgentCatchupUri(@NonNull String streamId, @NonNull String epgStart, @NonNull String epgDuration) {
        super(streamId);
        this.epgStart = epgStart;
        this.epgDuration = epgDuration;
    }

    /**
     * Constructs a new instance
     *
     * @param streamId       the stream id
     * @param epgStart       the start time of the epg in milliseconds since the Unix epoch
     * @param epgDuration    the duration of the epg in seconds
     * @param timeShiftDelay Offset seconds when time shift start. For example, it is 60 * 5 for 5 minutes ago
     */
    public RapidAgentCatchupUri(@NonNull String streamId, @NonNull String epgStart, @NonNull String epgDuration, long timeShiftDelay) {
        super(streamId);
        this.epgStart = epgStart;
        this.epgDuration = epgDuration;
        this.isTimeShift = true;
        this.timeShiftDelay = timeShiftDelay;
    }

    @Override
    public RapidAgentUri copySelf() {
        RapidAgentCatchupUri other;
        if (isTimeShift) {
            other = new RapidAgentCatchupUri(streamId, epgStart, epgDuration, timeShiftDelay);
        } else {
            other = new RapidAgentCatchupUri(streamId, epgStart, epgDuration);
        }
        copySelf(other);
        return other;
    }

    /**
     * Gets the start time of the epg in milliseconds since the Unix epoch
     *
     * @return the start time
     */
    public String getEpgStart() {
        return epgStart;
    }

    /**
     * Sets the start time of the epg in milliseconds since the Unix epoch
     *
     * @param epgStart the start time
     */
    public void setEpgStart(String epgStart) {
        this.epgStart = epgStart;
    }

    /**
     * Gets the duration of the epg in seconds
     *
     * @return the duration
     */
    public String getEpgDuration() {
        return epgDuration;
    }

    /**
     * Set the duration of the epg in seconds
     *
     * @param epgDuration the duration
     */
    public void setEpgDuration(String epgDuration) {
        this.epgDuration = epgDuration;
    }

    /**
     * Sets whether it is a time shift stream.
     *
     * @param timeShift whether it is a time shift stream
     */
    public void setTimeShift(boolean timeShift) {
        isTimeShift = timeShift;
    }

    /**
     * Returns whether it is a time shift stream.
     *
     * @return true if it is time shift stream
     */
    public boolean isTimeShift() {
        return isTimeShift;
    }

    /**
     * Sets the Offset seconds when time shift start. For example, it is 60 * 5 for 5 minutes ago.
     *
     * @param timeShiftDelay the offset seconds
     */
    public void setTimeShiftDelay(long timeShiftDelay) {
        this.timeShiftDelay = timeShiftDelay;
    }

    /**
     * Gets the Offset seconds when time shift start. For example, it is 60 * 5 for 5 minutes ago.
     *
     * @return the offset seconds
     */
    public long getTimeShiftDelay() {
        return timeShiftDelay;
    }

    @NonNull
    @Override
    public String toString() {
        return "RapidAgentCatchupUri{" +
                "streamId='" + streamId + '\'' +
                ", epgStart='" + epgStart + '\'' +
                ", epgDuration='" + epgDuration + '\'' +
                ", isTimeShift='" + isTimeShift + '\'' +
                ", timeShiftDelay='" + timeShiftDelay + '\'' +
                ", drmType='" + drmType + '\'' +
                ", customInfo='" + customInfo + '\'' +
                '}';
    }
}
