package com.ott.stream.rapid.agent;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.config.DeviceState;
import com.ott.stream.rapid.agent.config.ServerConfig;
import com.ott.stream.rapid.agent.prt.DefaultJniPrtListener;
import com.ott.stream.rapid.agent.prt.PrtEngine;
import com.ott.stream.rapid.agent.prt.PrtExternalDecrypt;
import com.ott.stream.rapid.agent.prt.PrtVersion;
import com.ott.stream.rapid.agent.prt.StreamInfoMgr;
import com.ott.stream.rapid.agent.utils.CommonUtil;
import com.ott.stream.rapid.agent.utils.LogUtil;
import com.ott.stream.rapid.agent.utils.NetworkManager;
import com.stream.prt.NetworkState;
import com.stream.prt.PrtEvent;
import com.stream.prt.utils.JniApiImpl;

import java.io.IOException;
import java.util.Map;

/**
 * Rapid Agent SDK
 */
public final class RapidAgentSDK {
    private static final String TAG = "RapidAgentSDK";

    /**
     * Initialization callback
     */
    public interface OnInitListener {
        /**
         * Invoked when it is finished
         *
         * @param success whether init successfully
         * @param message init sdk detail
         */
        void onInit(boolean success, String message);
    }

    /**
     * Update config callback
     */
    public interface OnUpdateListener {
        /**
         * Invoked when update error
         *
         * @param code    error code
         * @param message error message
         */
        void onUpdateError(int code, String message);
    }

    /**
     * RapidAgentSDK event callback
     */
    public interface OnPlayListener {
        /**
         * @hidden
         */
        default void onPrtStarted(RapidAgentCall call) {

        }

        /**
         * Invoked when receive new event
         *
         * @param event  event type
         * @param params special params of this event
         * @param desc   description of the event
         */
        void onPrtEvent(int event, String params, String desc);

        /**
         * Invoked when receive metric
         *
         * @param params metric detail
         */
        void onPrtMetric(Map<String, String> params);

        /**
         * Invoked after {@link RapidAgentSDK#setPrtEngineLog(boolean, int)}
         *
         * @param state state detail
         */
        void onPrtState(Map<String, String> state);

        /**
         * @hidden
         */
        default void onPrtProgress() {

        }

        /**
         * Invoked by prt engine to get player cache time
         *
         * @param channelId channel id
         * @return player cache time
         */
        default int getPrtPlayerCacheTime(int channelId) {
            return 0;
        }
    }

    private static class OnPlayListenerImpl implements OnPlayListener {
        private final OnPlayListener internalListener;

        public OnPlayListenerImpl(OnPlayListener internalListener) {
            this.internalListener = internalListener;
        }

        @Override
        public void onPrtEvent(int event, String params, String desc) {
            if (internalListener != null) {
                internalListener.onPrtEvent(event, params, desc);
            }
        }

        @Override
        public void onPrtMetric(Map<String, String> params) {
            if (internalListener != null) {
                internalListener.onPrtMetric(params);
            }
        }

        @Override
        public void onPrtState(Map<String, String> state) {
            if (internalListener != null) {
                internalListener.onPrtState(state);
            }
        }

        @Override
        public void onPrtProgress() {
            if (internalListener != null) {
                internalListener.onPrtProgress();
            }
        }

        @Override
        public int getPrtPlayerCacheTime(int channelId) {
            if (internalListener != null) {
                return internalListener.getPrtPlayerCacheTime(channelId);
            }
            return 0;
        }
    }

    /**
     * create channel fail
     */
    public static final int EVENT_CREATE_CHANNEL_FAIL = 0x900;
    /**
     * create channel timeout
     */
    public static final int EVENT_CREATE_CHANNEL_TIMEOUT = 0x901;
    /**
     * long time has no onMetric callback
     */
    public static final int EVENT_HAS_NO_METRIC = 0x902;

    /**
     * The http proxy has not read data for long time
     */
    public static final int EVENT_PROXY_NO_SIGNAL = PrtEvent.event_proxy_no_signa;
    /**
     * The http proxy fail to verify the caller identity
     */
    public static final int EVENT_PROXY_VERIFY = PrtEvent.event_proxy_verify;
    /**
     * Cant read data after change stream
     */
    public static final int EVENT_CHANGE_STREAM_NO_SIGNAL = PrtEvent.event_change_stream_no_signal;
    /**
     * Notify the player to restart after change stream
     */
    public static final int EVENT_CHANGE_STREAM_RESTART_PLAYER = PrtEvent.event_change_stream_restart_player;
    /**
     * Notify the app the prt server id has changed
     */
    public static final int EVENT_CHANGE_PRT_SERVER_ID = PrtEvent.event_change_prt_server_id;
    /**
     * Video buffer reach to max size
     */
    public static final int EVENT_VIDEO_BUFFER_REACH_MAX_SIZE = 0x2002;
    /**
     * The http proxy is ready and the player can start playing
     */
    public static final int EVENT_PROXY_READY = PrtEvent.event_front_start_player;
    /**
     * Notify app to create new channel because need to get channel info from tracker sometimes.
     *
     * @hidden
     */
    public static final int EVENT_FRONT_INFO_TO_PLAY = PrtEvent.event_front_info_to_play;
    /**
     * Fail to get index
     */
    public static final int EVENT_GET_INDEX_FAIL = /*PrtEvent.event_front_get_index_failed*/0x2007;
    /**
     * Notify the app that segment size and download time.
     */
    public static final int EVENT_SEGMENT_DOWNLOADED = PrtEvent.event_abr_info;
    /**
     * The connect token is expired
     */
    public static final int EVENT_CONNECT_TOKEN_EXPIRED = /*PrtEvent.event_connect_token_expired*/0x2101;
    /**
     * The index of live chan is mismatch to its type
     */
    public static final int EVENT_LIVE_INDEX_CHAN_MISMATCH = /*PrtEvent.event_live_index_chan_mismatch*/0x2102;
    /**
     * The block is disorder
     */
    public static final int EVENT_LIVE_MICRO_CHAN_DISORDER = /*PrtEvent.event_live_micro_chan_disorder*/0x2103;
    /**
     * Micro block stream is not available
     */
    public static final int EVENT_MB_NOT_AVAILABLE = /*PrtEvent.event_live_micro_restart_index_chan*/0x2104;

    /**
     * Gateway error: unknown
     */
    public static final int GATEWAY_ERROR_UNKNOWN = -1;

    /**
     * Gateway error: user token is expired
     */
    public static final int GATEWAY_ERROR_TOKEN_EXPIRED = 400;

    /**
     * The sdk is running on unknown device
     */
    public static final int DEVICE_PLATFORM_UNKNOWN = 0;
    /**
     * The sdk is running on box device
     */
    public static final int DEVICE_PLATFORM_BOX = 1;
    /**
     * The sdk is running on phone device
     */
    public static final int DEVICE_PLATFORM_PHONE = 2;

    /**
     * Ethernet connection
     */
    public static final int NETWORK_STATE_ETHERNET = 0;
    /**
     * Wifi connection
     */
    public static final int NETWORK_STATE_WIFI = 1;
    /**
     * Mobile connection
     */
    public static final int NETWORK_STATE_MOBILE = 2;
    /**
     * No network
     */
    public static final int NETWORK_STATE_DISCONNECTED = 3;
    /**
     * Unknown network state
     */
    public static final int NETWORK_STATE_UNKNOWN = 4;

    /**
     * default http client which use android system api
     */
    public static final int HTTP_CLIENT_TYPE_DEFAULT = 0;
    /**
     * ok http client
     */
    public static final int HTTP_CLIENT_TYPE_OKHTTP = 1;
    /**
     * pct http client
     */
    public static final int HTTP_CLIENT_TYPE_PCT = 2;

    private static volatile RapidAgentWorkThread sRapidAgentWorkThread;
    private static boolean sHasLoadLib = false;

    /**
     * set so path
     *
     * @param bikSoPath BikCore so path which default as null
     * @param prtSoPath prt engine so path which default as null
     */
    public static void loadLib(String bikSoPath, String prtSoPath) {
        if (sHasLoadLib) {
            return;
        }
        if (bikSoPath == null) {
            bikSoPath = "";
        }
        if (prtSoPath == null) {
            prtSoPath = "";
        }
        LogUtil.i(TAG, "init module bik='" + bikSoPath + "', prt='" + prtSoPath + "'.");
        JniApiImpl.init(bikSoPath, prtSoPath);
        sHasLoadLib = true;
    }

    /**
     * set the work path which the files created by the sdk save to
     *
     * @param path work path
     */
    public static void initWorkPath(@NonNull String path) {
        RapidAgentConstant.initWorkPath(path);
    }

    /**
     * Set device info, need to be invoked before init
     *
     * @param map device info
     */
    public static void setDeviceInfo(@NonNull Map<String, Object> map) {
        try {
            RapidAgentConstant.setDeviceType((String) map.get("deviceType"));
            RapidAgentConstant.setUserType((String) map.get("userType"));
            RapidAgentConstant.setBrand((String) map.get("brand"));
            RapidAgentConstant.setModel((String) map.get("model"));
        } catch (Exception e) {
            LogUtil.e(TAG, "setDeviceInfo error:" + e.getMessage());
        }
    }

    /**
     * Initialize RapidAgentSDK
     *
     * @param context   application context
     * @param platform  device platform, see DEVICE_PLATFORM_*
     * @param appName   application package name
     * @param vnoId     VNO ID represent a VNO client, and got after register on our website
     * @param secretKey a key used to encrypt content, and got after register on our website
     * @param vnoToken  VNO login token
     * @param userToken user token
     * @param uid       user id which obtain after register to business system.
     * @param serverUrl server url to init the sdk
     * @param listener  a listener that notify the app if the action is successful.
     */
    public static void init(Context context, int platform, @NonNull String appName, @NonNull String vnoId, @NonNull String secretKey,
                            @NonNull String vnoToken, @NonNull String userToken, @NonNull String uid, @NonNull String serverUrl, OnInitListener listener) {
        init(context, platform, appName, vnoId, null, secretKey, vnoToken, userToken, uid, serverUrl, listener);
    }

    /**
     * Initialize RapidAgentSDK
     *
     * @param context   application context
     * @param platform  device platform, see DEVICE_PLATFORM_*
     * @param appName   application package name
     * @param vnoId     VNO ID represent a VNO client, and got after register on our website
     * @param vnoTag    VNO TAG represent a VNO client, and got after register on our website
     * @param secretKey a key used to encrypt content, and got after register on our website
     * @param vnoToken  VNO login token
     * @param userToken user token
     * @param uid       user id which obtain after register to business system.
     * @param serverUrl server url to init the sdk
     * @param listener  a listener that notify the app if the action is successful.
     */
    public static void init(Context context, int platform, @NonNull String appName, @NonNull String vnoId, String vnoTag, @NonNull String secretKey,
                            @NonNull String vnoToken, @NonNull String userToken, @NonNull String uid, @NonNull String serverUrl, OnInitListener listener) {
        LogUtil.i(TAG, "start init sdk version " + sdkVersionName() + ", vno:" + vnoId + ", debug:" + BuildConfig.DEBUG);

        if (TextUtils.isEmpty(appName)) {
            if (listener != null) {
                listener.onInit(false, "app name is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(vnoId)) {
            if (listener != null) {
                listener.onInit(false, "vno id is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(secretKey)) {
            if (listener != null) {
                listener.onInit(false, "secret key is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(vnoToken)) {
            if (listener != null) {
                listener.onInit(false, "vno token is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(userToken)) {
            if (listener != null) {
                listener.onInit(false, "user token is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(uid)) {
            if (listener != null) {
                listener.onInit(false, "uid is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(serverUrl)) {
            if (listener != null) {
                listener.onInit(false, "server url is invalid");
            }
            return;
        }

        uninit();

        loadLib(null, null);
        if (sRapidAgentWorkThread != null) {
            sRapidAgentWorkThread.quitSafely();
            sRapidAgentWorkThread = null;
        }
        sRapidAgentWorkThread = new RapidAgentWorkThread(context, platform, appName, vnoId, vnoTag, secretKey, vnoToken, userToken, uid, serverUrl);
        sRapidAgentWorkThread.setOnInitListener(listener);
        sRapidAgentWorkThread.start();

        LogUtil.i(TAG, "waiting init work finished");
    }

    /**
     * uninitialize RapidAgentSDK
     */
    public static void uninit() {
        DeviceState.setLoginSuccess(false);
        ServerConfig.setChecksum(null);
    }

    /**
     * set update listener
     *
     * @param listener listener
     */
    public static void setOnUpdateListener(OnUpdateListener listener) {
        if (sRapidAgentWorkThread != null) {
            sRapidAgentWorkThread.setOnUpdateListener(listener);
        }
    }

    /**
     * Update the user token
     *
     * @param newUserToken new user token
     */
    public static void updateUserToken(String newUserToken) {
        LogUtil.i(TAG, "updateUserToken");
        RapidAgentConstant.setAccountToken(newUserToken);
        if (sRapidAgentWorkThread != null) {
            sRapidAgentWorkThread.attemptTriggerUpdate();
        }
    }

    /**
     * Sets the device network state, see NETWORK_STATE_*
     *
     * @param state network state
     */
    public static void setNetworkState(int state) {
        LogUtil.i(TAG, "setNetworkState " + state);
        switch (state) {
            case NETWORK_STATE_ETHERNET:
                PrtEngine.setNetworkState(NetworkState.ETHERNET, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_WIFI:
                PrtEngine.setNetworkState(NetworkState.WIFI, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_MOBILE:
                PrtEngine.setNetworkState(NetworkState.MOBILE, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_DISCONNECTED:
                PrtEngine.setNetworkState(NetworkState.DISCONNECTED, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_UNKNOWN:
                PrtEngine.setNetworkState(NetworkState.UNKNOWN, NetworkManager.getIpAddress());
                break;
        }
    }

    /**
     * Sets the fore/background when it is updated
     *
     * @param foreground it is on foreground now
     */
    public static void setForeBackgroundState(boolean foreground) {
        LogUtil.i(TAG, "setForeBackgroundState " + foreground);
        PrtEngine.setRunningMode(foreground ? 0 : 1);
    }

    /**
     * Sets prt engine log switch and period
     *
     * @param enable log switch
     * @param period log print period
     */
    public static void setPrtEngineLog(boolean enable, int period) {
        PrtEngine.setLog(enable, period);
    }

    /**
     * Starts a new stream
     *
     * @param rapidAgentUri  a uri with special info, see {@link RapidAgentUri}
     * @param onPlayListener listener
     * @return the id of the new stream
     * @throws IOException throw exception when error occur
     */
    public static synchronized RapidAgentCall startStream(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener) throws IOException {
        if (!DeviceState.isLoginSuccess()) {
            throw new IOException("must init sdk first");
        }
        return startupEngine(rapidAgentUri, onPlayListener);
    }

    /**
     * Stops the current steam
     *
     * @param call the target stream should stopped
     */
    public static synchronized void stopStream(@NonNull RapidAgentCall call) {
        call.shutdown();
    }

    /**
     * Sets the device network state, see HTTP_CLIENT_TYPE_*
     *
     * @param type http client type
     */
    public static void setHttpClientType(int type) {
        switch (type) {
            case HTTP_CLIENT_TYPE_DEFAULT:
                RapidAgentHttpClient.setHttpClientType(RapidAgentHttpClient.HttpClientType.Default);
                break;
            case HTTP_CLIENT_TYPE_OKHTTP:
                RapidAgentHttpClient.setHttpClientType(RapidAgentHttpClient.HttpClientType.OkHttp);
                break;
            case HTTP_CLIENT_TYPE_PCT:
                RapidAgentHttpClient.setHttpClientType(RapidAgentHttpClient.HttpClientType.PCT);
                break;
        }
    }

    /**
     * Sets external instance to decrypt the data which encrypted by app
     *
     * @param decrypt decrypt
     */
    public static void setExternalDecrypt(RapidAgentExternalDecrypt decrypt) {
        PrtExternalDecrypt.setExternalDecrypt(decrypt);
    }

    /**
     * Gets the sdk version name
     *
     * @return the sdk version name
     */
    public static String sdkVersionName() {
        return BuildConfig.VERSION_CODE + "_" + BuildConfig.GIT_HASH;
    }

    /**
     * Gets the engine version name
     *
     * @return the engine version name
     */
    public static String engineVersionName() {
        return PrtVersion.getVersionName();
    }

    private static synchronized RapidAgentCall startupEngine(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener) throws IOException {
        if (rapidAgentUri.isMicroBlock() && rapidAgentUri.getDrmType() == RapidAgentUri.DRM_TYPE_AES) {
            rapidAgentUri.setMicroBlock(false, 0);
            LogUtil.w(TAG, "startupEngine: mb is not support aes drm now");
        }
        RapidAgentCall call = new RapidAgentCall(rapidAgentUri);
        if (onPlayListener != null) {
            onPlayListener.onPrtStarted(call);
        }
        call.updateStatus(RapidAgentCall.Status.IDLE);
        if (rapidAgentUri instanceof RapidAgentLiveUri) {
            callEngine(rapidAgentUri, onPlayListener, call);
        } else {
            requestEngine(rapidAgentUri, onPlayListener, call);
        }
        return call;
    }

    private static synchronized void requestEngine(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener, RapidAgentCall call) throws IOException {
        LogUtil.i(TAG, "requestEngine agentUri=" + rapidAgentUri);
        call.updateStatus(RapidAgentCall.Status.REQUESTING);
        PrtEngine prtEngine = new PrtEngine();
        DefaultJniPrtListener jniPrtListener = new DefaultJniPrtListener();
        jniPrtListener.addPlayListener(new OnPlayListenerImpl(onPlayListener) {
            @Override
            public void onPrtEvent(int event, String params, String desc) {
                if (event == RapidAgentSDK.EVENT_FRONT_INFO_TO_PLAY) {
                    final Map<String, Object> frontInfo = CommonUtil.parseMapFromString(params);
                    PrtEngine.runOnWorkThread(() -> {
                        LogUtil.i(TAG, "start vod channel with front info");
                        call.updateStatus(RapidAgentCall.Status.REQUESTED);
                        rapidAgentUri.setFrontInfo(frontInfo);
                        try {
                            callEngine(rapidAgentUri, onPlayListener, call);
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        rapidAgentUri.setFrontInfo(null);
                    });
                } else {
                    if (onPlayListener != null) {
                        onPlayListener.onPrtEvent(event, params, desc);
                    }
                }
            }
        });
        jniPrtListener.addPlayListener(call);
        int channelId = prtEngine.startup(rapidAgentUri, jniPrtListener);
        call.setRequestEngine(channelId, prtEngine);
    }

    private static synchronized void callEngine(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener, RapidAgentCall call) throws IOException {
        LogUtil.i(TAG, "callEngine agentUri=" + rapidAgentUri);
        call.updateStatus(RapidAgentCall.Status.CALLING);
        PrtEngine prtEngine = new PrtEngine();
        DefaultJniPrtListener jniPrtListener = new DefaultJniPrtListener();
        jniPrtListener.addPlayListener(new OnPlayListenerImpl(onPlayListener));
        jniPrtListener.addPlayListener(call);
        int channelId = prtEngine.startup(rapidAgentUri, jniPrtListener);
        call.setCallEngine(channelId, prtEngine);
        if (channelId > 0) {
            call.updateStatus(RapidAgentCall.Status.CONNECTED);
            if (rapidAgentUri instanceof RapidAgentCatchupUri || rapidAgentUri instanceof RapidAgentVodUri) {
                LogUtil.i(TAG, "notify a dummy proxy ready event");
                jniPrtListener.onEvent(channelId, RapidAgentSDK.EVENT_PROXY_READY, null, null);
            }
            StreamInfoMgr.getStreamInfo(channelId).startupTime = call.getStartTime();
        }
    }

    static void triggerLogcat(int what, int extra) {
        if (sRapidAgentWorkThread != null) {
            sRapidAgentWorkThread.triggerLogcat(what, extra);
        }
    }
}
