package com.ott.stream.rapid.agent.config;

import android.text.TextUtils;
import android.util.ArrayMap;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.RapidAgentConstant;
import com.ott.stream.rapid.agent.player.PlayerConfig;
import com.ott.stream.rapid.agent.prt.PrtConfig;
import com.ott.stream.rapid.agent.utils.LogUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

public class ConfigParser {
    private static final String TAG = "ConfigParser";

    public static boolean parseConfig(String json, boolean ignoreNullJson) throws JSONException, ConfigParseException {
        if (TextUtils.isEmpty(json)) {
            throw new JSONException("decrypt fail: json is null");
        }

        JSONObject jsonObject;
        jsonObject = new JSONObject(json);
        if (jsonObject.has("code")) {
            int code = jsonObject.getInt("code");
            String msg = jsonObject.getString("msg");
            if (code != 0) {
                throw new ConfigParseException(code, msg, "config parse error, code : " + code + ", msg : " + msg);
            }
        }
        if (ignoreNullJson && !jsonObject.has("result")) {
            return false;
        }
        return parseConfig2(jsonObject.getString("result"), ignoreNullJson);
    }

    private static boolean parseConfig2(String json, boolean ignoreNullJson) throws JSONException {
        if (TextUtils.isEmpty(json) || TextUtils.equals(json, "null")) {
            if (ignoreNullJson) {
                return false;
            } else {
                throw new JSONException("parse fail: json is null");
            }
        }

        JSONObject jsonObject;
        jsonObject = new JSONObject(json);

        String checksum = jsonObject.getString("checksum");
        if (TextUtils.equals(checksum, ServerConfig.getChecksum())) {
            return false;
        }

        ServerConfig.setChecksum(checksum);
        ServerConfig.setVnoId(jsonObject.getString("vnoId"));

        String liveTurboJson = "";
        String catchupTurboJson = "";
        String vodTurboJson = "";
        if (jsonObject.has("turbos")) {
            String turbosJson = jsonObject.getString("turbos");
            JSONObject turbosJsonObj = new JSONObject(turbosJson);
            liveTurboJson = turbosJsonObj.getString("live");
            catchupTurboJson = turbosJsonObj.getString("catchup");
            vodTurboJson = turbosJsonObj.getString("vod");
        }

        try {
            PrtConfig.setLiveServerAddress(
                    parseServerAddress(jsonObject.getString("liveServerAddr"), liveTurboJson));
        } catch (JSONException e) {
            LogUtil.e(TAG, "invalid live tracker: " + e);
        }
        try {
            PrtConfig.setVodServerAddress(
                    parseServerAddress(jsonObject.getString("vodServerAddr"), vodTurboJson));
        } catch (JSONException e) {
            LogUtil.e(TAG, "invalid vod tracker: " + e);
        }
        try {
            PrtConfig.setCatchupServerAddress(
                    parseServerAddress(jsonObject.getString("catchupServerAddr"), catchupTurboJson));
        } catch (JSONException e) {
            LogUtil.e(TAG, "invalid catchup tracker: " + e);
        }

        if (jsonObject.has("sdkCfg")) {
            ConfigParser.parseSdkConfig(jsonObject.getString("sdkCfg"));
        } else {
            LogUtil.e(TAG, "using device local sdkCfg");
            ConfigParser.parseSdkConfig("");
        }
        if (jsonObject.has("playerCfg")) {
            ConfigParser.parsePlayerConfig(jsonObject.getString("playerCfg"));
        } else {
            LogUtil.e(TAG, "using device local playerCfg");
            ConfigParser.parsePlayerConfig("");
        }
        if (jsonObject.has("engineCfg")) {
            ConfigParser.parseEngineConfig(jsonObject.getString("engineCfg"));
        } else {
            LogUtil.e(TAG, "using device local engineCfg");
            ConfigParser.parseEngineConfig("");
        }

        return true;
    }

    private static ServerAddress parseServerAddress(String json, String turboJson) throws JSONException {
        ServerAddress address = new ServerAddress();
        String defaultPrt = "";
        String turboPrt = "";
        String tracker = "";

        JSONObject jsonObject = new JSONObject(json);
        try {
            defaultPrt = jsonObject.getString("defaultPrt");
            address.setDefaultPrtMap(parseAddressMap(defaultPrt));
        } catch (JSONException e) {
            address.setDefaultPrt(defaultPrt);
        }
        try {
            tracker = jsonObject.getString("tracker");
            address.setTrackerMap(parseAddressMap(tracker));
        } catch (JSONException e) {
            address.setTracker(tracker);
        }

        try {
            address.setTurboPrtMap(parseAddressMap(turboJson));
        } catch (JSONException e) {
            address.setTurboPrt(turboPrt);
        }
        return address;
    }

    private static Map<String, String> parseAddressMap(String json) throws JSONException {
        JSONArray jsonArray = new JSONArray(json);
        Map<String, String> map = new ArrayMap<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject item = jsonArray.getJSONObject(i);
            map.put(item.getString("serverAddr"), item.getString("serverId"));
        }
        return map;
    }

    public static void parseSdkConfig(String jsonData) throws JSONException {
        if (TextUtils.isEmpty(jsonData)) {
            jsonData = "{}";
        }
        parseSdkConfig(new JSONObject(jsonData));
    }

    public static void parseSdkConfig(JSONObject jsonObject) throws JSONException {
        Map<String, Object> map = parseMap(jsonObject);
        CommonConfig.setCommonOptions(map);
    }

    public static void parsePlayerConfig(String jsonData) throws JSONException {
        if (TextUtils.isEmpty(jsonData)) {
            jsonData = "{}";
        }
        parsePlayerConfig(new JSONObject(jsonData));
    }

    public static void parsePlayerConfig(JSONObject jsonObject) throws JSONException {
        Map<String, Map<String, Object>> map = new LinkedHashMap<>();
        Iterator<String> iterator = jsonObject.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            Object obj = jsonObject.get(key);
            if (obj instanceof JSONObject) {
                map.put(key, parseMap((JSONObject) obj));
            }
        }
        for (Map.Entry<String, Map<String, Object>> entry : map.entrySet()) {
            if (TextUtils.equals(entry.getKey(), "ijk_common")) {
                PlayerConfig.setIjkCommonOptions(entry.getValue());
            } else if (TextUtils.equals(entry.getKey(), "ijk_format")) {
                PlayerConfig.setIjkFormatOptions(entry.getValue());
            } else if (TextUtils.equals(entry.getKey(), "ijk_codec")) {
                PlayerConfig.setIjkCodecOptions(entry.getValue());
            } else if (TextUtils.equals(entry.getKey(), "ijk_sws")) {
                PlayerConfig.setIjkSwsOptions(entry.getValue());
            } else if (TextUtils.equals(entry.getKey(), "ijk_player")) {
                PlayerConfig.setIjkPlayerOptions(entry.getValue());
            }
        }
    }

    public static void parseEngineConfig(String jsonData) throws JSONException {
        if (TextUtils.isEmpty(jsonData)) {
            jsonData = "{}";
        }
        parseEngineConfig(new JSONObject(jsonData));
    }

    public static void parseEngineConfig(JSONObject jsonObject) throws JSONException {
        Map<String, Object> map = parseMap(jsonObject);
        PrtConfig.setDefaultOptions(map);
    }

    private static Map<String, Object> parseMap(JSONObject jsonObject) throws JSONException {
        Map<String, Object> map = new LinkedHashMap<>();
        Iterator<String> iterator = jsonObject.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            Object obj = jsonObject.get(key);
            if ((obj instanceof String) || (obj instanceof Long)) {
                map.put(key, obj);
            } else if ((obj instanceof Integer)) {
                int value = (int) obj;
                map.put(key, (long) value);
            } else if (obj instanceof JSONArray) {
                if (TextUtils.equals(key, "flashCfg")) {
                    map.putAll(parseFlashConfigs((JSONArray) obj));
                }
            } else if (obj instanceof JSONObject) {
                map.put(key, obj);
            } else {
                LogUtil.e(TAG, "unsupported content = " + obj);
            }
        }

        return map;
    }

    private static Map<String, Object> parseFlashConfigs(@NonNull JSONArray jsonArray) throws JSONException {
        int storageSize = RapidAgentConstant.getStorageSizeGb();
        Map<String, Object> map = new LinkedHashMap<>();
        int maxFlashPer = 50;
        int minFreeFlash = 100;
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject jsonObject = jsonArray.getJSONObject(i);
            int highRange = jsonObject.getInt("h");
            if (highRange > storageSize || i == jsonObject.length() - 1) {
                maxFlashPer = jsonObject.getInt("mfp");
                minFreeFlash = jsonObject.getInt("mff");
                break;
            }
        }
        map.put("maxFlashPer", maxFlashPer);
        map.put("minFreeFlash", minFreeFlash);
        return map;
    }

    public static void printAllConfig() {
        LogUtil.i(TAG, "=============start print all config=============");
        CommonConfig.printConfig();
        ServerConfig.printConfig();
        PlayerConfig.printConfig();
        PrtConfig.printConfig();
        LogUtil.i(TAG, "=============end print all config=============");
    }
}
