package com.ott.stream.rapid.agent.http;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.ott.stream.rapid.agent.utils.LogUtil;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;

public class DefaultHttpApi implements HttpApi {
    private static final String TAG = "DefaultAgentHttp";

    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];

    private final int connectTimeoutMillis;
    private final int readTimeoutMillis;
    private final boolean useCaches;
    private final boolean allowGzip;
    private final boolean followRedirects;

    public DefaultHttpApi(int connectTimeoutMillis, int readTimeoutMillis, boolean useCaches, boolean allowGzip, boolean followRedirects) {
        this.connectTimeoutMillis = connectTimeoutMillis;
        this.readTimeoutMillis = readTimeoutMillis;
        this.useCaches = useCaches;
        this.allowGzip = allowGzip;
        this.followRedirects = followRedirects;
    }

    @Override
    public String post(@NonNull String url, @Nullable Map<String, String> headers, @NonNull String body) throws IOException {
        return request(new URL(url), "POST", body.getBytes(StandardCharsets.UTF_8), headers);
    }

    @Override
    public String get(@NonNull String url, @Nullable Map<String, String> headers) throws IOException {
        return request(new URL(url), "GET", null, headers);
    }

    private String request(
            URL url,
            String httpMethod,
            @Nullable byte[] httpBody,
            Map<String, String> requestParameters)
            throws IOException {
        HttpURLConnection connection = makeConnection(url, httpMethod, httpBody, requestParameters);
        int responseCode = connection.getResponseCode();
        String responseMessage = connection.getResponseMessage();

        // Check for a valid response code.
        if (responseCode < 200 || responseCode > 299) {
            @Nullable InputStream errorStream = connection.getErrorStream();
            byte[] errorResponseBody;
            try {
                errorResponseBody =
                        errorStream != null ? toByteArray(errorStream) : EMPTY_BYTE_ARRAY;
            } catch (IOException e) {
                errorResponseBody = EMPTY_BYTE_ARRAY;
            }
            closeConnectionQuietly(connection);
            String errorMsg = responseCode + " " + responseMessage + " " + new String(errorResponseBody);
            LogUtil.e(TAG, "--> " + httpMethod + " Failed: " + errorMsg);
            throw new IOException(errorMsg);
        }

        StringBuilder responseStringBuilder = new StringBuilder();
        try {
            InputStream in = connection.getInputStream();
            if (isCompressed(connection)) {
                in = new GZIPInputStream(in);
            }
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(in))) {
                String line;
                while ((line = reader.readLine()) != null) {
                    responseStringBuilder.append(line);
                }
            }
        } catch (Exception e) {
            LogUtil.e(TAG, "--> " + httpMethod + " Failed: " + e);
            e.printStackTrace();
            throw new IOException(e);
        } finally {
            closeConnectionQuietly(connection);
        }

        LogUtil.i(TAG, "<-- "
                + responseCode
                + ' ' + responseStringBuilder.toString()
                + ' ' + url);

        Map<String, List<String>> fields = connection.getHeaderFields();
        for (Map.Entry<String, List<String>> entry : fields.entrySet()) {
            if ("Content-Type".equalsIgnoreCase(entry.getKey()) ||
                    "Transfer-Encoding".equalsIgnoreCase(entry.getKey()) ||
                    "Date".equalsIgnoreCase(entry.getKey())) {
                LogUtil.i(TAG, entry.getKey() + ": " + entry.getValue());
            }
        }

        return responseStringBuilder.toString();
    }

    private HttpURLConnection makeConnection(
            URL url,
            String httpMethod,
            @Nullable byte[] httpBody,
            Map<String, String> requestParameters)
            throws IOException {
        LogUtil.i(TAG, "--> " + httpMethod + " " + url);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setConnectTimeout(connectTimeoutMillis);
        connection.setReadTimeout(readTimeoutMillis);
        connection.setUseCaches(useCaches);

        if (requestParameters != null && !requestParameters.isEmpty()) {
            for (Map.Entry<String, String> property : requestParameters.entrySet()) {
                connection.setRequestProperty(property.getKey(), property.getValue());
            }
        }

        connection.setRequestProperty("Accept-Encoding", allowGzip ? "gzip" : "identity");
        connection.setInstanceFollowRedirects(followRedirects);
        connection.setDoOutput(httpBody != null);
        connection.setRequestMethod(httpMethod);

        for (Map.Entry<String, List<String>> entry : connection.getRequestProperties().entrySet()) {
            LogUtil.i(TAG, entry.getKey() + ": " + entry.getValue());
        }

        if (httpBody != null) {
            connection.setFixedLengthStreamingMode(httpBody.length);
            connection.connect();
            OutputStream os = connection.getOutputStream();
            os.write(httpBody);
            os.close();
        } else {
            connection.connect();
        }
        LogUtil.i(TAG, "--> END " + httpMethod);
        return connection;
    }

    /**
     * Closes the current connection quietly, if there is one.
     */
    private void closeConnectionQuietly(HttpURLConnection connection) {
        if (connection != null) {
            try {
                connection.disconnect();
            } catch (Exception e) {
                Log.e(TAG, "Unexpected error while disconnecting", e);
            }
        }
    }

    private static boolean isCompressed(HttpURLConnection connection) {
        String contentEncoding = connection.getHeaderField("Content-Encoding");
        return "gzip".equalsIgnoreCase(contentEncoding);
    }

    /**
     * Converts the entirety of an {@link InputStream} to a byte array.
     *
     * @param inputStream the {@link InputStream} to be read. The input stream is not closed by this
     *                    method.
     * @return a byte array containing all of the inputStream's bytes.
     * @throws IOException if an error occurs reading from the stream.
     */
    public static byte[] toByteArray(InputStream inputStream) throws IOException {
        byte[] buffer = new byte[1024 * 4];
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        int bytesRead;
        while ((bytesRead = inputStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, bytesRead);
        }
        return outputStream.toByteArray();
    }
}
