package com.ott.stream.rapid.agent.http;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.ott.stream.rapid.agent.utils.LogUtil;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

public class OkHttpApi implements HttpApi {
    private final OkHttpClient okHttpClient;

    public OkHttpApi(int connectTimeoutMillis, int readTimeoutMillis) {
        OkHttpLoggingInterceptor loggingInterceptor = new OkHttpLoggingInterceptor(message -> LogUtil.i("OkAgentHttp", message));
        loggingInterceptor.setLevel(OkHttpLoggingInterceptor.Level.BODY);
        okHttpClient = new OkHttpClient.Builder()
                .connectTimeout(connectTimeoutMillis, TimeUnit.MILLISECONDS)
                .readTimeout(readTimeoutMillis, TimeUnit.MILLISECONDS)
                .addInterceptor(loggingInterceptor)
                .build();
    }

    @Override
    public String post(@NonNull String url, @Nullable Map<String, String> headers, @NonNull String body) throws IOException {
        Request.Builder builder = new Request.Builder()
                .url(url)
                .post(RequestBody.create(null, body));
        if (headers != null && !headers.isEmpty()) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                builder.header(entry.getKey(), entry.getValue());
            }
        }
        try (Response response = okHttpClient.newCall(builder.build()).execute()) {
            ResponseBody responseBody = response.body();
            if (responseBody != null) {
                return responseBody.source().isOpen() ? responseBody.string() : null;
            }
            return null;
        }
    }

    @Override
    public String get(@NonNull String url, @Nullable Map<String, String> headers) throws IOException {
        Request.Builder builder = new Request.Builder()
                .url(url)
                .get();
        if (headers != null && !headers.isEmpty()) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                builder.header(entry.getKey(), entry.getValue());
            }
        }
        try (Response response = okHttpClient.newCall(builder.build()).execute()) {
            ResponseBody responseBody = response.body();
            if (responseBody != null) {
                return responseBody.source().isOpen() ? responseBody.string() : null;
            }
            return null;
        }
    }
}
