package com.ott.stream.rapid.agent.player;

import static com.ott.stream.rapid.agent.utils.CommonUtil.getStringOfRoConfig;
import static com.ott.stream.rapid.agent.utils.CommonUtil.parseInt;

import android.os.Handler;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.ott.stream.rapid.agent.RapidAgentConstant;
import com.ott.stream.rapid.agent.config.CommonConfig;
import com.ott.stream.rapid.agent.utils.LogUtil;
import com.google.android.exoplayer2.upstream.BandwidthMeter;
import com.google.android.exoplayer2.upstream.DataSource;
import com.google.android.exoplayer2.upstream.DataSpec;
import com.google.android.exoplayer2.upstream.TransferListener;

import org.json.JSONException;
import org.json.JSONObject;

import tv.danmaku.ijk.media.exo.DefaultBandwidthMeter;

public class RapidExoBandwidthMeter implements BandwidthMeter, TransferListener, SegmentDownloadedListener {
    private static final boolean ENABLE_DEBUG = RapidAgentConstant.getApplicationName().equals("com.ott.stream.rapid.agent.demo");

    private final DefaultBandwidthMeter internalBandwidthMeter;
    private final float defPercentile;
    private long currentBandwidth = 0;

    public RapidExoBandwidthMeter() {
        DefaultBandwidthMeter.Builder builder = new DefaultBandwidthMeter.Builder(null);
        internalBandwidthMeter = builder
                .setSlidingWindowMaxWeight(4000)
                .setSlideWindowPercentile(0.5f)
                .setInitialBitrateEstimate(1_000_000)
                .build();
        defPercentile = CommonConfig.getBandwidthEstimatePercent() / 0.7f;
    }

    private float getDebugPercentile() {
        String percentileString = getStringOfRoConfig("ijk.debug.bandwidth.percent", "100");
        int percentileValue = parseInt(percentileString, 100);
        return percentileValue / 100f;
    }

    @Override
    public long getBitrateEstimate() {
        float percentile = ENABLE_DEBUG ? getDebugPercentile() : 1.0f;
        currentBandwidth = (long) (internalBandwidthMeter.getBitrateEstimate() * defPercentile * percentile);
        return currentBandwidth;
    }

    @Nullable
    @Override
    public TransferListener getTransferListener() {
        return this;
    }

    @Override
    public void addEventListener(Handler eventHandler, EventListener eventListener) {
        internalBandwidthMeter.addEventListener(eventHandler, eventListener);
    }

    @Override
    public void removeEventListener(EventListener eventListener) {
        internalBandwidthMeter.removeEventListener(eventListener);
    }

    @Override
    public void onTransferInitializing(@NonNull DataSource source, @NonNull DataSpec dataSpec, boolean isNetwork) {

    }

    @Override
    public void onTransferStart(@NonNull DataSource source, @NonNull DataSpec dataSpec, boolean isNetwork) {

    }

    @Override
    public void onBytesTransferred(@NonNull DataSource source, @NonNull DataSpec dataSpec, boolean isNetwork, int bytesTransferred) {

    }

    @Override
    public void onTransferEnd(@NonNull DataSource source, @NonNull DataSpec dataSpec, boolean isNetwork) {

    }

    @Override
    public void onSegmentDownloaded(String param) {
        try {
            LogUtil.d("RapidBandwidthMeter", "onSegmentDownloaded " + param);
            JSONObject jsonObject = new JSONObject(param);
            int trackType = jsonObject.getInt("trackType");
            int dataSize = jsonObject.getInt("len");
            int elapsedTimeMs = jsonObject.getInt("useMs");
            if (trackType == 1/*TRACK_TYPE_VIDEO*/ || trackType == 2/*TRACK_TYPE_AUDIO_VIDEO_MIX*/) {
                internalBandwidthMeter.addSample(dataSize, elapsedTimeMs);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    @Override
    public long getCurrentBandwidth() {
        return currentBandwidth;
    }
}
