package com.ott.stream.rapid.agent.player;

import static com.ott.stream.rapid.agent.RapidAgentStatistic.CATEGORY_STARTUP;
import static com.ott.stream.rapid.agent.RapidAgentStatistic.CATEGORY_UNSPECIFIED;
import static com.ott.stream.rapid.agent.player.RapidAgentPlayer.STREAM_ERROR_MSG;
import static com.ott.stream.rapid.agent.utils.CommonUtil.parseChannelIdFromUrl;

import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_ACCELERATED_CODEC_FAIL;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_AUDIO_RENDERING_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_READ_STREAM_CHANGED;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START;

import android.net.Uri;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.RapidAgentStatistic;
import com.ott.stream.rapid.agent.config.CommonConfig;
import com.ott.stream.rapid.agent.prt.StreamInfoMgr;
import com.ott.stream.rapid.agent.utils.CommonUtil;
import com.ott.stream.rapid.player.DefaultPlayerStatistic;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Queue;

public final class RapidPlayerStatistic extends DefaultPlayerStatistic {
    private static final int DOWNLOAD_INFO_QUEUE_SIZE = 10;

    private static RapidAgentStatistic sRapidAgentStatistic;

    private int channelId;
    private int playerType;
    private boolean renderStarted = false;

    private final Queue<Map<String, Object>> downloadInfoQueue = new ArrayDeque<>(DOWNLOAD_INFO_QUEUE_SIZE);

    public static void setAgentStatistic(RapidAgentStatistic statistic) {
        sRapidAgentStatistic = statistic;
    }

    @Override
    public void setPlayerType(int playerType, int playerHash) {
        super.setPlayerType(playerType, playerHash);
        this.playerType = playerType;
    }

    @Override
    public void setDataSource(Uri uri) {
        super.setDataSource(uri);
        String path = uri.toString();
        channelId = parseChannelIdFromUrl(path);
    }

    @Override
    public void prepareAsync() {
        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        streamInfo.playerStartupTime = System.nanoTime();
        super.prepareAsync();
    }

    @Override
    public void onError(int what, int extra) {
        super.onError(what, extra);
        // engine error notify in PrtStatistic
        if (what != STREAM_ERROR_MSG) {
            int category = renderStarted ? CATEGORY_UNSPECIFIED : CATEGORY_STARTUP;
            sRapidAgentStatistic.notifyPlayerError(channelId, playerType, what, extra, "", category);
        }
    }

    @Override
    public void onInfo(int what, int extra, String info) {
        super.onInfo(what, extra, info);
        if (what == MEDIA_INFO_VIDEO_RENDERING_START || (what == MEDIA_INFO_AUDIO_RENDERING_START && extra == 1)) {
            renderStarted = true;
            StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
            streamInfo.playerReadyTime = System.nanoTime();
        } else if (what == MEDIA_INFO_ACCELERATED_CODEC_FAIL) {
            sRapidAgentStatistic.notifyPlayerError(channelId, playerType, -10000, 997, "no mediacodec", CATEGORY_UNSPECIFIED);
        } else if (what == MEDIA_INFO_READ_STREAM_CHANGED) {
            sRapidAgentStatistic.notifyAutoSwitchTrack(channelId, extra);
        }
    }

    @Override
    public void onCompletion() {
        super.onCompletion();
        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        if (streamInfo.isLive()) {
            sRapidAgentStatistic.notifyLiveCompleted(channelId, playerType);
        }
    }

    @Override
    public void onStatusChanged(@NonNull Map<String, String> params) {
        sRapidAgentStatistic.notifyPlayerStatus(channelId, params);
    }

    @Override
    public void startupPerformance(JSONObject jsonObject) {
        super.startupPerformance(jsonObject);
        try {
            sRapidAgentStatistic.notifyPlayerStartupTime(channelId, playerType, jsonObject);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void bufferingPerformance(int extra, long duration, String startReason, String endReason) {
        super.bufferingPerformance(extra, duration, startReason, endReason);
        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        if (streamInfo.isMicroBlock() && CommonConfig.getStatisticLevel() >= CommonConfig.STATISTIC_LEVEL_DEBUG) {
            sRapidAgentStatistic.notifyPlayerBuffering(channelId, playerType, extra, duration, startReason, getBufferingInfo(endReason));
        } else {
            sRapidAgentStatistic.notifyPlayerBuffering(channelId, playerType, extra, duration, startReason, endReason);
        }
    }

    @Override
    public void seekPerformance(long duration) {
        super.seekPerformance(duration);
//        sRapidAgentStatistic.notifySeekPerformance(channelId, playerType, duration);
    }

    @Override
    public void releasePerformance(long stopDuration, long releaseDuration, long lifecycleDuration) {
        super.releasePerformance(stopDuration, releaseDuration, lifecycleDuration);
//        sRapidAgentStatistic.notifyReleasePerformance(channelId, playerType, stopDuration, releaseDuration, lifecycleDuration);
    }

    @Override
    public void downloadPerformance(@NonNull Map<String, Object> params) {
        synchronized (downloadInfoQueue) {
            if (downloadInfoQueue.size() == DOWNLOAD_INFO_QUEUE_SIZE) {
                downloadInfoQueue.remove();
            }
            downloadInfoQueue.add(new HashMap<>(params));
        }
    }

    private String getBufferingInfo(String endReason) {
        int maxFrameCost = 0;
        Map<String, Object> maxBlockMap = null;
        ArrayList<Map<String, Object>> list;
        int totalSegmentCost = 0;
        int totalSegmentCount = 0;
        synchronized (downloadInfoQueue) {
            list = new ArrayList<>(downloadInfoQueue);
            downloadInfoQueue.clear();
        }
        int i = list.size() - 1;
        for (; i >= 0; i--) {
            Map<String, Object> params = list.get(i);
            int segmentCost = CommonUtil.parseInt(params.get("segment_cost"), 0);
            if (segmentCost > 1000) {
                break;
            }
        }
        for (; i >= 0; i--) {
            Map<String, Object> params = list.get(i);
            int frameCost = CommonUtil.parseInt(params.get("frame_cost"), 0);
            int segmentCost = CommonUtil.parseInt(params.get("segment_cost"), 0);
            if (maxFrameCost < frameCost) {
                maxFrameCost = frameCost;
                maxBlockMap = params;
            }
            totalSegmentCost += segmentCost;
            totalSegmentCount++;
        }
        if (maxBlockMap != null) {
            int frameCost = CommonUtil.parseInt(maxBlockMap.get("frame_cost"), 0);
            int segmentCost = CommonUtil.parseInt(maxBlockMap.get("segment_cost"), 0);
            int maxBlockCost = CommonUtil.parseInt(maxBlockMap.get("max_block_cost"), 0);
            int readVideoBlock = CommonUtil.parseInt(maxBlockMap.get("read_block_count"), 0);
            int dropVideoBlock = CommonUtil.parseInt(maxBlockMap.get("drop_block_count"), 0);
            int minVideoBlock = CommonUtil.parseInt(maxBlockMap.get("min_block_count"), 0);
            int maxVideoBlock = CommonUtil.parseInt(maxBlockMap.get("max_block_count"), 0);
            int avgSegmentCost = totalSegmentCost / totalSegmentCount;

            return String.format(Locale.US, "{end:%s,fc:%s,sc:%s,ac:%d,mc:%s,vr:%s,ve:%s,min:%d,max:%s}",
                    endReason, frameCost, segmentCost, avgSegmentCost, maxBlockCost, readVideoBlock, dropVideoBlock, minVideoBlock, maxVideoBlock);
        }

        return endReason;
    }
}
