package com.ott.stream.rapid.agent.utils;

import static com.ott.stream.rapid.agent.player.RapidAgentPlayer.PLAYER_TYPE_AMP;
import static com.ott.stream.rapid.agent.player.RapidAgentPlayer.PLAYER_TYPE_EXO;
import static com.ott.stream.rapid.agent.player.RapidAgentPlayer.PLAYER_TYPE_IJK;

import android.net.Uri;
import android.os.Environment;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;

public final class CommonUtil {
    public static int parseInt(Object obj, int defaultValue) {
        return (int) parseLong(obj, defaultValue);
    }

    public static long parseLong(Object obj, int defaultValue) {
        try {
            if (obj instanceof Long) {
                return (long) obj;
            } else if (obj instanceof Integer) {
                return (int) obj;
            } else if (obj instanceof Short) {
                return (short) obj;
            } else if (obj instanceof Byte) {
                return (byte) obj;
            } else if (obj instanceof String) {
                return Long.parseLong((String) obj);
            }
        } catch (Exception e) {
            return defaultValue;
        }
        return defaultValue;
    }

    public static float parseFloat(Object obj, float defaultValue) {
        try {
            if (obj instanceof Float) {
                return (float) obj;
            } else if (obj instanceof String) {
                return Float.parseFloat((String) obj);
            }
        } catch (Exception e) {
            return defaultValue;
        }
        return defaultValue;
    }

    public static String getPlayerTypeString(int type) {
        if (type == PLAYER_TYPE_IJK) {
            return "ijk";
        } else if (type == PLAYER_TYPE_EXO) {
            return "exo";
        } else if (type == PLAYER_TYPE_AMP) {
            return "amp";
        } else {
            return "oem";
        }
    }

    public static String parseServerId(String msg) {
        String result = "";
        if (TextUtils.isEmpty(msg)) {
            return result;
        }
        String[] array = msg.split("&");
        for (String item : array) {
            if (item.startsWith("desc=")) {
                return item.substring(5);
            }
        }
        return result;
    }

    public static String[] parseHostArray(Object obj) {
        try {
            String hosts = (String) obj;
            String[] array = hosts.split(",");
            for (int i = 0; i < array.length; i++) {
                if (!array[i].startsWith("http")) {
                    array[i] = "http://" + array[i];
                }
            }
            return array;
        } catch (Exception e) {
            return new String[0];
        }
    }

    public static Map<String, Object> parseMapFromString(String input) {
        Map<String, Object> output = new LinkedHashMap<>();
        if (TextUtils.isEmpty(input)) {
            return output;
        }
        String[] array = input.split("&");
        for (String item : array) {
            if (TextUtils.isEmpty(item)) {
                continue;
            }
            String[] map = item.split("=");
            if (map.length == 2) {
                output.put(map[0], map[1]);
            }
        }
        return output;
    }

    public static String getStringOfRoConfig(String cfgName, String defValue) {
        Object result = invoke("android.os.SystemProperties", "get", new Class<?>[]{String.class, String.class}, cfgName, defValue);
        if (result instanceof String) {
            return (String) result;
        }
        return defValue;
    }

    public static Object invoke(String className, String methodName, Class<?>[] parameterTypes, Object... args) {
        Object value = null;

        try {
            Class<?> clz = Class.forName(className);
            Method method = clz.getDeclaredMethod(methodName, parameterTypes);
            value = method.invoke(clz, args);
        } catch (NoSuchMethodException | IllegalAccessException | IllegalArgumentException |
                 InvocationTargetException | SecurityException | ClassNotFoundException e) {
            LogUtil.e("CommonUtils", "invoke error:" + e.getMessage());
        } catch (Exception e) {
            LogUtil.e("CommonUtils", "unknown Exception in invoke:" + e.getMessage());
        }
        return value;
    }

    public static int parseChannelIdFromUrl(@NonNull String url) {
        Uri uri = Uri.parse(url);
        int channelId = -1;

        if ("http".equals(uri.getScheme())) {
            String path = uri.getPath();
            if (path != null) {
                int end = path.indexOf('/', 1);
                channelId = parseInt(path.substring(1, end), -1);
            }
        } else if ("mls".equals(uri.getScheme())) {
            channelId = parseInt(uri.getQueryParameter("channel"), -1);
        }
        if (channelId < 0) {
            throw new IllegalArgumentException("can not parse channel id from " + url);
        }
        return channelId;
    }

    public static byte[] getBytes(char[] chars, int length) {
        Charset cs = StandardCharsets.UTF_8;
        CharBuffer cb = CharBuffer.allocate(length);
        cb.put(chars, 0, length);
        cb.flip();
        ByteBuffer bb = cs.encode(cb);
        return bb.array();
    }

    public static boolean externalStorageAvailable() {
        try {
            return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
        } catch (Exception e) {
            return false;
        }
    }
}
