package com.ott.stream.rapid.agent.utils;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Calendar;
import java.util.Locale;
import java.util.Random;
import java.util.UUID;

public class SPUtil {
    private static final String FILE_NAME = "RapidAgentSDK";
    private static final String DEVICE_ID = "device_id";
    private static final String ANDROID_ID = "android_id";
    private static final String CID = "cid";
    private static final String LOG_DATE = "log_date";
    private static final String RAPID_METRIC_HOST = "rapid_metric_host";
    private static final String PCT_METRIC_HOST = "pct_metric_host";
    private static final String PCT_LOG_SERVER = "pct_log_server";
    private static final String MB_IDEAL_PACKET = "mb_ideal_packet";
    private static final String MB_BANDWIDTH = "mb_bandwidth";
    private static final String H264_MEDIACODEC_RANK = "h264_mediacodec_rank";
    private static final String H265_MEDIACODEC_RANK = "h265_mediacodec_rank";

    private static SharedPreferences sSharedPreferences;

    public static void init(@Nullable Context context) {
        if (context == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : init");
            return;
        }
        if (sSharedPreferences != null) {
            return;
        }
        sSharedPreferences = context.getSharedPreferences(FILE_NAME, Activity.MODE_PRIVATE);
    }

    @NonNull
    public static String getDeviceId() {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getDeviceId");
            return generateDeviceId();
        }
        String deviceId = sSharedPreferences.getString(DEVICE_ID, null);
        if (!TextUtils.isEmpty(deviceId)) {
            return deviceId;
        }
        deviceId = generateDeviceId();
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putString(DEVICE_ID, deviceId);
        editor.apply();

        return deviceId;
    }

    private static String generateDeviceId() {
        String mac = NetworkManager.getInstance().getUserMac();
        if (!TextUtils.isEmpty(mac)) {
            return mac;
        }
        String androidId = NetworkManager.getInstance().getAndroidId();
        if (!TextUtils.isEmpty(androidId)) {
            return androidId;
        }
        return UUID.randomUUID().toString().replace("-", "");
    }

    @NonNull
    public static String getAndroidId() {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getAndroidId");
            return generateAndroidId();
        }
        String androidId = sSharedPreferences.getString(ANDROID_ID, null);
        if (!TextUtils.isEmpty(androidId)) {
            return androidId;
        }
        androidId = generateAndroidId();
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putString(ANDROID_ID, androidId);
        editor.apply();

        return androidId;
    }

    private static String generateAndroidId() {
        String androidId = NetworkManager.getInstance().getAndroidId();
        if (!TextUtils.isEmpty(androidId)) {
            return androidId;
        }
        return generateRandomAndroidId();
    }

    private static String generateRandomAndroidId() {
        Random random = new Random();
        byte[] androidIdBytes = new byte[8];
        random.nextBytes(androidIdBytes);
        StringBuilder stringBuilder = new StringBuilder();
        for (byte macAddressByte : androidIdBytes) {
            stringBuilder.append(String.format("%02X", macAddressByte));
        }
        return stringBuilder.toString();
    }

    public static long getCID(String did) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getCID");
            return ((long) did.hashCode() << 32) | timestampFrom2023();
        }
        long cid = sSharedPreferences.getLong(CID, 0);
        if (cid != 0) {
            return cid;
        }
        cid = ((long) did.hashCode() << 32) | timestampFrom2023();
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putLong(CID, cid);
        editor.apply();

        return cid;
    }

    private static int timestampFrom2023() {
        long current = System.currentTimeMillis();
        long relate2023 = current - 1672502400000L;//1672502400000L is 2023-01-01 00:00:00;
        return (int) (relate2023 / 100);
    }

    public static int getLogTimesOfToday() {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getLogTimesOfToday");
            return -1;
        }
        int year = Calendar.getInstance().get(Calendar.YEAR);
        int month = Calendar.getInstance().get(Calendar.MONTH) + 1;
        int day = Calendar.getInstance().get(Calendar.DAY_OF_MONTH);
        String key = String.format(Locale.US, "%s_%d_%d_%d", LOG_DATE, year, month, day);
        return sSharedPreferences.getInt(key, 0);
    }

    public static void setLogTimesOfToday(int times) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setLogTimesOfToday");
            return;
        }
        int year = Calendar.getInstance().get(Calendar.YEAR);
        int month = Calendar.getInstance().get(Calendar.MONTH) + 1;
        int day = Calendar.getInstance().get(Calendar.DAY_OF_MONTH);
        String key = String.format(Locale.US, "%s_%d_%d_%d", LOG_DATE, year, month, day);
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putInt(key, times);
        editor.apply();
    }

    @NonNull
    public static String getRapidMetricHost(@NonNull String dflValue) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getRapidMetricHost");
            return dflValue;
        }
        String host = sSharedPreferences.getString(RAPID_METRIC_HOST, dflValue);
        if (TextUtils.isEmpty(host)) {
            return dflValue;
        }

        return host;
    }

    public static void setRapidMetricHost(@NonNull String host) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setRapidMetricHost");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putString(RAPID_METRIC_HOST, host);
        editor.apply();
    }

    @NonNull
    public static String getPCTMetricHost(@NonNull String dflValue) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getPCTMetricHost");
            return dflValue;
        }
        String host = sSharedPreferences.getString(PCT_METRIC_HOST, dflValue);
        if (TextUtils.isEmpty(host)) {
            return dflValue;
        }

        return host;
    }

    public static void setPCTMetricHost(@NonNull String host) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setPCTMetricHost");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putString(PCT_METRIC_HOST, host);
        editor.apply();
    }

    @NonNull
    public static String getPCTLogServer(@NonNull String dflValue) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getPCTLogServer");
            return dflValue;
        }
        String host = sSharedPreferences.getString(PCT_LOG_SERVER, dflValue);
        if (TextUtils.isEmpty(host)) {
            return dflValue;
        }

        return host;
    }

    public static void setPCTLogServer(@NonNull String logServer) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setPCTLogServer");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putString(PCT_LOG_SERVER, logServer);
        editor.apply();
    }

    public static int getMBIdealPacket(int type, int dflValue) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getMBLastMultiple");
            return dflValue;
        }
        return sSharedPreferences.getInt(MB_IDEAL_PACKET + "_" + type, dflValue);
    }

    public static void setMBIdealPacket(int type, int value) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setMBLastMultiple");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putInt(MB_IDEAL_PACKET + "_" + type, value);
        editor.apply();
    }

    public static long getMBBandwidth(long dflValue) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getMBBandwidth");
            return dflValue;
        }
        return sSharedPreferences.getLong(MB_BANDWIDTH, dflValue);
    }

    public static void setMBBandwidth(long value) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setMBBandwidth");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putLong(MB_BANDWIDTH, value);
        editor.apply();
    }

    public static int getH264MediacodecRank() {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getH264MediacodecRank");
            return 0;
        }
        return sSharedPreferences.getInt(H264_MEDIACODEC_RANK, 0);
    }

    public static void setH264MediacodecRank(int value) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setH264MediacodecRank");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putInt(H264_MEDIACODEC_RANK, value);
        editor.apply();
    }

    public static int getH265MediacodecRank() {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : getH265MediacodecRank");
            return 0;
        }
        return sSharedPreferences.getInt(H265_MEDIACODEC_RANK, 0);
    }

    public static void setH265MediacodecRank(int value) {
        if (sSharedPreferences == null) {
            LogUtil.w("SPUtil", "Invalid Context Warning : setH265MediacodecRank");
            return;
        }
        SharedPreferences.Editor editor = sSharedPreferences.edit();
        editor.putInt(H265_MEDIACODEC_RANK, value);
        editor.apply();
    }
}
