package com.ott.stream.rapid.agent;

import static com.ott.stream.rapid.agent.RapidAgentSDK.DEVICE_PLATFORM_BOX;
import static com.ott.stream.rapid.agent.RapidAgentSDK.DEVICE_PLATFORM_PHONE;

import android.app.ActivityManager;
import android.content.Context;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.ott.stream.rapid.agent.utils.CommonUtil;
import com.ott.stream.rapid.agent.utils.LogUtil;
import com.ott.stream.rapid.agent.utils.SPUtil;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.regex.Pattern;

public final class RapidAgentConstant {
    private static int SDK_VERSION_CODE;
    private static int PLAYER_VERSION_CODE;
    private static int ENGINE_VERSION_CODE;
    private static int SYSTEM_VERSION_CODE;

    private static String VNO_ID;
    private static String VNO_TAG;
    private static String VNO_SECRET_KEY;
    private static String VNO_TOKEN;

    private static int PLATFORM_TYPE;
    private static String APPLICATION_NAME;
    private static String ACCOUNT_ID;
    private static String ACCOUNT_TOKEN;

    private static String DEVICE_ID;
    private static String ANDROID_ID;
    private static String AID;
    private static long CID;

    private static String DEVICE_TYPE = null;
    private static String USER_TYPE = null;
    private static String BRAND = null;
    private static String MODEL = null;
    private static String CPU_NAME;
    private static String CPU_INFO;
    private static String MEM_INFO;
    private static int STORAGE_SIZE_GB;

    private static String ROOT_DIR;
    private static String SDK_LOG_DIR;
    private static String PRT_WORK_DIR;
    private static String PRT_PM_CACHE_DIR;
    private static String PRT_CFG_FILE_NAME;
    private static String PCT_WORK_DIR;
    private static String RDNS_WORK_DIR;

    static void initWorkPath(@Nullable Context context) {
        String rootDir;
        if (context != null) {
            rootDir = Objects.requireNonNull(ContextCompat.getDataDir(context)).getAbsolutePath() + File.separator + "rapidtv" + File.separator;
        } else if (CommonUtil.externalStorageAvailable()) {
            rootDir = Environment.getExternalStorageDirectory().getAbsolutePath() + File.separator + "rapidtv" + File.separator;
        } else {
            rootDir = File.separator + "rapidtv" + File.separator;
        }
        if (context == null) {
            LogUtil.w("RapidAgentConstant", "Invalid Context Warning : initWorkPath " + rootDir);
        }
        initWorkPath(rootDir);
    }

    static void initWorkPath(@NonNull String rootDir) {
        if (!TextUtils.isEmpty(ROOT_DIR)) {
            return;
        }
        if (rootDir.endsWith(File.separator)) {
            ROOT_DIR = rootDir;
        } else {
            ROOT_DIR = rootDir + File.separator;
        }
        LogUtil.sensitive("base work path : " + ROOT_DIR);
        SDK_LOG_DIR = ROOT_DIR + "sdklog";
        File sdkLogDir = new File(SDK_LOG_DIR);
        if (!sdkLogDir.exists() && !sdkLogDir.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create sdk log dir");
        }
        PRT_WORK_DIR = ROOT_DIR + "prt";
        File prtLogDir = new File(PRT_WORK_DIR);
        if (!prtLogDir.exists() && !prtLogDir.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create prt log dir");
        }
        PRT_PM_CACHE_DIR = ROOT_DIR + "prt_pm" + File.separator + "cache";
        File pmCacheDir = new File(PRT_PM_CACHE_DIR);
        if (!pmCacheDir.exists() && !pmCacheDir.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create pm cache dir");
        }
        PCT_WORK_DIR = ROOT_DIR + "pct";
        File pctWorkPath = new File(PCT_WORK_DIR);
        if (!pctWorkPath.exists() && !pctWorkPath.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create pct work path");
        }
        RDNS_WORK_DIR = ROOT_DIR + "rdns";
        File rdnsWorkPath = new File(RDNS_WORK_DIR);
        if (!rdnsWorkPath.exists() && !rdnsWorkPath.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create rdns work path");
        }
        PRT_CFG_FILE_NAME = "cfgFile.txt";
    }

    static void init(@Nullable Context context, int platform, String appName, String vnoId, String vnoTag,
                     String secretKey, String vnoToken, String userToken, String uid) {
        initWorkPath(context);
        VNO_ID = vnoId;
        VNO_TAG = vnoTag != null ? vnoTag : "";
        VNO_SECRET_KEY = secretKey;
        VNO_TOKEN = vnoToken;

        PLATFORM_TYPE = platform;
        APPLICATION_NAME = appName;
        ACCOUNT_ID = uid;
        ACCOUNT_TOKEN = userToken;

        DEVICE_ID = SPUtil.getDeviceId();
        ANDROID_ID = SPUtil.getAndroidId();
        AID = ACCOUNT_ID + DEVICE_ID;
        CID = SPUtil.getCID(DEVICE_ID);

        if (DEVICE_TYPE == null) {
            DEVICE_TYPE = platformString(platform);
        }
        if (USER_TYPE == null) {
            USER_TYPE = userTypeString(platform);
        }
        if (BRAND == null) {
            BRAND = android.os.Build.BRAND;
        }
        if (MODEL == null) {
            MODEL = android.os.Build.MODEL;
        }
        CPU_NAME = readCpuName() + "-" + Build.HARDWARE;
        CPU_INFO = readCpuInfo();
        MEM_INFO = readMemInfo(context);
        STORAGE_SIZE_GB = readTotalRom();
    }

    static void setVersions(int playerVersionCode, int engineVersionCode) {
        SDK_VERSION_CODE = CommonUtil.parseInt(BuildConfig.VERSION_CODE, 0);
        PLAYER_VERSION_CODE = playerVersionCode;
        ENGINE_VERSION_CODE = engineVersionCode;
        SYSTEM_VERSION_CODE = Build.VERSION.SDK_INT;
        LogUtil.i("RapidAgentConstant", "setVersions prt=" + ENGINE_VERSION_CODE + ", player=" + PLAYER_VERSION_CODE + ", sdk=" + SDK_VERSION_CODE);
    }

    /**
     * Gets the number of cores available in this device, across all processors.
     * Requires: Ability to peruse the filesystem at "/sys/devices/system/cpu"
     *
     * @return The number of cores, or 1 if failed to get result
     */
    private static String readCpuInfo() {
        List<String> freqList = new ArrayList<>();
        try {
            //Get directory containing CPU info
            File dir = new File("/sys/devices/system/cpu/");
            //Filter to only list the devices we care about
            File[] files = dir.listFiles(file -> Pattern.matches("cpu[0-9]", file.getName()));

            if (files != null && files.length > 0) {
                byte[] content = new byte[1024];
                for (File file : files) {
                    try (FileInputStream inStream = new FileInputStream(file.getAbsolutePath() + File.separator + "cpufreq" + File.separator + "cpuinfo_max_freq")) {
                        int len = inStream.read(content);
                        freqList.add(new String(content, 0, len));
                    }
                }
            }
        } catch (Exception e) {
            LogUtil.e("RapidAgentConstant", "fail to read cpu info: " + e);
        }

        String cpuInfo = "[" + CPU_NAME + "] [" + Build.PRODUCT + "] [" + Build.DEVICE + "] " + freqList;
        return cpuInfo.replace("\n", "");
    }

    private static String readCpuName() {
        String str1 = "/proc/cpuinfo";
        String str2;
        String cpuName = "";

        try (BufferedReader bufferedReader = new BufferedReader(new FileReader(str1))) {
            while ((str2 = bufferedReader.readLine()) != null) {
                if (TextUtils.isDigitsOnly(str2)) {
                    continue;
                }
                String[] array = str2.split(":\\s+", 2);
                if (TextUtils.equals(array[0].trim(), "Hardware")) {
                    cpuName = array[1];
                    break;
                }
            }
        } catch (IOException e) {
            LogUtil.e("RapidAgentConstant", "fail to read cpu name: " + e);
        }
        return cpuName;
    }

    private static String readMemInfo(@Nullable Context context) {
        if (context == null) {
            LogUtil.w("RapidAgentConstant", "Invalid Context Warning : readMemInfo");
            return "";
        }
        try {
            ActivityManager manager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
            ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
            manager.getMemoryInfo(memoryInfo);
            return String.format(Locale.US, "%.2fG,%.2fG", memoryInfo.totalMem / 1024f / 1024f / 1024f, memoryInfo.availMem / 1024f / 1024f / 1024f);
        } catch (Exception e) {
            LogUtil.e("RapidAgentConstant", "fail to read mem info: " + e);
        }
        return "";
    }

    private static int readTotalRom() {
        File dataDir = Environment.getDataDirectory();
        StatFs stat = new StatFs(dataDir.getPath());
        long blockSize = stat.getBlockSizeLong();
        long totalBlocks = stat.getBlockCountLong();
        long size = totalBlocks * blockSize;
        long GB = 1024 * 1024 * 1024;
        return (int) (size / GB);
    }

    private static String userTypeString(int platform) {
        switch (platform) {
            case DEVICE_PLATFORM_BOX:
                return "stb";
            case DEVICE_PLATFORM_PHONE:
                return "phone";
            default:
                return "unknown";
        }
    }

    private static String platformString(int platform) {
        switch (platform) {
            case DEVICE_PLATFORM_BOX:
                return "stb";
            case DEVICE_PLATFORM_PHONE:
                return "phone";
            default:
                return "unknown";
        }
    }

    public static int getSdkVersion() {
        return SDK_VERSION_CODE;
    }

    public static int getPlayerVersion() {
        return PLAYER_VERSION_CODE;
    }

    public static int getEngineVersion() {
        return ENGINE_VERSION_CODE;
    }

    public static int getSystemVersion() {
        return SYSTEM_VERSION_CODE;
    }

    public static String getVnoId() {
        return VNO_ID;
    }

    public static String getVnoTag() {
        return VNO_TAG;
    }

    public static String getVnoSecretKey() {
        return VNO_SECRET_KEY;
    }

    public static String getVnoToken() {
        return VNO_TOKEN;
    }

    public static int getPlatformType() {
        return PLATFORM_TYPE;
    }

    public static String getApplicationName() {
        return APPLICATION_NAME;
    }

    public static String getAccountId() {
        return ACCOUNT_ID;
    }

    public static void setAccountToken(String accountToken) {
        ACCOUNT_TOKEN = accountToken;
    }

    public static String getAccountToken() {
        return ACCOUNT_TOKEN;
    }

    public static String getDeviceId() {
        return DEVICE_ID;
    }

    public static String getAndroidId() {
        return ANDROID_ID;
    }

    public static String getAID() {
        return AID;
    }

    public static long getCID() {
        return CID;
    }

    public static void setDeviceType(String deviceType) {
        DEVICE_TYPE = deviceType;
    }

    public static String getDeviceType() {
        return DEVICE_TYPE;
    }

    public static void setUserType(String userType) {
        USER_TYPE = userType;
    }

    public static String getUserType() {
        return USER_TYPE;
    }

    public static void setBrand(String brand) {
        RapidAgentConstant.BRAND = brand;
    }

    public static String getBrand() {
        return BRAND;
    }

    public static void setModel(String model) {
        RapidAgentConstant.MODEL = model;
    }

    public static String getModel() {
        return MODEL;
    }

    public static String getCpuName() {
        return CPU_NAME;
    }

    public static String getCpuInfo() {
        return CPU_INFO;
    }

    public static String getMemInfo() {
        return MEM_INFO;
    }

    public static int getStorageSizeGb() {
        return STORAGE_SIZE_GB;
    }

    public static String getRootDir() {
        return ROOT_DIR;
    }

    public static String getSdkLogDir() {
        return SDK_LOG_DIR;
    }

    public static String getPrtWorkDir() {
        return PRT_WORK_DIR;
    }

    public static String getPrtCfgFileName() {
        return PRT_CFG_FILE_NAME;
    }

    public static String getPrtPmCacheDir() {
        return PRT_PM_CACHE_DIR;
    }

    public static String getPctWorkDir() {
        return PCT_WORK_DIR;
    }

    public static String getRDnsWorkDir() {
        return RDNS_WORK_DIR;
    }
}
