package com.ott.stream.rapid.agent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.nio.ByteBuffer;

/**
 * 对于非标准加密的流，需要app来实现解密，需要支持流式解密。
 * <p>
 * 对于数据的要求：
 * 1、如果该条流由单个文件组成，比如mp4或ts，那么这个文件可以独立解密，不需要依赖其他数据。由于可能seek到任意一个位置，
 * 所以只能支持“逐字节加密，无需填充”的加密算法，比如AES-CTR。
 * 2、如果该条流由多个切片组成，比如hls或dash中的切片，那么切片之间不应存在依赖关系，每个切片可以独立解密。
 * 3、如果该条流由微切片组成，一个或多个微切片组成一个微切片组，微切片组之间不应存在依赖关系，每个微切片组可以独立解密。
 * <p>
 * 数据的解密过程称为解密会话，有依赖关系的数据块组成一个解密会话。解密会话由引擎创建，使用sessionId作为解密会话的唯一标识。
 * 引擎先通过sendEncryptedData向app发送加密数据，再通过receiveDecryptedData获取解密后的数据。app需要实现这两个接口，
 * 并在sdk初始化时通过{@link RapidAgentSDK#setExternalDecrypt(RapidAgentExternalDecrypt)}设置。
 */
public interface RapidAgentExternalDecrypt {
    /**
     * 解密会话的第一个数据块。
     */
    int INPUT_FLAG_START = 0x1;
    /**
     * 解密会话的最后一个数据块，有可能为空，只是为传递解密会话结束的标志。
     */
    int INPUT_FLAG_END = 0x2;
    /**
     * 数据不连续，中间丢失了部分数据块。
     */
    int INPUT_FLAG_DISCONTINUITY = 0x4;

    /**
     * 成功
     */
    int INPUT_SUCCESS = 1;
    /**
     * 未知错误
     */
    int INPUT_ERROR_UNKNOWN = 0;
    /**
     * 数据不连续，无法继续解密，引擎应该丢弃整个解密会话的剩余数据。
     */
    int INPUT_ERROR_CANT_DECRYPT = -1;
    /**
     * 已达到最大内存，引擎应该调用receiveDecryptedData降低内存。
     */
    int INPUT_ERROR_OOM = -2;

    /**
     * 返回已解密的新数据。
     */
    int OUTPUT_FLAG_AVAILABLE = 0x1;
    /**
     * 将当前所有的数据解密并返回。
     */
    int OUTPUT_FLAG_CURRENT = 0x2;
    /**
     * 返回整个解密会话剩余的所有数据。
     */
    int OUTPUT_FLAG_ALL = 0x4;

    /**
     * 成功
     */
    int OUTPUT_ERROR_UNKNOWN = 0;
    /**
     * 输入数据过少，无法进行解密
     */
    int OUTPUT_ERROR_NO_DATA = -1;
    /**
     * app正在处理，稍后重试
     */
    int OUTPUT_ERROR_RETRY = -2;
    /**
     * outputBuffer中的内存不足
     */
    int OUTPUT_ERROR_OOM = -3;
    /**
     * 与sendEncryptedData参数不匹配
     */
    int OUTPUT_ERROR_INVALID_PARAMS = -4;

    /**
     * 由引擎通过jni触发，向app发送加密数据。
     *
     * @param channelId   频道id，通过{@link com.ott.stream.rapid.agent.player.RapidAgentPlayer#setStateChangedListener(RapidAgentCall.OnStateChangedListener)}回调
     * @param trackId     轨道id
     * @param sessionId   解密会话id
     * @param blockId     数据块id
     * @param inputBuffer 加密数据，app使用前应该拷贝到app内存空间，引擎会不定时释放。
     * @param offset      在文件中的偏移
     * @param flags       参考INPUT_FLAG_*
     * @return 参考INPUT_SUCCESS和INPUT_ERROR_*
     */
    int sendEncryptedData(int channelId, int trackId, int sessionId, long blockId, @Nullable ByteBuffer inputBuffer, long offset, int flags);

    /**
     * 由引擎通过jni触发，从app获取解密后的数据。
     *
     * @param channelId    频道id，通过{@link com.ott.stream.rapid.agent.player.RapidAgentPlayer#setStateChangedListener(RapidAgentCall.OnStateChangedListener)}回调
     * @param trackId      轨道id
     * @param sessionId    解密会话id
     * @param outputBuffer 引擎申请的一块内存，app填入解密后的数据。
     * @param offset       在文件中的偏移
     * @param flags        参考OUTPUT_FLAG_*
     * @return >0：返回buffer的大小 <=0：参考OUTPUT_ERROR_*
     */
    int receiveDecryptedData(int channelId, int trackId, int sessionId, @NonNull ByteBuffer outputBuffer, long offset, int flags);
}
