package com.ott.stream.rapid.agent;

import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.config.CommonConfig;
import com.ott.stream.rapid.agent.utils.LogUtil;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * A flexible representation of the structure of stream.
 */
public abstract class RapidAgentUri {
    /**
     * Clear stream
     */
    public static final int DRM_TYPE_NULL = 0;
    /**
     * Widevine stream
     */
    public static final int DRM_TYPE_WIDEVINE = 1;
    /**
     * aes crypted stream
     */
    public static final int DRM_TYPE_AES = 4;

    /**
     * @hidden
     */
    public static final int PROTOCOL_UNKNOWN = 0;
    /**
     * @hidden
     */
    public static final int PROTOCOL_RPD = 1;
    /**
     * @hidden
     */
    public static final int PROTOCOL_MPT = 2;
    /**
     * @hidden
     */
    public static final int PROTOCOL_MPQ = 3;
    /**
     * @hidden
     */
    public static final int PROTOCOL_DRM = 4;

    /**
     * Only 1 frame in 1 block
     */
    public static final int MICRO_BLOCK_ULTRA_LOW = 1;
    /**
     * 1 gop which can be play 1 second in 1 block
     */
    public static final int MICRO_BLOCK_LOW = 2;
    /**
     * 1 gop which can be play random second in 1 block
     */
    public static final int MICRO_BLOCK_NORMAL = 3;
    /**
     * 1 gop which can be play 3 second in 1 block
     */
    public static final int MICRO_BLOCK_GOP3 = 4;

    /**
     * Steam id which required by all steam.
     */
    protected String streamId;
    /**
     * Optional. Steam name.
     */
    protected String streamName;
    /**
     * Required by all stream. Prt will make a authentication before playing stream.
     */
    protected String streamToken;
    /**
     * Front info from engine.
     */
    protected Map<String, Object> frontInfo;
    /**
     * Optional. The transport protocol should be used to download the source.
     */
    protected int protocol = PROTOCOL_UNKNOWN;
    /**
     * Required by encrypted steam.
     * see {@link RapidAgentUri#DRM_TYPE_NULL}, {@link RapidAgentUri#DRM_TYPE_WIDEVINE}, {@link RapidAgentUri#DRM_TYPE_AES}
     */
    protected int drmType = DRM_TYPE_NULL;
    /**
     * Required by encrypted steam. DRM license server URI.
     */
    protected String drmLicenseUrl;
    /**
     * Required by encrypted steam. The optional request headers attached to DRM license requests.
     */
    protected Map<String, String> drmHeaders;
    /**
     * The method of DRM license requests. MUST be POST or GET
     */
    protected String drmMethod;
    /**
     * The offline license key set id
     */
    protected byte[] drmOfflineLicenseKeySetId;
    /**
     * The offline license drm init info
     */
    protected String drmOfflineLicenseInitInfo;
    /**
     * The vno key
     */
    protected String vnoKey;
    /**
     * Whether it's an adaptive bitrate stream.
     */
    protected boolean isAdaptiveBitrate;
    /**
     * Whether it's a micro block stream.
     */
    protected boolean isMicroBlock;
    /**
     * Micro block level.
     * see {@link RapidAgentUri#MICRO_BLOCK_ULTRA_LOW}, {@link RapidAgentUri#MICRO_BLOCK_LOW}, {@link RapidAgentUri#MICRO_BLOCK_NORMAL}, {@link RapidAgentUri#MICRO_BLOCK_GOP3}
     */
    protected int microBlockLevel;
    /**
     * Optional. The special info for back-to-resource
     */
    protected String routeId;
    /**
     * Optional. The format of the demuxer, 1:mp4 0:ts
     */
    protected int format = -1;
    /**
     * Optional. The stream need to be decrypted by app
     */
    protected boolean needExternalDecrypt;
    /**
     * Optional. Custom info map
     */
    protected final Map<String, Object> customInfo = new LinkedHashMap<>();

    protected RapidAgentUri(String streamId) {
        this.streamId = streamId;
    }

    public abstract RapidAgentUri copySelf();

    protected void copySelf(RapidAgentUri other) {
        other.setStreamId(streamId);
        other.setStreamName(streamName);
        other.setStreamToken(streamToken);
        other.setFrontInfo(frontInfo);
        other.setProtocol(protocol);
        other.setDrmType(drmType);
        other.setDrmLicenseUrl(drmLicenseUrl);
        other.setDrmHeaders(drmHeaders);
        other.setDrmMethod(drmMethod);
        other.setDrmOfflineLicenseKeySetId(drmOfflineLicenseKeySetId);
        other.setDrmOfflineLicenseInitInfo(drmOfflineLicenseInitInfo);
        other.setVnoKey(vnoKey);
        other.setAdaptiveBitrate(isAdaptiveBitrate);
        other.setMicroBlock(isMicroBlock, microBlockLevel);
        other.setRouteId(routeId);
        other.setFormat(format);
        other.setNeedExternalDecrypt(needExternalDecrypt);
        other.setCustomInfo(customInfo);
    }

    /**
     * Gets the stream id
     *
     * @return the stream id
     */
    public String getStreamId() {
        return streamId;
    }

    /**
     * Sets the stream id
     *
     * @param streamId the stream id
     */
    public void setStreamId(String streamId) {
        this.streamId = streamId;
    }

    /**
     * Gets the stream name
     *
     * @return the stream name
     */
    public String getStreamName() {
        return streamName;
    }

    /**
     * Sets the stream name
     *
     * @param streamName the stream name
     */
    public void setStreamName(String streamName) {
        if (!TextUtils.isEmpty(streamName)) {
            this.streamName = streamName;
        }
    }

    /**
     * Gets the stream token
     *
     * @return the stream token
     */
    public String getStreamToken() {
        return streamToken;
    }

    /**
     * Sets the stream token
     *
     * @param streamToken stream play token
     */
    public void setStreamToken(String streamToken) {
        this.streamToken = streamToken;
    }

    /**
     * @param frontInfo front info
     * @hidden
     */
    public void setFrontInfo(Map<String, Object> frontInfo) {
        this.frontInfo = frontInfo;
    }

    /**
     * @return front info
     * @hidden
     */
    public Map<String, Object> getFrontInfo() {
        return frontInfo;
    }

    /**
     * @return protocol
     * @hidden
     */
    public int getProtocol() {
        return protocol;
    }

    /**
     * @param protocol protocol
     * @hidden
     */
    public void setProtocol(int protocol) {
        this.protocol = protocol;
    }

    /**
     * Gets the drm type, return {@link RapidAgentUri#DRM_TYPE_NULL}, {@link RapidAgentUri#DRM_TYPE_WIDEVINE}, {@link RapidAgentUri#DRM_TYPE_AES}
     *
     * @return the drm type
     */
    public int getDrmType() {
        return drmType;
    }

    /**
     * Sets the drm type, see {@link RapidAgentUri#DRM_TYPE_NULL}, {@link RapidAgentUri#DRM_TYPE_WIDEVINE}, {@link RapidAgentUri#DRM_TYPE_AES}
     *
     * @param drmType the drm type
     */
    public void setDrmType(int drmType) {
        this.drmType = drmType;
    }

    /**
     * Gets the default DRM license server URI.
     *
     * @return url
     */
    public String getDrmLicenseUrl() {
        return drmLicenseUrl;
    }

    /**
     * Sets the default DRM license server URI.
     *
     * @param drmLicenseUrl url
     */
    public void setDrmLicenseUrl(String drmLicenseUrl) {
        this.drmLicenseUrl = drmLicenseUrl;
    }

    /**
     * Gets the optional request headers attached to DRM license requests.
     *
     * @return the request headers
     */
    public Map<String, String> getDrmHeaders() {
        return drmHeaders;
    }

    /**
     * Sets the optional request headers attached to DRM license requests.
     *
     * @param drmHeaders the request headers
     */
    public void setDrmHeaders(Map<String, String> drmHeaders) {
        this.drmHeaders = drmHeaders;
    }

    /**
     * Gets the method of DRM license requests.
     *
     * @return http MUST be POST or GET
     */
    public String getDrmMethod() {
        return drmMethod;
    }

    /**
     * Sets the method of DRM license requests. MUST be POST or GET
     *
     * @param drmMethod http method
     */
    public void setDrmMethod(String drmMethod) {
        this.drmMethod = drmMethod;
    }

    /**
     * set offline license key set id
     *
     * @param set offline license set
     */
    public void setDrmOfflineLicenseKeySetId(byte[] set) {
        if (set != null) {
            this.drmOfflineLicenseKeySetId = set.clone();
        } else {
            this.drmOfflineLicenseKeySetId = null;
        }
    }

    /**
     * get offline license key set id
     *
     * @return offline license key set id
     */
    public byte[] getDrmOfflineLicenseKeySetId() {
        return drmOfflineLicenseKeySetId;
    }

    /**
     * set offline license drm init info
     *
     * @param drmOfflineLicenseInitInfo offline license drm init info
     */
    public void setDrmOfflineLicenseInitInfo(String drmOfflineLicenseInitInfo) {
        this.drmOfflineLicenseInitInfo = drmOfflineLicenseInitInfo;
    }

    /**
     * get offline license drm init info
     *
     * @return offline license drm init info
     */
    public String getDrmOfflineLicenseInitInfo() {
        return drmOfflineLicenseInitInfo;
    }

    /**
     * Gets the vno key
     *
     * @return the vno key
     */
    public String getVnoKey() {
        return vnoKey;
    }

    /**
     * Sets the vno key
     *
     * @param vnoKey the vno key
     */
    public void setVnoKey(String vnoKey) {
        this.vnoKey = vnoKey;
    }

    /**
     * Sets whether this is an adaptive bitrate stream.
     *
     * @param adaptiveBitrate whether this is an adaptive bitrate stream.
     */
    public void setAdaptiveBitrate(boolean adaptiveBitrate) {
        isAdaptiveBitrate = adaptiveBitrate;
    }

    /**
     * Returns whether this is an adaptive bitrate stream.
     *
     * @return whether this is an adaptive bitrate stream.
     */
    public boolean isAdaptiveBitrate() {
        return isAdaptiveBitrate;
    }

    /**
     * Sets whether this is a micro block stream.
     *
     * @param microBlock whether this is a micro block stream.
     * @param level      see {@link RapidAgentUri#MICRO_BLOCK_ULTRA_LOW}, {@link RapidAgentUri#MICRO_BLOCK_LOW}, {@link RapidAgentUri#MICRO_BLOCK_NORMAL}, {@link RapidAgentUri#MICRO_BLOCK_GOP3}
     */
    public void setMicroBlock(boolean microBlock, int level) {
        if (!CommonConfig.isMicroBlockEnabled()) {
            LogUtil.w("RapidAgentUri", "mb feature is disable");
            return;
        }
        if (isMicroBlock && microBlockLevel <= 0) {
            throw new IllegalArgumentException("invalid micro block level");
        }
        isMicroBlock = microBlock;
        microBlockLevel = level;
    }

    /**
     * Returns whether this is a micro block stream.
     *
     * @return whether this is a micro block stream.
     */
    public boolean isMicroBlock() {
        return isMicroBlock;
    }

    /**
     * Returns the micro block level
     *
     * @return the micro block level
     */
    public int getMicroBlockLevel() {
        return microBlockLevel;
    }

    /**
     * Gets the special info for back-to-resource
     *
     * @return route id
     */
    public String getRouteId() {
        return routeId;
    }

    /**
     * Sets the special info for back-to-resource
     *
     * @param routeId route id
     */
    public void setRouteId(String routeId) {
        this.routeId = routeId;
    }

    /**
     * Gets the format
     *
     * @return format
     */
    public int getFormat() {
        return format;
    }

    /**
     * Sets the format
     *
     * @param format format
     */
    public void setFormat(int format) {
        this.format = format;
    }

    /**
     * Gets whether need external decrypt
     *
     * @return the result
     */
    public boolean isNeedExternalDecrypt() {
        return needExternalDecrypt;
    }

    /**
     * Sets whether need external decrypt
     *
     * @param needExternalDecrypt whether need external decrypt
     */
    public void setNeedExternalDecrypt(boolean needExternalDecrypt) {
        this.needExternalDecrypt = needExternalDecrypt;
    }

    /**
     * Gets the custom info map
     *
     * @return custom info map
     */
    public Map<String, Object> getCustomInfo() {
        return customInfo;
    }

    /**
     * Sets the custom info map
     *
     * @param map the custom info map
     */
    public void setCustomInfo(@NonNull Map<String, Object> map) {
        customInfo.putAll(map);
    }
}
