package com.ott.stream.rapid.agent;

import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.config.CommonConfig;
import com.ott.stream.rapid.agent.utils.LogUtil;

import java.util.Map;

/**
 * Representation of a vod stream
 */
public final class RapidAgentVodUri extends RapidAgentUri {
    /**
     * Constructs a new instance
     *
     * @param streamId the stream id
     */
    private RapidAgentVodUri(String streamId) {
        super(streamId);
    }

    /**
     * Creates a {@link RapidAgentVodUri} whose protocol is mpt
     *
     * @param streamId the stream id
     * @param routeId  the special info for back-to-resource
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createMptUri(@NonNull String streamId, @NonNull String routeId) {
        return createUri(streamId, routeId, PROTOCOL_MPT);
    }

    /**
     * Creates a {@link RapidAgentVodUri} whose protocol is rpd
     *
     * @param streamId the stream id
     * @param routeId  the special info for back-to-resource
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createRpdUri(@NonNull String streamId, @NonNull String routeId) {
        return createUri(streamId, routeId, PROTOCOL_RPD);
    }

    private static RapidAgentVodUri createUri(@NonNull String streamId, @NonNull String routeId, int protocol) {
        RapidAgentVodUri uri = new RapidAgentVodUri(streamId);
        uri.setRouteId(routeId);
        uri.customInfo.put("routeId", routeId);
        String forceProtocol = CommonConfig.getCdnVodProtocol();
        if (!TextUtils.isEmpty(forceProtocol)) {
            forceProtocol = forceProtocol.toLowerCase();
            LogUtil.w("RapidAgentVodUri", "create vod uri: cdn vod protocol is : " + forceProtocol);
            if (forceProtocol.contains("mpt")) {
                protocol = PROTOCOL_MPT;
            } else if (forceProtocol.contains("rpd")) {
                protocol = PROTOCOL_RPD;
            }
        }
        uri.setProtocol(protocol);
        return uri;
    }

    @Override
    public void setFrontInfo(Map<String, Object> frontInfo) {
        super.setFrontInfo(frontInfo);
        if (frontInfo != null && frontInfo.containsKey("pt")) {
            String frontProtocol = (String) frontInfo.get("pt");
            if (frontProtocol != null) {
                frontProtocol = frontProtocol.toLowerCase();
                if (frontProtocol.contains("mpt") && protocol != RapidAgentUri.PROTOCOL_MPT) {
                    protocol = RapidAgentUri.PROTOCOL_MPT;
                    LogUtil.w("RapidAgentVodUri", "protocol of front info has changed to mpt");
                } else if (frontProtocol.contains("rpd") && protocol != RapidAgentUri.PROTOCOL_RPD) {
                    protocol = RapidAgentUri.PROTOCOL_RPD;
                    LogUtil.w("RapidAgentVodUri", "protocol of front info has changed to rpd");
                }
            }
        }
    }

    @Override
    public RapidAgentUri copySelf() {
        RapidAgentVodUri other = new RapidAgentVodUri(streamId);
        copySelf(other);
        return other;
    }

    @NonNull
    @Override
    public String toString() {
        return "RapidAgentVodUri{" +
                "streamId='" + streamId + '\'' +
                ", routeId='" + routeId + '\'' +
                ", drmType='" + drmType + '\'' +
                ", decrypt='" + needExternalDecrypt + '\'' +
                ", customInfo='" + customInfo + '\'' +
                '}';
    }
}
