package com.ott.stream.rapid.agent.debug;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.utils.LogUtil;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

public class LogcatHelper {
    private static volatile LogcatHelper sInstance;

    private static final SimpleDateFormat SIMPLE_DATE_FORMAT = new SimpleDateFormat("yyyyMMddHHmm", Locale.US);

    private Thread currentThread;
    private volatile boolean isRunning;

    static {
        SIMPLE_DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("GMT+8:00"));
    }

    private LogcatHelper() {
    }

    public static LogcatHelper getInstance() {
        if (sInstance == null) {
            synchronized (LogcatHelper.class) {
                if (sInstance == null) {
                    sInstance = new LogcatHelper();
                }
            }
        }
        return sInstance;
    }

    public synchronized boolean startReadLogcat(String logFilePath, String logFileName, int what, int extra) {
        if (!DebugConfig.shouldLogcat(what, extra)) {
            return false;
        }
        if (isRunning) {
            return false;
        }
        currentThread = new Thread() {
            @Override
            public void run() {
                BufferedReader reader = null;
                FileWriter fileWriter = null;
                BufferedWriter bufferedWriter = null;
                int size = 0;
                try {
                    File logFileDir = new File(logFilePath);
                    if (!logFileDir.exists() && !logFileDir.mkdirs()) {
                        LogUtil.e("LogcatHelper", "fail to create log file dir");
                        return;
                    }
                    File logFile = new File(logFilePath, logFileName);
                    if (logFile.exists()) {
                        if (!logFile.delete()) {
                            LogUtil.e("LogcatHelper", "fail to delete the old file");
                            return;
                        }
                    }
                    if (!logFile.createNewFile()) {
                        LogUtil.e("LogcatHelper", "fail to create log file");
                        return;
                    }
                    fileWriter = new FileWriter(logFile, true);
                    bufferedWriter = new BufferedWriter(fileWriter);

                    LogUtil.i("LogcatHelper", "start logcat to file " + logFileName);
                    isRunning = true;
                    int pid = android.os.Process.myPid();
                    Process process = new ProcessBuilder("logcat",
                            "-v", "threadtime",
                            "--pid", String.valueOf(pid)).start();
                    reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                    char[] buf = new char[1024];
                    int count;
                    while (isRunning && (count = reader.read(buf)) > 0) {
                        bufferedWriter.write(buf, 0, count);
                        size += count;
                    }
                    if (sendLogFile(logFile, what, extra)) {
                        DebugConfig.increaseLogTimes();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                } finally {
                    isRunning = false;
                    LogUtil.i("LogcatHelper", "finish logcat and size:" + size + " for today time:" + DebugConfig.getTodayTimes());
                }
                try {
                    if (reader != null) {
                        reader.close();
                    }
                    if (bufferedWriter != null) {
                        bufferedWriter.close();
                    }
                    if (fileWriter != null) {
                        fileWriter.close();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        };
        currentThread.start();
        return true;
    }

    private boolean sendLogFile(@NonNull File file, int what, int extra) {
        String path = file.getParent();
        String time = SIMPLE_DATE_FORMAT.format(new Date());
        String newFileName = String.format(Locale.US, "rapid_sdk_%s_msg_%d-%d_time_%d.log", time, what, extra, DebugConfig.getTodayTimes());
        File newFile = new File(path, newFileName);
        if (!file.renameTo(newFile)) {
            LogUtil.e("LogcatHelper", "fail to rename file " + newFile.getPath());
            return false;
        }
        // send to server
        LogUtil.i("LogcatHelper", "send to server " + newFile.getPath());
        if (!newFile.delete()) {
            LogUtil.w("LogcatHelper", "fail to delete file " + newFile.getPath());
        }
        return true;
    }

    public synchronized void stopReadLogcat() {
        LogUtil.i("LogcatHelper", "trigger stop logcat");
        isRunning = false;
        if (currentThread != null && currentThread.isAlive()) {
            try {
                currentThread.join();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }
}

