package com.ott.stream.rapid.agent.http;

import android.net.Uri;
import android.text.TextUtils;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.ott.stream.rapid.agent.pct.PctInvalidUrlException;
import com.ott.stream.rapid.pct.RapidAgentPCTSDK;
import com.ott.stream.rapid.agent.pct.PctLockException;
import com.ott.stream.rapid.agent.utils.CommonUtil;
import com.ott.stream.rapid.agent.utils.LogUtil;
import com.pct.core.api.PctApi;
import com.pct.core.api.PctRequest;
import com.pct.core.api.PctResponse;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.zip.GZIPInputStream;

public class PCTHttpApi implements HttpApi {
    private static final String TAG = "PCTAgentHttp";

    public static final int PCT_ERROR_UNCONNECTED = -1;
    public static final int PCT_ERROR_LOCK_PERIOD = -2;

    private final Map<String, String> webIdMap = new HashMap<>();
    private final int timeoutSeconds;

    public PCTHttpApi(int timeoutMillis) {
        this(timeoutMillis, new HashMap<>());
    }

    public PCTHttpApi(int timeoutMillis, @NonNull Map<String, String> map) {
        this.timeoutSeconds = timeoutMillis / 1000;
        webIdMap.putAll(RapidAgentPCTSDK.getDflWebIdMap());
        webIdMap.putAll(map);
    }

    @Override
    public String post(@NonNull String url, @Nullable Map<String, String> headers, @NonNull String body) throws IOException {
        return pctRequest(url, headers, body, PctRequest.Method.POST);
    }

    @Override
    public String get(@NonNull String url, @Nullable Map<String, String> headers) throws IOException {
        return pctRequest(url, headers, null, PctRequest.Method.GET);
    }

    private String pctRequest(@NonNull String url, @Nullable Map<String, String> headers, @Nullable String body, PctRequest.Method method) throws IOException {
        Uri uri = Uri.parse(url);
        String domain;
        if (uri.getPort() >= 0) {
            domain = uri.getHost() + ":" + uri.getPort();
        } else {
            domain = uri.getHost();
        }
        String webId = webIdMap.get(domain);
        if (TextUtils.isEmpty(webId)) {
            throw new PctInvalidUrlException("can not find web id for url:" + url);
        }
        String api = uri.getPath();
        Map<String, String> newHeaders = new HashMap<>();
        if (headers != null) {
            newHeaders.putAll(headers);
        }
        if (body != null && !newHeaders.containsKey("Content-Length")) {
            newHeaders.remove("content-length");
            newHeaders.put("Content-Length", String.valueOf(body.length()));
        }
        PctRequest request = new PctRequest(webId, api, method, newHeaders, body != null ? body.getBytes(StandardCharsets.UTF_8) : null);

        LogUtil.i(TAG, "--> " + method + " " + url + "(" + webId + ")");
        for (Map.Entry<String, String> entry : newHeaders.entrySet()) {
            LogUtil.i(TAG, entry.getKey() + ": " + entry.getValue());
        }

        final Object lock = new Object();
        AtomicReference<PctResponse.PctRspCode> responseCodeRef = new AtomicReference<>(PctResponse.PctRspCode.NONE);
        AtomicReference<byte[]> responseStringRef = new AtomicReference<>();
        long ret = RapidAgentPCTSDK.httpRequest(request, PctApi.RspType.ALL_END, timeoutSeconds, (PctApi.ITaskCallBack) (taskId, response, end) -> {
            if (response != null) {
                responseCodeRef.set(response.getCode());
                responseStringRef.set(response.getDataBytes());
            }
            synchronized (lock) {
                lock.notifyAll();
            }
        });
        if (ret < 0) {
            LogUtil.i(TAG, "<-- ERROR " + "invalid task id " + ret + " " + url);
            if (ret == PCT_ERROR_LOCK_PERIOD) {
                throw new PctLockException("pct is in lock period");
            } else {
                throw new IOException("invalid task id " + ret + " " + url);
            }
        }
        synchronized (lock) {
            try {
                lock.wait(timeoutSeconds * 1000L);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        PctResponse.PctRspCode responseCode = responseCodeRef.get();
        byte[] responseBody = responseStringRef.get();
        String responseString = responseBody != null ? new String(responseBody) : "";
        LogUtil.i(TAG, "<-- "
                + responseCode
                + '\n' + responseString
                + '\n' + url);
        if (responseCode == PctResponse.PctRspCode.HTTP_OK && responseBody != null) {
            String result = parseHttpDatagram(responseBody);
            LogUtil.v(TAG, "body result = " + result);
            return result;
        }
        return null;
    }

    private String parseHttpDatagram(@NonNull byte[] httpDatagram) {
        int responseCode = -1;
        boolean isChunk = false;
        boolean isGzip = false;
        int contentLength = 0;
        byte[] body;
        int bodyStart = 0;
        int bodyLen;

        for (int i = 0; i < httpDatagram.length - 4; i++) {
            if (httpDatagram[i] == 13 &&
                    httpDatagram[i + 1] == 10 &&
                    httpDatagram[i + 2] == 13 &&
                    httpDatagram[i + 3] == 10) {
                bodyStart = i + 4;
                break;
            }
        }

        try (BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(httpDatagram)))) {
            int lineNum = 0;
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.isEmpty()) {
                    break;
                }
                if (lineNum++ == 0) {
                    String[] statusLines = line.split(" ");
                    if (statusLines.length >= 2) {
                        String statusCode = statusLines[1];
                        responseCode = CommonUtil.parseInt(statusCode, -1);
                    }
                    continue;
                }
                Pair<String, String> header = parseHeader(line);
                if (header.first.equalsIgnoreCase("Transfer-Encoding") && header.second.equalsIgnoreCase("chunked")) {
                    isChunk = true;
                } else if (header.first.equalsIgnoreCase("Content-Encoding") && header.second.equalsIgnoreCase("gzip")) {
                    isGzip = true;
                } else if (header.first.equalsIgnoreCase("Content-Length")) {
                    contentLength = CommonUtil.parseInt(header.second, 0);
                }
            }
            if (responseCode != 200) {
                return null;
            }
            if (isChunk) {
                line = reader.readLine();
                bodyStart += line.length() + 2;
                bodyLen = Integer.parseInt(line, 16);
            } else {
                bodyLen = contentLength > 0 ? contentLength : httpDatagram.length - bodyStart;
            }
            if (bodyLen <= 0) {
                return null;
            }
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
        try (ByteArrayInputStream in = new ByteArrayInputStream(httpDatagram)) {
            long skipped = in.skip(bodyStart);
            if (skipped != bodyStart) {
                return null;
            }
            body = new byte[bodyLen];
            int readLen = in.read(body, 0, bodyLen);
            if (readLen != bodyLen) {
                return null;
            }
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
        if (!isGzip) {
            return new String(body, 0, bodyLen);
        }

        StringBuilder stringBuilder = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(new GZIPInputStream(new ByteArrayInputStream(body, 0, bodyLen))))) {
            String line;
            while ((line = reader.readLine()) != null) {
                stringBuilder.append(line);
            }
            return stringBuilder.toString();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    private Pair<String, String> parseHeader(@NonNull String line) {
        int index = line.indexOf(':');
        if (index < 0) {
            return Pair.create("unknown", "unknown");
        }
        String first = line.substring(0, index).trim();
        String second = line.substring(index + 1).trim();
        return Pair.create(first, second);
    }
}
