package com.ott.stream.rapid.agent.player;

import static com.ott.stream.rapid.agent.player.RapidAgentPlayer.PLAYER_TYPE_UNKNOWN;
import static tv.danmaku.ijk.media.player.IjkMediaPlayer.IJK_LOG_UNKNOWN;

import android.util.Pair;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.utils.CommonUtil;
import com.ott.stream.rapid.agent.utils.LogUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

public final class PlayerConfig {
    private static final Map<String, Object> IJK_COMMON_OPTIONS = new LinkedHashMap<>();
    private static final Map<String, Object> IJK_FORMAT_OPTIONS = new LinkedHashMap<>();
    private static final Map<String, Object> IJK_CODEC_OPTIONS = new LinkedHashMap<>();
    private static final Map<String, Object> IJK_SWS_OPTIONS = new LinkedHashMap<>();
    private static final Map<String, Object> IJK_PLAYER_OPTIONS = new LinkedHashMap<>();

    private static final int DEFAULT_FORCE_USING_IJK_INTERVAL = 5;

    static {
        IJK_PLAYER_OPTIONS.put("start-on-prepared", 1);
        IJK_PLAYER_OPTIONS.put("framedrop", 1);
        IJK_PLAYER_OPTIONS.put("audio-write-policy", 0);
        IJK_PLAYER_OPTIONS.put("mediacodec-all-videos", 1);
        IJK_PLAYER_OPTIONS.put("mediacodec-naked-csd", 1);
        IJK_PLAYER_OPTIONS.put("mediacodec-retry-time", 100);
        // 软解默认使用ANativeWindow来渲染，ANativeWindow会被永久占用，无法再被MediaCodec使用，导致一直使用软解。
        // 所以设置为软解使用gles来渲染
        IJK_PLAYER_OPTIONS.put("overlay-format", "fcc-_es2");
        // 默认支持字幕
        IJK_PLAYER_OPTIONS.put("subtitle", 1);

        IJK_FORMAT_OPTIONS.put("allowed_extensions", "ALL");
        IJK_FORMAT_OPTIONS.put("live_start_index", -3);
        IJK_FORMAT_OPTIONS.put("reconnect_streamed", 1);
        IJK_FORMAT_OPTIONS.put("probesize", 150000);
        IJK_FORMAT_OPTIONS.put("enlargeprobesize", 5000000);
        IJK_FORMAT_OPTIONS.put("http_multiple", 0);
        // 探测阶段播放器只读取部分切片数据，剩下的数据在创建解码器之后读取。但是引擎只会不断地发数据，如果切片比较大(大于4M)时，socket缓冲区会被写满，
        // 导致无法再写数据，然后引擎就会停止发送。所以会导致切片读取不完整，这里解决的办法是增加AVIO的缓冲区，先把socket缓冲区的数据读取到AVIO中。
        IJK_FORMAT_OPTIONS.put("initial-buffer-size", 1048576);
        IJK_FORMAT_OPTIONS.put("handle_codec_changed", 1);

        IJK_CODEC_OPTIONS.put("skip_loop_filter", 48);
    }

    public static int getDefaultPlayerType() {
        if (IJK_COMMON_OPTIONS.containsKey("player_type")) {
            Object obj = IJK_COMMON_OPTIONS.get("player_type");
            return CommonUtil.parseInt(obj, PLAYER_TYPE_UNKNOWN);
        }

        return PLAYER_TYPE_UNKNOWN;
    }

    public static int getDefaultLogLevel() {
        if (IJK_COMMON_OPTIONS.containsKey("log_level")) {
            Object obj = IJK_COMMON_OPTIONS.get("log_level");
            return CommonUtil.parseInt(obj, IJK_LOG_UNKNOWN);
        }

        return IJK_LOG_UNKNOWN;
    }

    public static void setIjkCommonOptions(Map<String, Object> map) {
        IJK_COMMON_OPTIONS.putAll(map);
    }

    public static void setIjkFormatOptions(Map<String, Object> map) {
        IJK_FORMAT_OPTIONS.putAll(map);
    }

    static Map<String, Object> getIjkFormatOptions() {
        return IJK_FORMAT_OPTIONS;
    }

    public static void setIjkCodecOptions(Map<String, Object> map) {
        IJK_CODEC_OPTIONS.putAll(map);
    }

    static Map<String, Object> getIjkCodecOptions() {
        return IJK_CODEC_OPTIONS;
    }

    public static void setIjkSwsOptions(Map<String, Object> map) {
        IJK_SWS_OPTIONS.putAll(map);
    }

    static Map<String, Object> getIjkSwsOptions() {
        return IJK_SWS_OPTIONS;
    }

    public static void setIjkPlayerOptions(Map<String, Object> map) {
        IJK_PLAYER_OPTIONS.putAll(map);
    }

    static Map<String, Object> getIjkPlayerOptions() {
        return IJK_PLAYER_OPTIONS;
    }

    public static int getForceUsingIjkInterval() {
        if (IJK_COMMON_OPTIONS.containsKey("force_ijk_i")) {
            Object obj = IJK_COMMON_OPTIONS.get("force_ijk_i");
            return CommonUtil.parseInt(obj, DEFAULT_FORCE_USING_IJK_INTERVAL);
        }

        return DEFAULT_FORCE_USING_IJK_INTERVAL;
    }

    public static boolean enableRebuildFeature() {
        if (IJK_COMMON_OPTIONS.containsKey("rebuild_enable")) {
            Object obj = IJK_COMMON_OPTIONS.get("rebuild_enable");
            int intObj = CommonUtil.parseInt(obj, 0);
            return intObj == 1;
        }

        return true;
    }

    public static Map<Pair<Integer, Integer>, Integer> getRebuildEvents() {
        Map<Pair<Integer, Integer>, Integer> events = new HashMap<>();
        if (IJK_COMMON_OPTIONS.containsKey("rebuild_events")) {
            Object obj = IJK_COMMON_OPTIONS.get("rebuild_events");
            String eventString = null;
            if (obj instanceof String) {
                eventString = (String) obj;
            }
            if (eventString != null) {
                eventString = eventString.replace(" ", "");
                String[] stringArray = eventString.split(";");
                for (String item : stringArray) {
                    String[] info = item.split(",");
                    if (info.length != 3) {
                        continue;
                    }
                    int eventWhat = CommonUtil.parseInt(info[0], 0);
                    int eventExtra = CommonUtil.parseInt(info[1], 0);
                    int playerType = CommonUtil.parseInt(info[2], 0);
                    events.put(Pair.create(eventWhat, eventExtra), playerType);
                }
            }
        }

        return events;
    }

    public static Map<Integer, Map<String, Object>> getJitterBufferConfig() {
        Map<Integer, Map<String, Object>> map = new HashMap<>();
        if (IJK_COMMON_OPTIONS.containsKey("jbuf_cfg")) {
            Object obj = IJK_COMMON_OPTIONS.get("jbuf_cfg");
            String configString = null;
            if (obj instanceof String) {
                configString = (String) obj;
            }
            if (configString != null) {
                try {
                    JSONArray jsonArray = new JSONArray(configString);
                    for (int i = 0; i < jsonArray.length(); i++) {
                        JSONObject jsonObject = jsonArray.getJSONObject(i);
                        if (jsonObject.has("ty")) {
                            int type = jsonObject.getInt("ty");
                            Map<String, Object> playMap = new HashMap<>();
                            parseJitterBufferItem(jsonObject, "jb", "check-jitter-buffer", playMap);
                            parseJitterBufferItem(jsonObject, "a", "jitter-buffer-cost-percent", playMap);
                            parseJitterBufferItem(jsonObject, "b", "jitter-buffer-max-percent", playMap);
                            parseJitterBufferItem(jsonObject, "c", "jitter-buffer-min-percent", playMap);
                            parseJitterBufferItem(jsonObject, "d", "jitter-buffer-safe-multiple", playMap);
                            parseJitterBufferItem(jsonObject, "e", "jitter-buffer-safe-frames", playMap);
                            parseJitterBufferItem(jsonObject, "f", "jitter-buffer-capacity-packets", playMap);
                            parseJitterBufferItem(jsonObject, "g", "jitter-buffer-target-frames", playMap);
                            map.put(type, playMap);
                        }
                    }
                } catch (JSONException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return map;
    }

    private static void parseJitterBufferItem(@NonNull JSONObject jsonObject, @NonNull String inputName, @NonNull String outputName, @NonNull Map<String, Object> playMap) throws JSONException {
        if (jsonObject.has(inputName)) {
            int value = jsonObject.getInt(inputName);
            playMap.put(outputName, value);
        }
    }

    public static void printConfig() {
        LogUtil.d("PlayConfig", "player common config : " + IJK_COMMON_OPTIONS);
        LogUtil.d("PlayConfig", "player format config : " + IJK_FORMAT_OPTIONS);
        LogUtil.d("PlayConfig", "player codec config : " + IJK_CODEC_OPTIONS);
        LogUtil.d("PlayConfig", "player sws config : " + IJK_SWS_OPTIONS);
        LogUtil.d("PlayConfig", "player ijk config : " + IJK_PLAYER_OPTIONS);
    }
}
