package com.ott.stream.rapid.agent.player;

import androidx.annotation.NonNull;

import com.ott.stream.rapid.agent.RapidAgentCall;
import com.ott.stream.rapid.agent.RapidAgentSDK;
import com.ott.stream.rapid.player.IRapidMediaPlayer;

import java.io.IOException;
import java.util.Map;

import tv.danmaku.ijk.media.common.ErrorNo;

public interface RapidAgentPlayer extends IRapidMediaPlayer {
    /**
     * Stream error msg
     */
    int STREAM_ERROR_MSG = -20002;
    /**
     * Stream info msg
     */
    int STREAM_INFO_MSG = -20005;

    /**
     * Stream metric info callback
     */
    interface OnStreamMetricListener {
        /**
         * Stream metric info callback
         *
         * @param params info
         */
        void onStreamMetric(Map<String, String> params);
    }

    /**
     * Stream state info callback
     */
    interface OnStreamStateListener {
        /**
         * Stream state info callback
         *
         * @param params info
         */
        void onStreamState(Map<String, String> params);
    }

    /**
     * set OnStreamMetricListener callback
     *
     * @param listener callback
     */
    void setOnStreamMetricListener(OnStreamMetricListener listener);

    /**
     * set OnStreamStateListener callback
     *
     * @param listener callback
     */
    void setOnStreamStateListener(OnStreamStateListener listener);

    /**
     * Sets call status changed listener
     *
     * @param stateChangedListener call status changed listener
     */
    void setOnCallStateChangedListener(RapidAgentCall.OnStateChangedListener stateChangedListener);

    /**
     * Update the user token and stream token
     *
     * @param newUserToken   new user token
     * @param newStreamToken new stream token
     * @throws IOException throw exception if fail to update the token
     */
    void updateTokens(String newUserToken, String newStreamToken) throws IOException;

    /**
     * send app log to prt log server
     *
     * @param msg message to send
     * @hidden
     */
    void sendFrontLog(String msg);

    /**
     * Get play metric
     *
     * @return play metric
     */
    @NonNull
    Map<String, Object> getPlayMetric();

    /**
     * App should notify the segment downloaded info, only valid for {@link RapidAgentScalablePlayer}
     *
     * @param params segment downloaded info
     */
    void onSegmentDownloaded(String params);

    /**
     * parse the error message
     *
     * @param what  error what
     * @param extra error extra
     * @return error message
     */
    static String error2String(int what, int extra) {
        if (what == STREAM_ERROR_MSG) {
            switch (extra) {
                case RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL:
                    return "EVENT_CREATE_CHANNEL_FAIL";
                case RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT:
                    return "EVENT_CREATE_CHANNEL_TIMEOUT";
                case RapidAgentSDK.EVENT_PROXY_NO_SIGNAL:
                    return "EVENT_PROXY_NO_SIGNAL";
                case RapidAgentSDK.EVENT_CHANGE_STREAM_NO_SIGNAL:
                    return "EVENT_CHANGE_STREAM_NO_SIGNAL";
                case RapidAgentSDK.EVENT_PROXY_VERIFY:
                    return "EVENT_PROXY_VERIFY";
                case RapidAgentSDK.EVENT_GET_INDEX_FAIL:
                    return "EVENT_GET_INDEX_FAIL";
                case RapidAgentSDK.EVENT_CONNECT_TOKEN_EXPIRED:
                    return "EVENT_CONNECT_TOKEN_EXPIRED";
                case RapidAgentSDK.EVENT_MB_NOT_AVAILABLE:
                    return "EVENT_MB_NOT_AVAILABLE";
                default:
                    return "Unknown prt engine error";
            }
        } else if (what == -10000) {
            return ErrorNo.error2string(extra);
        } else if (what == -2) {
            return "ExoPlayer:" + extra;
        }
        return "Unknown";
    }
}
