package com.ott.stream.rapid.agent.utils;

import android.content.Context;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.provider.Settings;
import android.text.TextUtils;

import androidx.annotation.Nullable;

import java.io.FileInputStream;
import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

/**
 * Created by elegant on 16/3/6.
 */
public class NetworkManager {
    private static volatile NetworkManager instance;

    public static final String WIFI_INTERFACE = "wlan0";
    public static final String ETHERNET_INTERFACE = "eth0";

    @Nullable
    private String wifiMac;
    @Nullable
    private String ethernetMac;

    @Nullable
    private final WifiManager wifiManager;
    @Nullable
    private final String androidId;

    private NetworkManager(@Nullable Context context) {
        String tmpAndroidId = null;
        if (context != null) {
            wifiManager = (WifiManager) context.getApplicationContext().getSystemService(Context.WIFI_SERVICE);
            try {
                tmpAndroidId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            LogUtil.w("NetworkManager", "Invalid Context Warning : new NetworkManager");
            wifiManager = null;
        }
        androidId = tmpAndroidId;
        LogUtil.sensitive("ANDROID_ID = " + androidId);
    }

    public static void init(Context app) {
        if (instance == null) {
            synchronized (NetworkManager.class) {
                if (instance == null) {
                    instance = new NetworkManager(app);
                    instance.refreshNetworkInfo();
                }
            }
        }
    }

    public static NetworkManager getInstance() {
        if (instance == null) {
            throw new IllegalStateException("please init first");
        }
        return instance;
    }

    //刷新网络信息
    public void refreshNetworkInfo() {
        //初始化mac
        try {
            Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
            while (networkInterfaces.hasMoreElements()) {
                NetworkInterface networkInterface = networkInterfaces.nextElement();
                if (WIFI_INTERFACE.equals(networkInterface.getDisplayName())) {
                    wifiMac = parseMac(networkInterface.getHardwareAddress());
                    continue;
                }
                if (ETHERNET_INTERFACE.equals(networkInterface.getDisplayName())) {
                    ethernetMac = parseMac(networkInterface.getHardwareAddress());
                    continue;
                }
            }
            if (TextUtils.isEmpty(wifiMac) && wifiManager != null) {
                WifiInfo wifiInfo = wifiManager.getConnectionInfo();
                String tmpWifiMac = wifiInfo.getMacAddress(); //if wifi is off it will be 020000000000
                if (!TextUtils.isEmpty(tmpWifiMac) && !TextUtils.equals("020000000000", tmpWifiMac) && !TextUtils.equals("02:00:00:00:00:00", tmpWifiMac)) {
                    wifiMac = tmpWifiMac;
                }
            }
            if (TextUtils.isEmpty(ethernetMac)) {
                ethernetMac = getMacFromFile(); // 从文件中获取
            }
            if (TextUtils.isEmpty(wifiMac)) {
                LogUtil.w("NetworkManager", "Invalid Context Warning : wifiMac is null");
            } else {
                LogUtil.sensitive("wifiMac = " + wifiMac);
            }
            if (TextUtils.isEmpty(ethernetMac)) {
                LogUtil.w("NetworkManager", "Invalid Context Warning : ethernetMac is null");
            } else {
                LogUtil.sensitive("ethernetMac = " + ethernetMac);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    //解析mac
    private String parseMac(byte[] mac) {
        StringBuilder sb = new StringBuilder();
        if (mac != null) {
            sb.delete(0, sb.length());
            for (byte b : mac) {
                sb.append(parseByte(b));
            }
            return sb.substring(0, sb.length() - 1);
        }
        return null;
    }

    private String parseByte(byte b) {
        String s = "00" + Integer.toHexString(b) + ":";
        return s.substring(s.length() - 3);
    }

    private static String getMacFromFile() {
        FileInputStream localFileInputStream = null;
        String mac = "";
        try {
            localFileInputStream = new FileInputStream(
                    "/sys/class/net/eth0/address");
            byte[] arrayOfByte = new byte[17];
            localFileInputStream.read(arrayOfByte, 0, 17);
            mac = new String(arrayOfByte);
            if (mac.contains("-"))
                mac = mac.replace("-", ":").trim();
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        } finally {
            if (localFileInputStream != null) {
                try {
                    localFileInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        if (TextUtils.isEmpty(mac)) {
            LogUtil.e("NetworkManager", "getMac by file return empty!");
            return "";
        }

        return mac.toLowerCase();
    }

    public String getUserMac() {
        String mac;
        if ((mac = trimToNull(ethernetMac)) == null) {
            mac = trimToNull(wifiMac);
        }
        if (mac == null) return "";
        return mac.replace(":", "").toLowerCase();
    }

    private static String trimToNull(final String str) {
        if (TextUtils.isEmpty(str)) {
            return null;
        }
        final String ts = str.trim();
        if (TextUtils.isEmpty(str)) {
            return null;
        }
        return ts;
    }

    @Nullable
    public String getAndroidId() {
        return androidId;
    }

    public static String getIpAddress() {
        try {
            for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements(); ) {
                NetworkInterface intf = (NetworkInterface) en.nextElement();
                for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements(); ) {
                    InetAddress inetAddress = (InetAddress) enumIpAddr.nextElement();
                    if (!inetAddress.isLoopbackAddress() && inetAddress instanceof Inet4Address) {
                        return inetAddress.getHostAddress();
                    }
                }
            }
        } catch (SocketException ex) {
            ex.printStackTrace();
        }
        return "";
    }
}
