package com.ott.stream.rapid.pct;

import android.content.Context;
import android.os.Environment;
import android.os.Handler;
import android.os.HandlerThread;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.network.rdns.RdnsApi;
import com.network.rdns.RdnsParams;
import com.pct.core.api.PctApi;
import com.pct.core.api.PctParams;
import com.pct.core.api.PctRequest;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.zip.GZIPOutputStream;

public class RapidAgentPCTSDK {
    private static final String TAG = "RapidAgentPCTSDK";

    private static final String DEFAULT_LOG_SERVER = "RpNrlbVUgI5m27bdMUiSorzjvdXxKw2LsToa+ArhAqvk";
    private static boolean sHasLoadLib = false;
    private static MetricCallback sMetricCallback = null;
    private static String sPCTWorkPath;
    private static String sRDNSWorkPath;

    private static final Queue<Pair<String, Map<String, Object>>> METRIC_EVENT_CACHE = new ArrayDeque<>();
    private static final Map<String, String> DEFAULT_WEB_ID_MAP = new HashMap<>();
    private static boolean sInitPCT = false;
    private static boolean sInitRDNS = false;
    private static String sLogServer;

    private static String sVnoId;

    private static final HandlerThread PCT_WORK_THREAD;
    private static final Handler PCT_WORK_HANDLER;

    static {
        PCT_WORK_THREAD = new HandlerThread("RapidPCTWork") {
            @Override
            public void run() {
                Log.i(TAG, "PCT work thread is started");
                super.run();
                Log.i(TAG, "PCT work thread is finished");
            }
        };
        PCT_WORK_THREAD.start();
        PCT_WORK_HANDLER = new Handler(PCT_WORK_THREAD.getLooper());
    }

    public interface MetricCallback {
        void notifyPCTEvent(String event, Map<String, Object> params);
    }

    public static void setMetricCallback(MetricCallback callback) {
        Log.i(TAG, "set metric callback: " + callback);
        sMetricCallback = callback;
        if (sMetricCallback != null) {
            for (Pair<String, Map<String, Object>> entry : METRIC_EVENT_CACHE) {
                sMetricCallback.notifyPCTEvent(entry.first, entry.second);
            }
            METRIC_EVENT_CACHE.clear();
        }
    }

    private static final PctApi.IMetricCallBack sInsideMetricCallBack = (event, params) -> {
        Log.i(TAG, "PCT metric callback: " + event);
        Map<String, Object> metrics = new LinkedHashMap<>();
        if (params != null) {
            metrics.putAll(params);
        }
        metrics.put("vno_id", sVnoId);
        if (sMetricCallback != null) {
            sMetricCallback.notifyPCTEvent(event, metrics);
        } else {
            if (METRIC_EVENT_CACHE.size() > 10) {
                METRIC_EVENT_CACHE.remove();
            }
            METRIC_EVENT_CACHE.add(new Pair<>(event, metrics));
        }
    };

    /**
     * set so path
     *
     * @param bikSoPath BikCore so path which default as null
     * @param pctSoPath pct engine so path which default as null
     */
    public static void loadLib(String bikSoPath, String pctSoPath) {
        if (sHasLoadLib) {
            return;
        }
        if (bikSoPath == null) {
            bikSoPath = "";
        }
        if (pctSoPath == null) {
            pctSoPath = "";
        }
        Log.i(TAG, "init module bik='" + bikSoPath + "', pct='" + pctSoPath + "'.");
        PctApi.loadSo(pctSoPath, bikSoPath);
        sHasLoadLib = true;
    }

    /**
     * get pct version
     *
     * @return pct version
     * @throws RuntimeException throws exception if call it before init pct sdk
     */
    public static String getVersion() throws RuntimeException {
        if (sInitPCT) {
            return PctApi.getVersion();
        }
        return "";
    }

    private static void initWorkPath(@Nullable Context context) {
        String rootDir;
        if (context != null) {
            rootDir = Objects.requireNonNull(ContextCompat.getDataDir(context)).getAbsolutePath() + File.separator + "rapidtv" + File.separator;
        } else if (externalStorageAvailable()) {
            rootDir = Environment.getExternalStorageDirectory().getAbsolutePath() + File.separator + "rapidtv" + File.separator;
        } else {
            rootDir = File.separator + "rapidtv" + File.separator;
        }
        if (context == null) {
            Log.w("RapidAgentConstant", "Invalid Context Warning : initWorkPath " + rootDir);
        }
        if (!rootDir.endsWith(File.separator)) {
            rootDir = rootDir + File.separator;
        }
        sPCTWorkPath = rootDir + "pct";
        File pctWorkPath = new File(sPCTWorkPath);
        if (!pctWorkPath.exists() && !pctWorkPath.mkdirs()) {
            Log.e("RapidAgentConstant", "can not create pct work path");
        }
        sRDNSWorkPath = rootDir + "rdns";
        File rdnsWorkPath = new File(sRDNSWorkPath);
        if (!rdnsWorkPath.exists() && !rdnsWorkPath.mkdirs()) {
            Log.e("RapidAgentConstant", "can not create rdns work path");
        }
    }

    /**
     * init pct and rdns module
     *
     * @param context application context
     * @param vnoId   VNO ID represent a VNO client, and got after register on our website
     * @param params  PCT params
     */
    public static void initPCT(Context context, @NonNull String vnoId, @NonNull PctParams params) {
        Log.i(TAG, "start init sdk version " + BuildConfig.VERSION_CODE + "_" + BuildConfig.GIT_HASH);
        loadLib(null, null);
        initWorkPath(context);
        SPUtil.init(context);
        sLogServer = SPUtil.getPCTLogServer(DEFAULT_LOG_SERVER);
        PctParams.SdkBuilder sdkBuilder = new PctParams.SdkBuilder(params);
        sdkBuilder
                .vnoId(vnoId)
                .logServer(sLogServer)
                .workPath(sPCTWorkPath)
                .metricCallBack(sInsideMetricCallBack)
                .asyncInitEngine(false);
        // Add PCT_WORK_HANDLER to fix issue: crash when ASan is integrated
        PCT_WORK_HANDLER.post(() -> PctApi.initialize(params));
        sInitPCT = true;
        sInitRDNS = true;
        sVnoId = vnoId;
        Log.i(TAG, "finish init PCT SDK");
    }

    public static void refreshLogServer(String newLogServer) {
        if (!sInitPCT) {
            return;
        }
        if (!TextUtils.equals(newLogServer, sLogServer)) {
            sLogServer = newLogServer;
            PctApi.updateLogServer(sLogServer);
            SPUtil.setPCTLogServer(sLogServer);
        }
    }

    /**
     * make a http request via pct
     *
     * @param request  http request info such as http method、header、body
     * @param callback callback
     * @return the task id.
     * task id &gt;= 0, the request sent successfully and callback will be invoked.
     * task id &lt; 0, fail to send the request and callback never will be invoked.
     */
    public static long httpRequest(PctRequest request, PctApi.ITaskCallBack callback) {
        if (!sInitPCT) {
            Log.e(TAG, "please init pct first");
            return -1;
        }
        return PctApi.httpRequest(request, callback);
    }

    /**
     * make a http request via pct
     *
     * @param request    http request info such as http method、header、body
     * @param rspType    request type, {@link com.pct.core.api.PctApi.RspType}
     * @param timeoutSec timeout seconds
     * @param callback   callback
     * @return the task id.
     * task id &gt;= 0, the request sent successfully and callback will be invoked.
     * task id &lt; 0, fail to send the request and callback never will be invoked.
     */
    public static long httpRequest(PctRequest request, PctApi.RspType rspType, int timeoutSec, PctApi.ITaskCallBack callback) {
        if (!sInitPCT) {
            Log.e(TAG, "please init pct first");
            return -1;
        }
        return PctApi.httpRequest(request, rspType, timeoutSec, callback);
    }

    /**
     * init rdns module only
     *
     * @param context application context
     * @param vnoId   VNO ID represent a VNO client, and got after register on our website
     * @param params  rdns params
     * @return whether init successfully
     */
    public static boolean initRDns(Context context, String vnoId, RdnsParams params) {
        loadLib(null, null);
        initWorkPath(context);
        RdnsParams.SdkBuilder sdkBuilder = new RdnsParams.SdkBuilder(params);
        sdkBuilder
                .vnoId(vnoId)
                .workPath(sRDNSWorkPath)
                .metricCallBack(sInsideMetricCallBack);
        sVnoId = vnoId;
        return (sInitRDNS = RdnsApi.initialize(params));
    }

    /**
     * domain name resolution through DNS protocol
     *
     * @param domain   domain
     * @param tmSecs   timeout seconds
     * @param callback callback
     */
    public static void dnsGetHostByNameAsync(String domain, int tmSecs, RdnsApi.RdnsCallBack callback) {
        if (!sInitRDNS) {
            Log.e(TAG, "please init pct first");
            return;
        }
        RdnsApi.dnsGetHostByNameAsync(domain, tmSecs, callback);
    }

    /**
     * domain name resolution through DNS protocol
     *
     * @param domain   domain
     * @param tmSecs   timeout seconds
     * @param callback callback
     * @param blackIp  black ip
     */
    public static void dnsGetHostByNameAsync(String domain, int tmSecs, RdnsApi.RdnsCallBack callback, String blackIp) {
        if (!sInitRDNS) {
            Log.e(TAG, "please init pct first");
            return;
        }
        RdnsApi.dnsGetHostByNameAsync(domain, tmSecs, callback, blackIp);
    }

    /**
     * domain name resolution through DHT protocol
     *
     * @param domain   domain
     * @param tmSecs   timeout seconds
     * @param callback callback
     */
    public static void dhtGetHostByNameAsync(String domain, int tmSecs, RdnsApi.RdnsCallBack callback) {
        if (!sInitRDNS) {
            Log.e(TAG, "please init pct first");
            return;
        }
        RdnsApi.dhtGetHostByNameAsync(domain, tmSecs, callback);
    }

    /**
     * domain name resolution through DHT protocol
     *
     * @param domain   domain
     * @param tmSecs   timeout seconds
     * @param callback callback
     * @param blackIp  black ip
     */
    public static void dhtGetHostByNameAsync(String domain, int tmSecs, RdnsApi.RdnsCallBack callback, String blackIp) {
        if (!sInitRDNS) {
            Log.e(TAG, "please init pct first");
            return;
        }
        RdnsApi.dhtGetHostByNameAsync(domain, tmSecs, callback, blackIp);
    }

    /**
     * Set the default webId map, and HTTP requests within the SDK will be mapped to webId through this map.
     * If no webId is configured for the corresponding URL, HTTP requests will be initiated through the default method.
     *
     * @param map default webId map
     */
    public static void setDefaultWebId(@NonNull Map<String, String> map) {
        DEFAULT_WEB_ID_MAP.putAll(map);
    }

    public static Map<String, String> getDflWebIdMap() {
        return DEFAULT_WEB_ID_MAP;
    }

    /**
     * update network type
     *
     * @param networkType network type
     */
    public static void updateNetworkType(PctApi.NetworkType networkType) {
        PctApi.updateNetworkType(networkType);
    }

    public static void updateConfig(String json) {
        if (sInitPCT) {
            PctApi.updateConfig(json);
        }
        if (sInitRDNS) {
            RdnsApi.updateConfig(json);
        }
    }

    private byte[] gzipTest(String bodyString) throws IOException {
        String header = "http/1.1 200\r\n" +
                "Content-Type: text/html\r\n" +
                "Content-Encoding  :  gzip\r\n" +
                "Date: Mon, 27 May 2024 02:41:37 GMT\r\n" +
                "\r\n";
        byte[] headerByte = header.getBytes(StandardCharsets.UTF_8);
        byte[] bodyByte = compressString(bodyString);
        byte[] responseByte = new byte[headerByte.length + bodyByte.length];
        System.arraycopy(headerByte, 0, responseByte, 0, headerByte.length);
        System.arraycopy(bodyByte, 0, responseByte, headerByte.length, bodyByte.length);
        return responseByte;
    }

    private static byte[] compressString(String str) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        GZIPOutputStream gzip = new GZIPOutputStream(bos);
        gzip.write(str.getBytes());
        gzip.close();
        byte[] result = bos.toByteArray();
        gzip.close();
        return result;
    }

    private static boolean externalStorageAvailable() {
        try {
            return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
        } catch (Exception e) {
            return false;
        }
    }
}