package com.ott.stream.rapid.player.utils;

import android.media.MediaCodec;
import android.media.MediaExtractor;
import android.media.MediaFormat;
import android.os.Build;
import android.util.Log;
import android.view.Surface;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;

import tv.danmaku.ijk.media.player.IjkMediaCodecInfo;
import tv.danmaku.ijk.media.player.IjkMediaPlayer;

public class MediaCodecTest {
    private static final String TAG = "MediaCodecTest";

    private static class Sample {
        ByteBuffer byteBuffer;
        int size;
        int flags;
        long sampleTime;

        Sample(ByteBuffer byteBuffer, int size, int flags, long sampleTime) {
            this.byteBuffer = byteBuffer;
            this.size = size;
            this.flags = flags;
            this.sampleTime = sampleTime;
        }

        @NonNull
        @Override
        public String toString() {
            return "Sample{" +
                    "size=" + size +
                    ", flags=" + flags +
                    ", sampleTime=" + sampleTime +
                    '}';
        }
    }

    public static void start(final String path, final Surface surface) {
        start(path, surface, 10000, 25, false);
    }

    public static void start(final String path, final Surface surface, final int displayMs, final int frameCount, final boolean asyncMode) {
        new Thread(() -> {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                try {
                    startInternal(path, surface, displayMs, frameCount, asyncMode);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }).start();
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    private static void startInternal(String path, Surface surface, int displayMs, int frameCount, boolean asyncMode) throws IOException {
        MediaExtractor extractor = new MediaExtractor();
        extractor.setDataSource(path);
        String videoMimeType = null;
        MediaFormat videoTrackFormat = null;
        for (int i = 0; i < extractor.getTrackCount(); i++) {
            videoTrackFormat = extractor.getTrackFormat(i);
            videoMimeType = videoTrackFormat.getString(MediaFormat.KEY_MIME);
            if (videoMimeType.startsWith("video/")) {
                extractor.selectTrack(i);
                break;
            }
        }
        if (videoMimeType == null) {
            throw new IllegalStateException("invalid video mime type");
        }

        Queue<Sample> sampleQueue = new ArrayDeque<>();
        for (int i = 0; i < frameCount; i++) {
            ByteBuffer byteBuffer = ByteBuffer.allocate(512 * 1024);
            long time = extractor.getSampleTime();
            int flags = extractor.getSampleFlags();
            int readSampleData = extractor.readSampleData(byteBuffer, 0);
            if (readSampleData < 0) {
                throw new IllegalStateException("readSampleData error");
            }
            sampleQueue.add(new Sample(byteBuffer, readSampleData, flags, time));
            extractor.advance();
        }

        long startTimeMs = System.currentTimeMillis();
        log("MediaCodec work start", startTimeMs);
        String codecName;
        if (videoMimeType.equals("video/avc")) {
            codecName = findMediaCodec(videoMimeType, 100, 40);
        } else if (videoMimeType.equals("video/hevc")) {
            codecName = findMediaCodec(videoMimeType, 1, 93);
        } else {
            codecName = findMediaCodec(videoMimeType, 0, 0);
        }
        log("MediaCodec select codec:" + codecName, startTimeMs);
        MediaCodec mediaCodec = MediaCodec.createByCodecName(codecName);
        log("MediaCodec created", startTimeMs);
        mediaCodec.configure(videoTrackFormat, surface, null, 0);
        log("MediaCodec configured " + videoTrackFormat, startTimeMs);
        if (asyncMode) {
            mediaCodec.setCallback(new MediaCodec.Callback() {
                final Map<Long, Long> map = new HashMap<>();

                @Override
                public void onInputBufferAvailable(@NonNull MediaCodec codec, int index) {
                    if (!sampleQueue.isEmpty()) {
                        Sample sample = sampleQueue.remove();
                        map.put(sample.sampleTime, System.nanoTime() / 1000_000);
                        ByteBuffer inputBuffer = mediaCodec.getInputBuffer(index);
                        inputBuffer.put(sample.byteBuffer);
                        mediaCodec.queueInputBuffer(index, 0, sample.size, sample.sampleTime, sample.flags);
                        log("MediaCodec queueInputBuffer sample:" + sample, startTimeMs);
                    }
                }

                @Override
                public void onOutputBufferAvailable(@NonNull MediaCodec codec, int index, @NonNull MediaCodec.BufferInfo info) {
                    mediaCodec.releaseOutputBuffer(index, true);
                    map.remove(info.presentationTimeUs);
                    log("MediaCodec dequeueOutputBuffer presentationTime:" + info.presentationTimeUs + " " + map.size(), startTimeMs);
                }

                @Override
                public void onError(@NonNull MediaCodec codec, @NonNull MediaCodec.CodecException e) {
                    log("error:" + e.getMessage(), startTimeMs);
                }

                @Override
                public void onOutputFormatChanged(@NonNull MediaCodec codec, @NonNull MediaFormat format) {
                    log("format changed:" + mediaCodec.getOutputFormat(), startTimeMs);
                }
            });
        }
        mediaCodec.start();
        log("MediaCodec started", startTimeMs);
        if (!asyncMode) {
            int outputSize = 0;
            int retryTimes = 0;
            final Map<Long, Long> map = new HashMap<>();
            MediaCodec.BufferInfo bufferInfo = new MediaCodec.BufferInfo();
            while (outputSize < frameCount) {
                if (!sampleQueue.isEmpty()) {
                    int inputIndex = mediaCodec.dequeueInputBuffer(5000);
                    if (inputIndex < 0) {
                        continue;
                    }
                    ByteBuffer inputBuffer = mediaCodec.getInputBuffer(inputIndex);
                    Sample sample = sampleQueue.remove();
                    inputBuffer.put(sample.byteBuffer);
                    map.put(sample.sampleTime, System.nanoTime() / 1000_000);
                    log("MediaCodec queueInputBuffer sample:" + sample, startTimeMs);
                    mediaCodec.queueInputBuffer(inputIndex, 0, sample.size, sample.sampleTime, sample.flags);
                } else if (retryTimes++ > 1000) {
                    log("MediaCodec reach max retry times", startTimeMs);
                    break;
                }
                int outputIndex = mediaCodec.dequeueOutputBuffer(bufferInfo, 5000);
                if (outputIndex < 0) {
                    if (outputIndex == MediaCodec.INFO_OUTPUT_FORMAT_CHANGED) {
                        log("MediaCodec format changed:" + mediaCodec.getOutputFormat(), startTimeMs);
                    }
                } else {
                    mediaCodec.releaseOutputBuffer(outputIndex, true);
                    map.remove(bufferInfo.presentationTimeUs);
                    log("MediaCodec dequeueOutputBuffer presentationTime:" + bufferInfo.presentationTimeUs + " " + map.size(), startTimeMs);
                    outputSize++;
                }
            }
        }
        log("MediaCodec displaying for " + displayMs + "ms", startTimeMs);
        try {
            Thread.sleep(displayMs);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        log("MediaCodec stopping", startTimeMs);
        mediaCodec.stop();
        log("MediaCodec stopped", startTimeMs);
        mediaCodec.release();
        log("MediaCodec released", startTimeMs);
        extractor.release();
    }

    private static String findMediaCodec(String mimeType, int profile, int level) {
        IjkMediaPlayer.DefaultMediaCodecSelector selector = IjkMediaPlayer.DefaultMediaCodecSelector.sInstance;
        IjkMediaCodecInfo codecInfo = selector.onIjkMediaCodecSelect(null, mimeType, profile, level, false);
        return codecInfo.mCodecInfo.getName();
    }

    private static void log(String msg, long startTimeMs) {
        long now = System.currentTimeMillis();
        Log.i(TAG, String.format("[%3dms] %s", (now - startTimeMs), msg));
    }
}
