package com.ott.stream.rapid.player;

import static com.ott.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_IJK;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_AUDIO_DECODED_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_AUDIO_RENDERING_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_AUDIO_SEEK_RENDERING_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_BUFFERING_END;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_BUFFERING_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_COMPONENT_OPEN;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_DRM_KEY_LOADED;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_FIND_STREAM_INFO;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_OPEN_INPUT;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_READ_FIRST_AUDIO_FRAME;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_READ_FIRST_VIDEO_FRAME;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_STARTUP_INFO;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_STREAM_FORMAT_SUMMARY;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_VIDEO_DECODED_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_VIDEO_ROTATION_CHANGED;
import static tv.danmaku.ijk.media.player.IMediaPlayer.MEDIA_INFO_VIDEO_SEEK_RENDERING_START;

import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Map;

public class DefaultPlayerStatistic implements PlayerStatistic {
    private static final String TAG = "DefaultPlayerStatistic";

    private int playerType;
    private int playerHash;
    private Uri sourceUri;
    private long createdTimeNs;
    private long preparingTimeNs;
    private String startupInfo;
    private long bufferingStartTimeNs;
    private String bufferingStartReason;
    private long seekStartTimeNs;
    private long stopStartTimeNs;
    private long stopEndTimeNs;
    private long releaseStartTimeNs;

    private void logEvent(String msg) {
        float seconds = (System.nanoTime() - createdTimeNs) / 1_000_000_000f;
        Log.d(TAG, String.format("%s [eventTime=%.2f, hash=%x, uri:%s]", msg, seconds, playerHash, sourceUri));
    }

    @Override
    public void setPlayerType(int playerType, int playerHash) {
        this.playerType = playerType;
        this.playerHash = playerHash;
        createdTimeNs = System.nanoTime();
        logEvent("created");
    }

    @Override
    public void setDataSource(Uri uri) {
        logEvent("setDataSource " + uri);
        sourceUri = uri;
    }

    @Override
    public void prepareAsync() {
        logEvent("prepareAsync");
        preparingTimeNs = System.nanoTime();
    }

    @Override
    public void start() {
        logEvent("start");
    }

    @Override
    public void stopStart() {
        logEvent("stopStart");
        stopStartTimeNs = System.nanoTime();
    }

    @Override
    public void stopFinished() {
        logEvent("stopFinished");
        stopEndTimeNs = System.nanoTime();
    }

    @Override
    public void pause() {
        logEvent("pause");
    }

    @Override
    public void seekTo(long msec) {
        logEvent("seekTo " + msec);
        seekStartTimeNs = System.nanoTime();
    }

    @Override
    public void releaseStart() {
        logEvent("releaseStart");
        releaseStartTimeNs = System.nanoTime();
    }

    @Override
    public void releaseFinished() {
        logEvent("releaseFinished");
        if (createdTimeNs == 0) {
            return;
        }
        long now = System.nanoTime();
        long stopDuration = (stopEndTimeNs - stopStartTimeNs) / 1_000_000;
        long releaseDuration = (now - releaseStartTimeNs) / 1_000_000;
        long lifecycleDuration = (now - createdTimeNs) / 1_000_000;
        releasePerformance(stopDuration, releaseDuration, lifecycleDuration);
    }

    @Override
    public void reset() {
        logEvent("reset");
    }

    @Override
    public void onError(int what, int extra) {
        logEvent("onError what:" + what + ",extra:" + extra);
    }

    @Override
    public void onInfo(int what, int extra, String info) {
        logEvent("onInfo:" + infoCode2String(what) + ",extra:" + extra + ",info:" + info);
        if (what == MEDIA_INFO_STARTUP_INFO) {
            startupInfo = info;
        } else if (what == MEDIA_INFO_VIDEO_RENDERING_START || (what == MEDIA_INFO_AUDIO_RENDERING_START && extra == 1)) {
            if (startupInfo == null) {
                long duration = (System.nanoTime() - preparingTimeNs) / 1_000_000;
                startupInfo = String.format("{video_render:%s}", duration);
            }
            try {
                JSONObject jsonObject = new JSONObject(startupInfo);
                startupPerformance(jsonObject);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else if (what == MEDIA_INFO_BUFFERING_START) {
            bufferingStartTimeNs = System.nanoTime();
            bufferingStartReason = info;
        } else if (what == MEDIA_INFO_BUFFERING_END) {
            long duration = (System.nanoTime() - bufferingStartTimeNs) / 1_000_000;
            bufferingPerformance(extra, duration, bufferingStartReason, info);
        } else if (what == MEDIA_INFO_VIDEO_SEEK_RENDERING_START) {
            long duration = (System.nanoTime() - seekStartTimeNs) / 1_000_000;
            seekPerformance(duration);
        }
    }

    @Override
    public void onCompletion() {
        logEvent("onCompletion");
    }

    @Override
    public void onStatusChanged(@NonNull Map<String, String> params) {
        logEvent("onStatusChanged:" + params);
    }

    @Override
    public void onSeekComplete() {
        if (playerType != PLAYER_TYPE_IJK) {
            long duration = (System.nanoTime() - seekStartTimeNs) / 1_000_000;
            seekPerformance(duration);
        }
    }

    @Override
    public void startupPerformance(JSONObject jsonObject) {
        logEvent("startupPerformance:" + jsonObject);
    }

    @Override
    public void bufferingPerformance(int extra, long duration, String startReason, String endReason) {
        logEvent("bufferingPerformance " + duration + ", " + startReason + ", " + endReason);
    }

    @Override
    public void seekPerformance(long duration) {
        logEvent("seekPerformance " + duration);
    }

    @Override
    public void releasePerformance(long stopDuration, long releaseDuration, long lifecycleDuration) {
        logEvent("releasePerformance " + stopDuration + ", " + releaseDuration + ", " + lifecycleDuration);
    }

    @Override
    public void downloadPerformance(@NonNull Map<String, Object> params) {

    }

    private static String infoCode2String(int code) {
        switch (code) {
            case MEDIA_INFO_VIDEO_RENDERING_START:
                return "video_rendering_start";
            case MEDIA_INFO_BUFFERING_START:
                return "buffering_start";
            case MEDIA_INFO_BUFFERING_END:
                return "buffering_end";
            case MEDIA_INFO_STREAM_FORMAT_SUMMARY:
                return "stream_format_summary";
            case MEDIA_INFO_STARTUP_INFO:
                return "startup_info";
            case MEDIA_INFO_VIDEO_ROTATION_CHANGED:
                return "video_rotation_changed";
            case MEDIA_INFO_AUDIO_RENDERING_START:
                return "audio_rendering_start";
            case MEDIA_INFO_AUDIO_DECODED_START:
                return "audio_decoded_start";
            case MEDIA_INFO_VIDEO_DECODED_START:
                return "video_decoded_start";
            case MEDIA_INFO_OPEN_INPUT:
                return "open_input";
            case MEDIA_INFO_COMPONENT_OPEN:
                return "component_open";
            case MEDIA_INFO_READ_FIRST_VIDEO_FRAME:
                return "first_video_frame";
            case MEDIA_INFO_READ_FIRST_AUDIO_FRAME:
                return "first_audio_frame";
            case MEDIA_INFO_FIND_STREAM_INFO:
                return "find_stream_info";
            case MEDIA_INFO_DRM_KEY_LOADED:
                return "drm_key_loaded";
            case MEDIA_INFO_VIDEO_SEEK_RENDERING_START:
                return "video_seek_rendering_start";
            case MEDIA_INFO_AUDIO_SEEK_RENDERING_START:
                return "audio_seek_rendering_start";
            default:
                return String.valueOf(code);
        }
    }
}
