package com.ott.stream.rapid.player.egl;

import android.opengl.EGL14;
import android.opengl.EGLConfig;
import android.opengl.EGLContext;
import android.opengl.EGLDisplay;
import android.opengl.EGLExt;
import android.opengl.EGLSurface;
import android.util.Log;
import android.view.Surface;

import java.util.Arrays;

public class EGL {

    private EGLDisplay eglDisplay = EGL14.EGL_NO_DISPLAY;
    private EGLSurface eglSurface = EGL14.EGL_NO_SURFACE;
    private EGLContext eglContext = EGL14.EGL_NO_CONTEXT;

    private EGLDisplay previousDisplay = EGL14.EGL_NO_DISPLAY;
    private EGLSurface previousDrawSurface = EGL14.EGL_NO_SURFACE;
    private EGLSurface previousReadSurface = EGL14.EGL_NO_SURFACE;
    private EGLContext previousContext = EGL14.EGL_NO_CONTEXT;

    private final EGLConfig[] eglConfig = new EGLConfig[1];

    public void init() {
        init(EGL14.EGL_NO_CONTEXT);
    }

    /**
     * init EGL
     *
     * @param shareContext the EGL context to share
     */
    public void init(EGLContext shareContext) {
        eglDisplay = EGL14.eglGetDisplay(EGL14.EGL_DEFAULT_DISPLAY);
        if (eglDisplay == null || eglDisplay == EGL14.EGL_NO_DISPLAY) {
            checkEglException("init eglGetDisplay");
        }
        int[] version = new int[2];
        if (!EGL14.eglInitialize(eglDisplay, version, 0, version, 1)) {
            checkEglException("init eglInitialize");
        }
        int[] attribList = new int[]{
                EGL14.EGL_RENDERABLE_TYPE, EGL14.EGL_OPENGL_ES2_BIT,
                EGL14.EGL_RED_SIZE, /* redSize= */ 8,
                EGL14.EGL_GREEN_SIZE, /* greenSize= */ 8,
                EGL14.EGL_BLUE_SIZE, /* blueSize= */ 8,
                EGL14.EGL_ALPHA_SIZE, /* alphaSize= */ 8,
                EGL14.EGL_DEPTH_SIZE, /* depthSize= */ 0,
                EGL14.EGL_STENCIL_SIZE, /* stencilSize= */ 0,
                EGL14.EGL_NONE};
        int[] numConfigs = new int[1];
        if (!EGL14.eglChooseConfig(eglDisplay, attribList, 0, eglConfig, 0,
                eglConfig.length, numConfigs, 0)) {
            checkEglException("init eglChooseConfig");
        }
        eglContext = EGL14.eglCreateContext(eglDisplay, eglConfig[0], shareContext,
                new int[]{EGL14.EGL_CONTEXT_CLIENT_VERSION, 2, EGL14.EGL_NONE}, 0
        );
        if (eglContext == null || eglContext == EGL14.EGL_NO_CONTEXT) {
            EGL14.eglTerminate(eglDisplay);
            throw new IllegalArgumentException("RapidEGL createContext failed : " + EGL14.eglGetError());
        }
        Log.i("RapidEGL", "egl init success " + Arrays.toString(version));
    }

    /**
     * set the surface for output
     *
     * @param surface the surface to bind
     */
    public void setSurface(Surface surface) {
        setSurface(surface, false);
    }

    /**
     * set the surface for output
     *
     * @param surface  the surface to bind
     * @param autoBind auto bind current surface
     */
    public void setSurface(Surface surface, boolean autoBind) {
        if (eglSurface != EGL14.EGL_NO_SURFACE) {
            EGL14.eglDestroySurface(eglDisplay, eglSurface);
            checkEglException("setSurface eglDestroySurface");
        }
        int[] surfaceAttrib = new int[]{EGL14.EGL_NONE};
        if (surface == null) {
            eglSurface = EGL14.eglCreatePbufferSurface(eglDisplay, eglConfig[0], surfaceAttrib, 0);
            checkEglException("setSurface eglCreatePbufferSurface");
        } else {
            eglSurface = EGL14.eglCreateWindowSurface(eglDisplay, eglConfig[0], surface, surfaceAttrib, 0);
            checkEglException("setSurface eglCreateWindowSurface");
        }
        if (autoBind && !EGL14.eglMakeCurrent(eglDisplay, eglSurface, eglSurface, eglContext)) {
            checkEglException("setSurface eglMakeCurrent");
        }
    }

    /**
     * bind EGL to calling thread
     */
    public void bind() {
        previousDisplay = EGL14.eglGetCurrentDisplay();
        previousDrawSurface = EGL14.eglGetCurrentSurface(EGL14.EGL_DRAW);
        previousReadSurface = EGL14.eglGetCurrentSurface(EGL14.EGL_READ);
        previousContext = EGL14.eglGetCurrentContext();
        if (!EGL14.eglMakeCurrent(eglDisplay, eglSurface, eglSurface, eglContext)) {
            checkEglException("bind eglMakeCurrent");
        }
    }

    /**
     * swap buffers. if the EGL binds to a surface and you want to render to this surface, please call
     * this method after rendering.
     */
    public boolean swapBuffers() {
        return EGL14.eglSwapBuffers(eglDisplay, eglSurface);
    }

    /**
     * presentation time set presentation time
     *
     * @param timestamp timestamp in nanoseconds
     */
    public boolean setPresentationTime(long timestamp) {
        return EGLExt.eglPresentationTimeANDROID(eglDisplay, eglSurface, timestamp);
    }

    /**
     * unbind EGL to the calling thread and restore the previous EGL
     */
    public void unbind() {
        EGL14
                .eglMakeCurrent(previousDisplay, previousDrawSurface, previousReadSurface, previousContext);
        checkEglException("unbind eglMakeCurrent");
    }

    /**
     * release resources
     */
    public void release() {
        if (eglDisplay != EGL14.EGL_NO_DISPLAY) {
            EGL14.eglMakeCurrent(eglDisplay, EGL14.EGL_NO_SURFACE, EGL14.EGL_NO_SURFACE, EGL14.EGL_NO_CONTEXT);
            checkEglException("release eglMakeCurrent");
            EGL14.eglDestroySurface(eglDisplay, eglSurface);
            checkEglException("release eglDestroySurface");
            EGL14.eglDestroyContext(eglDisplay, eglContext);
            checkEglException("release eglDestroyContext");
            EGL14.eglReleaseThread();
            checkEglException("release eglReleaseThread");
            EGL14.eglTerminate(eglDisplay);
            checkEglException("release eglTerminate");
        }
        eglDisplay = EGL14.EGL_NO_DISPLAY;
        eglContext = EGL14.EGL_NO_CONTEXT;
        eglSurface = EGL14.EGL_NO_SURFACE;
        Log.i("RapidEGL", "egl release success");
    }

    private static void checkEglException(String errorMessage) {
        int error = EGL14.eglGetError();
        if (error != EGL14.EGL_SUCCESS) {
            errorMessage = errorMessage + " " + error;
            Log.e("RapidEGL", "checkEglException: " + errorMessage);
            throw new IllegalStateException(errorMessage);
        }
    }
}
