package com.ott.stream.rapid.player.egl;

import android.opengl.GLES20;
import android.view.Surface;

import java.util.ArrayList;
import java.util.concurrent.LinkedBlockingQueue;

public class MultiTextureRenderer extends AbsTextureRenderer implements MapsTexture.OnFrameAvailableListener {
    private static class TextureManager {
        private final int maxTextures;
        private final ArrayList<MapsTexture> allList = new ArrayList<>();
        private final LinkedBlockingQueue<MapsTexture> idleQueue = new LinkedBlockingQueue<>();

        private TextureManager(int maxTextures) {
            this.maxTextures = maxTextures;
        }

        private void init() {
            for (int i = 0; i < maxTextures; i++) {
                MapsTexture surfaceTexture = new MapsTexture();
                surfaceTexture.init(GLES20.GL_TEXTURE0 + i);
                allList.add(surfaceTexture);
                idleQueue.add(surfaceTexture);
            }
        }

        private MapsTexture build() {
            return idleQueue.remove();
        }

        private void recycle(Surface surface) {
            MapsTexture texture = find(surface);
            idleQueue.add(texture);
        }

        private void disable(Surface surface) {
            MapsTexture texture = find(surface);
            texture.setOnFrameAvailableListener(null);
        }

        private void release() {
            for (MapsTexture texture : allList) {
                texture.release();
            }
        }

        private MapsTexture find(Surface surface) {
            for (MapsTexture texture : allList) {
                if (surface == texture.getSurface()) {
                    return texture;
                }
            }
            throw new IllegalStateException("the surface is not match to any texture");
        }
    }

    private final TextureManager textureManager;

    public MultiTextureRenderer() {
        this(2);
    }

    public MultiTextureRenderer(int maxTextures) {
        if (maxTextures < 2) {
            maxTextures = 2;
        } else if (maxTextures > 5) {
            maxTextures = 5;
        }
        this.textureManager = new TextureManager(maxTextures);
    }

    @Override
    public void surfaceDestroyed() {
        runOnEglThread(textureManager::release);
        super.surfaceDestroyed();
    }

    @Override
    public Surface createInputSurface() {
        MapsTexture texture = textureManager.build();
        texture.setOnFrameAvailableListener(this);
        return texture.getSurface();
    }

    @Override
    public void releaseInputSurface(Surface surface) {
        textureManager.recycle(surface);
    }

    @Override
    public void disableInputSurface(Surface surface) {
        textureManager.disable(surface);
    }

    @Override
    protected void onPrepared() {
        textureManager.init();
        super.onPrepared();
    }

    @Override
    public void onFrameAvailable(MapsTexture pipeline) {
        requestRender(pipeline);
    }
}
