package com.stream.prt;

import com.stream.prt.log.Logger;
import com.stream.prt.log.LoggerFactory;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/* jni回传数据分发 */
public class PrtListenerManager {
    private final static Logger logger = LoggerFactory.getLogger(PrtListenerManager.class.getName());

    private static String TAG = PrtListenerManager.class.getSimpleName();
    private static volatile PrtListenerManager mInstance;
    private static HashMap<PrtManagerCallBack, JniPrtListener> mListenerMap = new HashMap<>();
    private PrtCacheEventListener prtCacheEventListener;

    public static PrtListenerManager getInstance() {
        if (mInstance == null) {
            synchronized (PrtListenerManager.class) {
                if (mInstance == null) {
                    mInstance = new PrtListenerManager();
                }
            }
        }
        return mInstance;
    }

    private PrtListenerManager() {
    }

    public void setPrtCacheEventListener(PrtCacheEventListener l) {
        prtCacheEventListener = l;
    }

    public PrtCacheEventListener getPrtCacheEventListener() {
        return prtCacheEventListener;
    }


    /* 注册listener */
    public void registerListener(int channelId, PrtManagerCallBack callBack, JniPrtListener listener) {
        if ((channelId < 0)) {
            throw new IllegalArgumentException("para error - channelId = " + channelId + "; listener = " + listener);
        }

        if (null == listener) {
            return;
        }

        synchronized (mListenerMap) {
            if (mListenerMap.get(callBack) == null) {
                logger.debug("registerListener  -------> put channelId:%s", channelId);
                mListenerMap.put(callBack, listener);
            }
        }
        logger.info("registerListener channelId:%d size size:%d", channelId, mListenerMap.size());
    }

    /* 移除listener */
    public void unRegisterListener(int channelId, int requestId) {
        synchronized (mListenerMap) {
            logger.debug("unRegisterListener size:%s", mListenerMap.size());
            Iterator<Map.Entry<PrtManagerCallBack, JniPrtListener>> iterator = mListenerMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<PrtManagerCallBack, JniPrtListener> entry = iterator.next();
                PrtManagerCallBack callBack = entry.getKey();
                if (callBack.getRequestId() == requestId && callBack.getChannelId() == channelId) {
                    logger.debug("unRegisterListener  -------> remove");
                    iterator.remove();
                }
            }
        }
        logger.info("unRegisterListener channelId:%d requestId:%d size size:%d", channelId, requestId, mListenerMap.size());
    }

    /* 根据channelId查找对应listener */
    public JniPrtListener findListener(int channelId, int requestId) {
        if (channelId < 0 || requestId < 0) {
            logger.error(TAG, "channelId is invalid, do not need to dispatch");
            return null;
        }

        if ((mListenerMap == null) || (mListenerMap.size() <= 0)) {
            logger.debug(TAG, "no listener found, do not need to dispatch");
            return null;
        }

        synchronized (mListenerMap) {
            for (Map.Entry<PrtManagerCallBack, JniPrtListener> entry : mListenerMap.entrySet()) {
                PrtManagerCallBack callBack = entry.getKey();
                logger.debug(TAG, "requestId:" + callBack.getRequestId() + ",channelId:" + callBack.getChannelId());
                if (requestId == PrtCallBack.DEFAULT_REQUEST_ID) {
                    if (callBack.getChannelId() == channelId) {
                        return entry.getValue();
                    }
                } else {
                    if (callBack.getRequestId() == requestId && callBack.getChannelId() == channelId) {
                        return entry.getValue();
                    }
                }
            }
        }
        return null;
    }

    public interface PrtManagerCallBack {
        int getRequestId();

        int getChannelId();
    }

    public interface PrtCacheEventListener {
        void onCacheEvent(int channelId, int event, String params, String desc);
    }
}
