package com.valor.common.andriodapk.parser;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.io.Files;
import com.valor.common.andriodapk.model.ApkMetaInfo;
import net.dongliu.apk.parser.ApkFile;
import org.apache.http.client.fluent.Request;
import org.dom4j.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Map;

public class AndroidApkParser {
    private static final Logger logger = LoggerFactory.getLogger(AndroidApkParser.class);

    public static ApkMetaInfo parseRemoteApk2(String url) {
        File file = null;
        try {
            String downloadTmpFileName = Files.getNameWithoutExtension(url) + "." + Files.getFileExtension(url);
            file = File.createTempFile("tmp-apk-", downloadTmpFileName);
            Request.Get(url).execute().saveContent(file);
            return parseApk(file);
        } catch (Exception e) {
            logger.error("Download file[{}] exception:{}", url, e);
        } finally {
            try {
                if (file != null) {
                    file.delete();
                }
            } catch (Exception e) {
                logger.error("Delete file[{}] exception:{}", file.getAbsolutePath(), e);
            }
        }

        return null;
    }


    public static ApkMetaInfo parseApk(String filePath) {
        return parseApk(new File(filePath));
    }


    public static void parseElement(Element element, Map<String, String> valueMap, String parentName) {
        if (element.attributeCount() > 0) {
            String name = "";
            String value = "";

            Attribute attrName = element.attribute("name");
            if (attrName != null) {
                name = attrName.getValue();
                Attribute attrValue = element.attribute("value");
                if (attrValue != null) {
                    value = attrValue.getValue();
                }

                if (!Strings.isNullOrEmpty(value)) {
                    valueMap.put(name, value);
                }
            }
        }

        for (Element subElement : element.elements()) {
            parseElement(subElement, valueMap, parentName);
        }
    }

    public static Map<String, String> getAndroidManifest(String manifestXml) {
        try {
            Document document = DocumentHelper.parseText(manifestXml);
            Element rootElement = document.getRootElement();
            Map<String, String> valueMap = Maps.newLinkedHashMap();
            parseElement(rootElement, valueMap, "");
            return valueMap;
        } catch (DocumentException e) {
            logger.info("Parese manifestXml exception", e);
        }
        return null;
    }

    public static ApkMetaInfo parseApk(File file) {
        try (ApkFile apkFile = new ApkFile(file)) {
            if (!file.exists()) {
                logger.info("File Not exist!file:{}", file.getAbsolutePath());
                return null;
            }

            ApkMetaInfo apkMetaInfo = new ApkMetaInfo();

            apkMetaInfo.setPackageName(apkFile.getApkMeta().getPackageName());
            apkMetaInfo.setName(apkFile.getApkMeta().getName());
            apkMetaInfo.setLabel(apkFile.getApkMeta().getLabel());
            apkMetaInfo.setVersionName(apkFile.getApkMeta().getVersionName());
            apkMetaInfo.setVersionCode(apkFile.getApkMeta().getVersionCode());
            apkMetaInfo.setMinSdkVersion(apkFile.getApkMeta().getMinSdkVersion());
            apkMetaInfo.setMaxSdkVersion(apkFile.getApkMeta().getMaxSdkVersion());
            apkMetaInfo.setTargetSdkVersion(apkFile.getApkMeta().getTargetSdkVersion());
            apkMetaInfo.setIcon(apkFile.getApkMeta().getIcon());

            if (!Strings.isNullOrEmpty(apkFile.getManifestXml())) {
                Map<String, String> mainfestMap = getAndroidManifest(apkFile.getManifestXml());
                apkMetaInfo.setReleaseId(mainfestMap.getOrDefault("UMENG_CHANNEL", ""));
            }

            return apkMetaInfo;
        } catch (IOException e) {
            logger.error("Apk file parse exception.apk[{}]", file.getAbsolutePath(), e);
        }

        return null;
    }
}
